﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __ShaderCodeGeneratorGeneric_H__
#define __ShaderCodeGeneratorGeneric_H__

#include "ShaderCodeListCs.h"
#include "ShaderCodeGeneratorGeneric_PCH.h"

using namespace System::Collections::Generic;
using namespace System::Runtime::InteropServices;


namespace EffectMaker {
namespace ShaderCodeGeneratorGeneric {


public ref class ShaderCodeGeneratorGeneric
{
public :
    /// <summary>
    /// Constructor.
    /// </summary>
    ShaderCodeGeneratorGeneric(System::IntPtr argPwnd);

    /// <summary>
    /// Destructor.
    /// </summary>
    virtual ~ShaderCodeGeneratorGeneric();

    /// <summary>
    /// シェーダコードを設定します.
    /// </summary>
    void SetShaderCodes(ShaderCodeListCs ^pShaderCodes);

    /// <summary>
    /// Generate shader code for the given emitter.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    /// <param name="pVertexShader">The output vertex shader code.</param>
    /// <param name="pFragmentShader">The output fragment shader code.</param>
    /// <param name="pComputeShader">The output compute shader code.</param>
    /// <returns>True on success.</returns>
    bool GenerateShaderCode(ShaderConversionInputData ^pInputData,
                            [Out] System::String^% pVertexShader,
                            [Out] System::String^% pFragmentShader,
                            [Out] System::String^% pConputeShader);

    /// <summary>
    /// Generate shader assembly for the given emitter.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    /// <param name="pVertexShader">The output vertex shader assembly.</param>
    /// <param name="pFragmentShader">The output fragment shader assembly.</param>
    /// <param name="pComputeShader">The output compute shader code.</param>
    /// <param name="pErrorList">The array of compile error information.</param>
    /// <returns>True on success.</returns>
    bool GenerateShaderAssembly(ShaderConversionInputData ^pInputData,
                                [Out] System::String^% pVertexShader,
                                [Out] System::String^% pFragmentShader,
                                [Out] System::String^% pConputeShader,
                                [Out] List<ShaderCompileErrorInfo^>^% pErrorList);

    /// <summary>
    /// Generate vfx shader key and codes for the given emitter.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    /// <param name="pShaderKey">The output shader key code.</param>
    /// <param name="pVertexShader">The output vertex shader assembly.</param>
    /// <param name="pFragmentShader">The output fragment shader assembly.</param>
    /// <returns>True on success.</returns>
    bool GenerateVfxShaderKeyAndCodes(ShaderConversionInputData ^pInputData,
                                     [Out] System::String^% pShaderKey,
                                     [Out] System::String^% pVertexShader,
                                     [Out] System::String^% pFragmentShader,
                                     [Out] System::String^% pComputeShader);

protected :
    /// <summary>
    /// Finalize.
    /// </summary>
    !ShaderCodeGeneratorGeneric();


private :
    /// <summary>
    /// GLEWを初期化します
    /// </summary>
    bool InitializeGLEW();

    /// <summary>
    /// Convert the input data into the native input data buffer
    /// so the shader converter can access it.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    void ConvertInputData(ShaderConversionInputData ^pInputData);

    /// <summary>
    /// Convert and copy the managed input data into the native memory,
    /// so the shader converter can access it.
    /// </summary>
    /// <param name="pManagedData">The managed input data.</param>
    /// <param name="ppEmitterData">The pointer to the native emitter data.</param>
    /// <param name="ppRawData">The pointer to the native emitter binary and custom shaders.</param>
    void SetupNativeInputData(ShaderConversionInputData ^pManagedData,
                              ShaderConverterEmitterData **ppEmitterData,
                              unsigned char **ppRawData);

    /// <summary>
    /// Release the memory allocated for the native input data.
    /// </summary>
    void ReleaseInputDataBuffer();

    /// <summary>
    /// シェーダコードをコンパイルしてエラーログを取得します。
    /// </summary>
    /// <param name="type">シェーダのタイプ(GL_VERTEX_SHADER / GL_FRAGMENT_SHADER)</param>
    /// <param name="source">シェーダコードへのダブルポインタ</param>
    /// <return>エラーがあればログ文字列、なければ空文字列.</return>
    std::string CompileCheck(GLuint type, const GLchar **source);

private :
    ShaderConverter   *m_pConverter;

    int                m_inputEmitterCount;
    ShaderCodeListCpp *m_pShaderCodes;

    unsigned char     *m_pInputBuffer;

    HWND m_hWnd;
    HDC m_hDC;
    HGLRC m_hRC;

    bool bIsInitializedGLEW;

};

} // namespace ShaderCodeGeneratorGeneric
} // namespace EffectMaker

#endif // __ShaderCodeGeneratorGeneric_H__
