﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
#if !DEBUG
    #define HANDLE_ALL_EXCEPTION
#endif

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Runtime;
using System.Runtime.InteropServices;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Application.AutoBackup;
using EffectMaker.Application.CommandLine;
using EffectMaker.Application.Properties;
using EffectMaker.BusinessLogic.Commands;
using EffectMaker.BusinessLogic.Connecter;
using EffectMaker.BusinessLogic.EffectCombinerEditor;
using EffectMaker.BusinessLogic.GfxToolsUtility;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Manager;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.BusinessLogic.Synchronization;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.Communicator;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Serializer;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.DataModelLogic.Delegates;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Compiler;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Debugging.Profiling;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.DebugConsole;
using EffectMaker.UIControls.Threading;
using EffectMaker.UIDialogs.MessageDialogs;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.Application
{
    /// <summary>
    /// The application class.
    /// </summary>
    public static class TheApp
    {
        /// <summary>
        /// コマンドライン引数で指定されたプロジェクト設定ファイルパスです。
        /// </summary>
        private static string argProjectConfigPath = string.Empty;

        /// <summary>
        /// マルチプロセス同期用のオブジェクト
        /// </summary>
        private static Mutex mx = new Mutex(false, "NW_EffectMaker_FE2");

        /// <summary>
        /// コンバーターランチャーから起動したか否かのフラグ
        /// </summary>
        private static bool isConverter = false;

        /// <summary>The flag indicating if the fatal error had been handled.</summary>
        public static bool fatalErrorHandled = false;

        /// <summary>
        /// マルチプロセス同期用のオブジェクト
        /// </summary>
        public static Mutex Mx { get { return mx; } }

        /// <summary>
        /// Get the synchronization context for the application.
        /// </summary>
        public static SynchronizationContext SyncContext { get; private set; }

        /// <summary>
        /// Get the flag indicating whether the application is running on command line mode.
        /// </summary>
        public static bool IsCommandLineMode
        {
            get { return OptionStore.RuntimeOptions.IsCommandLineMode; }
            set { OptionStore.RuntimeOptions.IsCommandLineMode = value; }
        }

        /// <summary>
        /// Get the main form.
        /// </summary>
        public static MainForm MainForm { get; private set; }

        /// <summary>
        /// Get the working folder.
        /// </summary>
        public static string WorkingFolder { get; private set; }

        /// <summary>
        /// Get the root view model.
        /// </summary>
        public static WorkspaceRootViewModel RootViewModel { get; private set; }

        /// <summary>
        /// コンソールランチャーからのエントリーポイントです。
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        /// <returns>終了コード</returns>
        public static int RunAsConverter(string[] args)
        {
            TheApp.isConverter = true;
            return TheApp.Main(args);
        }

        /// <summary>
        /// The application entry point.
        /// </summary>
        /// <param name="args">Arguments provided by the command line.</param>
        /// <returns>Returns 0 on success, or a non-zero error code.</returns>
        [STAThread]
        private static int Main(string[] args)
        {
            int result = 0;

            // ロガーを初期化
            Logger.Initialize();

            #if HANDLE_ALL_EXCEPTION
            {
                System.Windows.Forms.Application.ThreadException += OnThreadException;
                Thread.GetDomain().UnhandledException += OnUnhandledException;
            }
            #endif

            try
            {
                WorkingFolder = Environment.CurrentDirectory;

                // Check 64bit CPU
                if (Environment.Is64BitProcess == false)
                {
                    // メッセージボックスにエラーを表示
                    System.Windows.Forms.MessageBox.Show(
                        Properties.Resources.ErrorEnvironmentNot64Bit,
                        Properties.Resources.ErrorCaption,
                        MessageBoxButtons.OK,
                        MessageBoxIcon.Error);
                    return 0;
                }

                // マルチコアJITを有効にする
                ProfileOptimization.SetProfileRoot(IOConstants.ExecutableFolderPath);
                ProfileOptimization.StartProfile("App.JIT.Profile");

                // 動作モードを CUI にするか GUI にするか決定
                TheApp.IsCommandLineMode = TheApp.isConverter || CommandLineApplication.ShouldEnableCommandLine(args);

                // 自身のプロセスをテーブルに追加
                ProcessTable.AddMyProcess();

                if (TheApp.IsCommandLineMode == true)
                {
                    result = TheApp.RunCommandLineApplication(args);
                }
                else
                {
                    result = TheApp.RunWindowApplication(args) ? 0 : 1;
                }

                // 自身のプロセスをテーブルから削除
                ProcessTable.DeleteMyProcess();
            }
        #if HANDLE_ALL_EXCEPTION
            catch (Exception ex)
            {
                OnFatalErrorOccured(ex, 0);

                result = TheApp.IsCommandLineMode ? (int)CommandLineApplication.ExitCodeKind.ExceptionOccured : 1;
            }
        #else
            finally
            {
            }
        #endif

            // ロガーを破棄
            Logger.Release();

            return result;
        }

        /// <summary>
        /// Run command line application.
        /// </summary>
        /// <param name="args">Arguments provided by the command line.</param>
        /// <returns>True on success.</returns>
        private static int RunCommandLineApplication(string[] args)
        {
            TheApp.MainForm = null;

            // --debugオプションでアタッチダイアログを表示
            #if DEBUG
            if (args.Any(s => s == "--debug"))
            {
                args = args.Where(s => s != "--debug").ToArray();
                Debugger.Break();
            }
            #endif

            // SynchronizationContextを設定するためFormを作成.
            if (TheApp.isConverter)
            {
                TheApp.SyncContext = null;
            }
            else
            {
                using (var form = new Form())
                {
                    TheApp.SyncContext = SynchronizationContext.Current;
                }
            }

            bool result = false;

            // Initialize the application.
            try
            {
                TheApp.Mx.WaitOne();
                result = TheApp.Initialize();
            }
            finally
            {
                TheApp.Mx.ReleaseMutex();
            }

            // Run the application.
            if (result)
            {
                result = CommandLineApplication.Run(args);
            }

            CleanupTemporaryFolders();

            // CUI版はデバッグコンソールが自動で閉じられないため手動で閉じる
            #if DEBUG
            if (AppData.DebugConsole != null)
            {
                AppData.DebugConsole.Close();

                System.Windows.Forms.Application.DoEvents();  // フォームを確実に閉じるためウィンドウメッセージを処理
            }
            #endif

            return CommandLineApplication.ExitCode;
        }

        /// <summary>
        /// GUI モードでアプリケーションを実行します。
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        /// <returns>アプリケーションが正常に終了したとき true を返します。</returns>
        private static bool RunWindowApplication(string[] args)
        {
            Mutex guiMutex;

            // ミューテックスによって多重起動をブロックする
            {
                // 実行ファイルパスをキーに12文字のハッシュを生成
                byte[] hashKey = System.Text.Encoding.GetEncoding("Shift-JIS").GetBytes(IOConstants.ExecutableFolderPath);
                byte[] hashData = System.Security.Cryptography.SHA1.Create().ComputeHash(hashKey);
                string hashString = Convert.ToBase64String(hashData).Substring(0, 12);

                string mutexName = string.Format("NW_EffectMaker_FE2_GUI #{0}", hashString);

                // ミューテックスを作成
                bool createdNew;
                guiMutex = new Mutex(true, mutexName, out createdNew);

                if (createdNew == false)
                {
                    System.Windows.Forms.MessageBox.Show(
                        Resources.ErrorMultiLaunch,
                        Resources.WarningCaption,
                        MessageBoxButtons.OK,
                        MessageBoxIcon.Warning);

                    return false;
                }
            }

            List<string> startupFiles = new List<string>();

            // コマンドライン引数について処理
            {
                CompatibleCommandLineParams parameters = CommandLineApplication.ParseArgs(args, true);

                // プロジェクト設定ファイルパスを取得
                if (parameters != null && parameters.ProjectConfigFile != null)
                {
                    argProjectConfigPath = PathUtility.ToAbsolutePath(parameters.ProjectConfigFile, TheApp.WorkingFolder);
                }

                // アプリケーション起動時に開く中間ファイルパスを取得
                if (parameters != null && parameters.StartupFiles != null)
                {
                    foreach (var path in parameters.StartupFiles)
                    {
                        // 環境変数 & 相対パス & .bat からの相対パスに対応
                        var absPath = PathUtility.ToAbsolutePath(path, TheApp.WorkingFolder);

                        string extension = System.IO.Path.GetExtension(absPath);

                        if ((extension == IOConstants.EmitterSetFileExtension ||
                             extension == IOConstants.PreviewFileExtension ||
                             extension == IOConstants.WorkspaceFileExtension) &&
                            File.Exists(absPath))
                        {
                            startupFiles.Add(absPath);
                        }
                    }
                }
            }

            System.Windows.Forms.Application.EnableVisualStyles();
            System.Windows.Forms.Application.SetCompatibleTextRenderingDefault(false);

            // メインフォームを作成
            TheApp.MainForm = new MainForm();
            TheApp.SyncContext = SynchronizationContext.Current;

            // UILogViewをロガーに登録
            Logger.RegisterLogHandler(TheApp.MainForm.LogView);
            TheApp.MainForm.FormClosed += (s, e) => { Logger.UnregisterLogHandler(TheApp.MainForm.LogView); };

            // スプラッシュウィンドウを表示
            SplashForm.ShowSplash(TheApp.MainForm);

            // メインフォームがアクティブになったときの処理を設定
            SplashForm.MainFormActivated += (s, e) =>
            {
                // スタートアップファイルを開く
                TheApp.MainForm.LoadFiles(startupFiles);

                // 今日のヒントを表示
                if (OptionStore.RootOptions.Interface.UseTipsDialog)
                {
                    TipsDialog tipsDialog = new TipsDialog(true);

                    tipsDialog.Show(TheApp.MainForm);
                }
            };

            // Initialize the application.
            if (TheApp.Initialize() == true)
            {
                // オートバックアップを開始
                AppData.AutoBackupManager = new AutoBackupManager();
                AppData.AutoBackupManager.Initialize(TheApp.RootViewModel);

                // 通信サーバーを起動
                AppData.BridgeCommandProcessor = new BridgeCommandProcessor(TheApp.MainForm, TheApp.SyncContext);
                AppData.BridgeCommandProcessor.Initialize();

                // Add the global key event handler.
                System.Windows.Forms.Application.AddMessageFilter(
                    EffectMaker.UIControls.Input.GlobalKeyEventHandler.Instance);

                // Initialize the focus manager only on window mode.
                EffectMaker.UIControls.Focus.ControlFocusManager.Initialize();

                // アプリケーションを実行
                System.Windows.Forms.Application.Run(TheApp.MainForm);

                // 通信サーバーを停止
                AppData.BridgeCommandProcessor.Release();
                AppData.BridgeCommandProcessor = null;

                // オートバックアップを終了.
                AppData.AutoBackupManager.Release();
                AppData.AutoBackupManager = null;
            }

            TheApp.Shutdown();

            // ミューテックスを解放
            guiMutex.ReleaseMutex();

            return true;
        }

        /// <summary>
        /// Initialize the application.
        /// </summary>
        /// <returns>True on success.</returns>
        private static bool Initialize()
        {
            var timer = new ProfileTimer("Application initialization");

            SpecManager.Initialize(IOConstants.ExecutableFolderPath);
            BusinessLogic.Manager.TextureManager.InitializeOnStartUp();

            InitializeFoundamentalConfigurations();

            using (new ProfileTimer("Data model related initialization"))
            {
                var specificDataModelsAssembly = Assembly.LoadFrom(Path.Combine(
                    IOConstants.CoreModulesPath,
                    "EffectMaker.DataModel.Specific.dll"));

                DataModelPostConstructor.Initialize();
                DataModelSerializerManager.AddDataModelSerializers(specificDataModelsAssembly);
            }

            using (new ProfileTimer("G3D XSD base folder"))
            {
                if (Directory.Exists(IOConstants.G3dXsdBasePath) == false)
                {
                    Logger.Log("LogView", LogLevels.Warning, Resources.WarningG3dXsdFileNotFound);
                }

                // プリミティブマネージャの初期設定
                G3dIfLib.G3dIfUtility.XsdBasePath = IOConstants.G3dXsdBasePath;
                PrimitiveManager.Instance.AppFolderPath = IOConstants.CoreModulesPath;
            }

            using (new ProfileTimer("Register commands"))
            {
                // コマンドの登録を行う
                CommandManager.RegisterCommand(typeof(SetPropertyCommand));
                CommandManager.RegisterCommand(typeof(SetOptionCommand));
                CommandManager.RegisterCommand(typeof(RenameCommand));
                CommandManager.RegisterCommand(typeof(CreateEmitterCommand));
                CommandManager.RegisterCommand(typeof(CreateEmitterSetCommand));
                CommandManager.RegisterCommand(typeof(CreatePreviewCommand));
                CommandManager.RegisterCommand(typeof(RemoveEmitterCommand));
                CommandManager.RegisterCommand(typeof(RemoveEmitterSetCommand));
                CommandManager.RegisterCommand(typeof(RemovePreviewCommand));
                CommandManager.RegisterCommand(typeof(SaveEmitterSetCommand));
                CommandManager.RegisterCommand(typeof(SavePreviewCommand));
                CommandManager.RegisterCommand(typeof(LoadEmitterSetCommand));
                CommandManager.RegisterCommand(typeof(LoadPreviewCommand));
                CommandManager.RegisterCommand(typeof(CreateFieldCommand));
                CommandManager.RegisterCommand(typeof(RemoveFieldCommand));
                CommandManager.RegisterCommand(typeof(CreateReservedShaderCommand));
                CommandManager.RegisterCommand(typeof(RemoveReservedShaderCommand));
                CommandManager.RegisterCommand(typeof(CreateCustomActionCommand));
                CommandManager.RegisterCommand(typeof(RemoveCustomActionCommand));
                CommandManager.RegisterCommand(typeof(CreateEmitterExtParamsCommand));
                CommandManager.RegisterCommand(typeof(RemoveEmitterExtParamsCommand));
                CommandManager.RegisterCommand(typeof(SetDataModelCommand));
                CommandManager.RegisterCommand(typeof(AddHierarchicalNodeCommand));
                CommandManager.RegisterCommand(typeof(SwapNodesCommand));
                CommandManager.RegisterCommand(typeof(PasteGroupCommand));
                CommandManager.RegisterCommand(typeof(PastePropertyPageCommand));
                CommandManager.RegisterCommand(typeof(ResetDefaultValueCommand));
                CommandManager.RegisterCommand(typeof(ReopenEmitterSetCommand));
                CommandManager.RegisterCommand(typeof(UnparentEmitterCommand));

                // Undo/Redoバッファの最大サイズ
                CommandManager.SetUndoRedoParameters(
                    OptionStore.RootOptions.Basic.Undo.MaximumStackItems,
                    OptionStore.RootOptions.Basic.Undo.IsEnabled);
            }

            using (new ProfileTimer("Initialize user data manager"))
            {
                UserDataManager.Initialize(typeof(BinaryConversionInfoBase));
            }

            using (new ProfileTimer("Load project configurations"))
            {
                // 引数にプロジェクトファイル指定がある場合
                if (!string.IsNullOrEmpty(argProjectConfigPath))
                {
                    OptionStore.LoadProjectConfig(argProjectConfigPath);
                }
                // Load the project config.
                // This should be done after debug console is initialized,
                // in case any problem happened but cannot be logged to
                // the debug console.
                else
                {
                    var path = OptionStore.RootOptions.FileEvent.UserSettingsFilePath;
                    if (!string.IsNullOrEmpty(path) && !IsCommandLineMode)
                    {
                        OptionStore.LoadProjectConfig(OptionStore.RootOptions.FileEvent.UserSettingsFilePath);
                    }
                    else
                    {
                        OptionStore.InitializeProjectConfig();
                    }
                }

                // ファイルイベントのマージを行う
                OptionUtil.MergeFileEvent();
            }

            using (new ProfileTimer("User data related initializations"))
            {
                // Compose file paths for the user data managers.
                string generalUserDataListFilePath = Path.Combine(
                    IOConstants.ExecutableFolderPath,
                    "Addins",
                    "Addins.xml");

                // Compose file paths for the reserved shaders.
                string reservedShaderListFilePath = Path.Combine(
                    IOConstants.ExecutableFolderPath,
                    "Addins",
                    "ReservedShaders.xml");

                UserDataHelper.Instance.LoadGeneralUserData(generalUserDataListFilePath, true);
                UserDataHelper.Instance.LoadReservedShaderUserData(reservedShaderListFilePath, true);
            }

            using (new ProfileTimer("Initialize user data binary conversion info"))
            {
                BinaryConversionInfoManager.InitializeUserBinaryConversionInfo();
            }

            using (new ProfileTimer("Create workspace"))
            {
                // ルートビューモデルを作成
                RootViewModel = new WorkspaceRootViewModel();
                RootViewModel.CreateNewWorkspace();
            }

            using (new ProfileTimer("Initialize model info manager"))
            {
                // ModelInfoManagerを初期化
                ModelInfoManager.Initialize(
                    RootViewModel.GetChild<ViewerViewModel>().DataModel,
                    TheApp.SyncContext);
            }

            using (new ProfileTimer("Set up message manager"))
            {
                MessageManager.Instance.AddMessageNotifier(ModelInfoManager.Notifier);
                MessageManager.Instance.IsAutoConnectEnabled = OptionStore.RootOptions.Viewer.AutoConnect;
                CameraController.Initialize(TheApp.SyncContext);
            }

            using (new ProfileTimer("Initialize effect combiner communication manager"))
            {
                var combiner = EffectCombinerCommunicationManager.CommunicationBridge;
                if (combiner != null)
                {
                    combiner.Init();
                }
            }

            //// BTS:471
            //// Assembly.Load("Microsoft.Scripting");

            timer.Stop();

            // デバッグコンソールを描画するため強制的にウィンドウメッセージを処理する
            #if DEBUG
            if (AppData.DebugConsole != null)
            {
                System.Windows.Forms.Application.DoEvents();
            }
            #endif

            // Siglo上だとPreJitに失敗する(G3dIfLibがちゃんと初期化できない？)ので無効化
            ////PreJit();

            return true;
        }

        /// <summary>
        /// 事前JITする
        /// </summary>
        private static void PreJit()
        {
            Task.Factory.StartNew(() => BusinessLogic.Manager.TextureManager.Instance.PreJit(Properties.Resources.forPrejit_ftxb, ".ftxb"));
            Task.Factory.StartNew(() => BusinessLogic.Manager.PrimitiveManager.Instance.PreJit(Properties.Resources.forPrejit_fmdb, ".fmdb"));
        }

        /// <summary>
        /// Called when the application is about to shutdown.
        /// </summary>
        private static void Shutdown()
        {
            TcpConnecter.ShutdownPortMapper();

            var combiner = EffectCombinerCommunicationManager.CommunicationBridge;
            if (combiner != null)
            {
                combiner.Deinit();
            }

            // Save options.
            OptionStore.SaveRootOptions();

            // プロジェクトコンフィグの現状を固定ファイル名で保存するように
            OptionStore.SaveProjectConfig(Path.Combine(IOConstants.ExecutableFolderPath, "lastUsedPrjConfig.xml"));

            CleanupTemporaryFolders();

            // デバッグコンソールの破棄チェック
            #if DEBUG
            {
                Debug.Assert(AppData.DebugConsole == null);
            }
            #endif
        }

        /// <summary>
        /// アプリケーション終了時に消せる一時ファイルを削除します。
        /// インスタンスユニークな一時ファイルは問答無用で削除します。
        /// UserDataManager.csで生成されるハッシュコード付きのフォルダはここで削除します。
        /// </summary>
        private static void CleanupTemporaryFolders()
        {
            using (new GlobalSyncSection())
            {
                bool isSingleProcess = (ProcessTable.GetProcesses().Count() == 1);

                if (isSingleProcess)
                {
                    string tmpFolderPath = IOConstants.TemporaryFolderPath;
                    IOUtility.SafeDeleteDirectory(tmpFolderPath);
                }

                if (Directory.Exists(IOConstants.ObsoleteUserDataConvertFolderPath) && isSingleProcess)
                {
                    var oudcFolders = Directory.EnumerateDirectories(IOConstants.ObsoleteUserDataConvertFolderPath);
                    var deleteList = oudcFolders.Where(folder => folder.Contains(IOConstants.ApplicationInstanceHash)).ToArray();

                    foreach (var folder in deleteList)
                    {
                        IOUtility.SafeDeleteDirectory(folder);
                    }
                }

                IOUtility.SafeDeleteTemporaryDirectories();
            }
        }

        /// <summary>
        /// Initialize fundamental configurations.
        /// This code is not supposed to be modified, and all other initializations
        /// should happen after this.
        /// </summary>
        /// <returns>True on success.</returns>
        private static bool InitializeFoundamentalConfigurations()
        {
            using (new ProfileTimer("Create temp folder, synchronization context and debug console"))
            {
                // テンポラリフォルダを作成
                using (new GlobalSyncSection())
                {
                    IOUtility.SafeCreateTemporaryDirectory(IOConstants.AppDataWorkPath, IOUtility.TemporaryDirectoryUsage.ForGlobal);

                    IOUtility.SafeCreateDirectory(IOConstants.TemporaryFolderPath);
                }

                // UI言語初期化
                string dllFnm = Path.Combine(
                    EffectMaker.BusinessLogic.IO.IOConstants.ExecutableFolderPath,
                    @"en\EffectMaker.resources.dll");

                OptionStore.RuntimeOptions.CurrentCulture = File.Exists(dllFnm) ? "en-US" : "ja-JP";

                var cultureInfo = CultureInfo.CreateSpecificCulture(OptionStore.RuntimeOptions.CurrentCulture);
                Thread.CurrentThread.CurrentCulture = cultureInfo;
                Thread.CurrentThread.CurrentUICulture = cultureInfo;

                // Set synchronization context for the UI.
                UISynchronizationContextHolder.SynchronizationContext = TheApp.SyncContext;

                // デバッグコンソールを作成
                #if DEBUG
                {
                    AppData.DebugConsole = new DebugConsole();

                    Logger.RegisterLogHandler(AppData.DebugConsole);

                    AppData.DebugConsole.Show();

                    AppData.DebugConsole.SizeChanged += (s, e) => { OptionStore.RootOptions.Interface.DebugConsoleBounds = AppData.DebugConsole.Bounds; };
                    AppData.DebugConsole.LocationChanged += (s, e) => { OptionStore.RootOptions.Interface.DebugConsoleBounds = AppData.DebugConsole.Bounds; };

                    // コンソールを閉じたときにログハンドラの登録を消す
                    // FormClosedイベントは、アプリ終了時に呼ばれないのでHandleDestroyedイベントを使用
                    AppData.DebugConsole.HandleDestroyed += (s, e) =>
                    {
                        Logger.UnregisterLogHandler(AppData.DebugConsole);
                        AppData.DebugConsole = null;
                    };
                }
                #endif
            }

            using (new ProfileTimer("Initialize runtime compiler"))
            {
                string globalTempDirectoryPath = Path.Combine(IOConstants.AppDataWorkPath, "GeneratedAssemblies");
                string localTempDirectoryPath = Path.Combine(globalTempDirectoryPath, IOConstants.ProcessId);

                // テンポラリフォルダを作成
                using (new GlobalSyncSection())
                {
                    // 並列実行中でなければテンポラリフォルダを削除する
                    if (ProcessTable.GetProcesses().Count == 1)
                    {
                        IOUtility.SafeDeleteDirectory(globalTempDirectoryPath, true);
                    }

                    IOUtility.SafeCreateTemporaryDirectory(globalTempDirectoryPath, IOUtility.TemporaryDirectoryUsage.Manual);

                    // このプロセスで使用するテンポラリフォルダを作成
                    IOUtility.SafeDeleteDirectory(localTempDirectoryPath, true);
                    IOUtility.SafeCreateTemporaryDirectory(localTempDirectoryPath, IOUtility.TemporaryDirectoryUsage.Manual);
                }

                // ランタイムコンパイラを初期化
                RuntimeCompiler.Initialize(localTempDirectoryPath);
            }

            using (new ProfileTimer("Load application configurations"))
            {
                // load the root options
                OptionStore.LoadRootOptions();
            }

            // デバッグコンソールの位置とサイズを調整
            #if DEBUG
            using (new ProfileTimer("Adjust debug console"))
            {
                if (AppData.DebugConsole != null)
                {
                    Rectangle bounds = OptionStore.RootOptions.Interface.DebugConsoleBounds;

                    if (bounds.IsEmpty == false)
                    {
                        AppData.DebugConsole.Bounds = AppData.DebugConsole.AdjustBoundsToVisibleScreenArea(bounds);
                    }
                }
            }
            #endif

            return true;
        }

        /// <summary>
        /// Handle ThreadException event from application.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private static void OnThreadException(object sender, ThreadExceptionEventArgs e)
        {
            OnFatalErrorOccured(e.Exception, 1);
            System.Windows.Forms.Application.Exit();
        }

        /// <summary>
        /// Handle UnhandledException event from any thread.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private static void OnUnhandledException(object sender, UnhandledExceptionEventArgs e)
        {
            var exception = e.ExceptionObject as Exception;
            OnFatalErrorOccured(exception, 2);
            System.Windows.Forms.Application.Exit();
        }

        /// <summary>
        /// 致命的なエラーをダイアログに表示します。
        /// </summary>
        /// <param name="log">エラーログ。</param>
        private static void ShowFatalErrorDialog(string log)
        {
            var pos = new Point(20, 20);
            const int DlgWidth = 700;

            // ダイアログ作成
            Form dlg = new Form();
            dlg.SuspendLayout();

            // ラベル：致命的なエラーが発生しました。アプリケーションを終了します。
            Label label1 = new Label();
            label1.Text = EffectMaker.Application.Properties.Resources.ApplicationFatalErrorMessage;
            label1.Location = pos;
            label1.AutoSize = true;

            // ラベル：Error log:
            pos.Y += 50;
            Label label2 = new Label();
            label2.Text = "Error log:";
            label2.Location = pos;
            label2.AutoSize = true;

            // テキストボックス：エラーログ
            pos.Y += 20;
            TextBox txtBox = new TextBox();
            txtBox.Text = log;
            txtBox.Location = pos;
            txtBox.Size = new System.Drawing.Size(DlgWidth - 60, 200);
            txtBox.Multiline = true;
            txtBox.ReadOnly = true;
            txtBox.SelectionStart = 0;
            txtBox.ScrollBars = ScrollBars.Both;
            txtBox.Font = new Font(txtBox.Font.FontFamily, 7);

            // ボタン：ＯＫ
            pos.Y += txtBox.Size.Height + 20;
            Button btnOk = new Button() { Text = "OK" };
            btnOk.Click += (sender, e) => { dlg.Close(); };
            btnOk.Location = pos;

            // ボタン：ＣＯＰＹ
            Button btnCopy = new Button() { Text = "Copy" };
            btnCopy.Click += (sender, e) =>
            {
                SyncContext.Post(() => Clipboard.SetDataObject(log, true));
            };
            btnCopy.Location = new Point(pos.X + btnOk.Size.Width + 20, pos.Y);

            // コントロール配置
            dlg.Controls.Add(label1);
            dlg.Controls.Add(label2);
            dlg.Controls.Add(txtBox);
            dlg.Controls.Add(btnOk);
            dlg.Controls.Add(btnCopy);
            dlg.Icon = Icon.FromHandle(Resources.SymbolError1.GetHicon());
            dlg.Text = EffectMaker.Application.Properties.Resources.ErrorCaption;
            dlg.Width = DlgWidth;
            dlg.Height = pos.Y + 80;
            dlg.FormBorderStyle = FormBorderStyle.FixedSingle;
            dlg.ResumeLayout(false);
            dlg.PerformLayout();

            // ダイアログオープン
            dlg.ShowDialog();
        }

        /// <summary>
        /// Handle fatal error.
        /// </summary>
        /// <param name="error">The exception.</param>
        /// <param name="from">0: from WinMain, 1: from ThreadException, 2: from UnhandledException.</param>
        private static void OnFatalErrorOccured(Exception error, int from)
        {
            // Only handle once.
            if (fatalErrorHandled == true)
            {
                return;
            }

            fatalErrorHandled = true;
            var callerNames = new[] { "WinMain", "ThreadException", "UnhandledException" };

            // Output error message.
            try
            {
                string logFilePath =
                    EffectMaker.BusinessLogic.IO.IOConstants.ApplicationErrorLogPath;

                string exceptionMessage = GetErrorMessage(error);
                string stackTrace = GetStackTrace(error);

                // Delete the log file if exceeds the maximum size 100KB.
                FileInfo info = new FileInfo(logFilePath);
                if (info.Exists == true && info.Length >= 100 * 1000)
                {
                    File.Delete(logFilePath);
                }

                System.Text.StringBuilder sb = new System.Text.StringBuilder();

                using (var writer = new StreamWriter(logFilePath, true))
                {
                    sb.AppendLine("--------------------------------------------------------------------------------");

                    // ビルド日時
                    Assembly entry = Assembly.GetEntryAssembly();
                    Version version = entry.GetName().Version;
                    DateTime buildDate = new DateTime(2000, 1, 1, 0, 0, 0);
                    buildDate = buildDate.AddDays((double)version.Build);
                    buildDate = buildDate.AddSeconds((double)version.Revision * 2.0);
                    sb.AppendFormat(
                        "Build date : {0} {1}",
                        buildDate.ToShortDateString(),
                        buildDate.ToShortTimeString());
                    sb.AppendLine(string.Empty);

                    // エラーが起きた日時
                    sb.AppendFormat(
                        "Error date : {0} {1}",
                        DateTime.Now.ToShortDateString(),
                        DateTime.Now.ToShortTimeString());
                    sb.AppendLine(string.Empty);

                    sb.AppendLine("--------------------------------------------------------------------------------");
                    sb.AppendLine("[From]:");
                    sb.AppendLine("   " + callerNames[from]);
                    sb.AppendLine("[Message]:");
                    sb.AppendLine("   " + exceptionMessage);
                    sb.AppendLine("[StackTrace]:");
                    sb.AppendLine(stackTrace);
                    sb.AppendLine("[CommandStack]:");
                    CommandStack stack = CommandManager.GetActiveStack();
                    if (stack != null)
                    {
                        IEnumerable<CommandBase> undo = stack.UndoBuffer;
                        foreach (CommandBase cmd in undo)
                        {
                            sb.AppendLine("   " + cmd.ToString());
                        }
                    }
                    sb.AppendLine(string.Empty);

                    writer.Write(sb.ToString());
                }

                if (TheApp.IsCommandLineMode)
                {
                    OutputFatalErrorMessage(sb.ToString());
                }
                else
                {
                    ShowFatalErrorDialog(sb.ToString());
                }
            }
            catch
            {
                // Do nothing, we are already handling a fatal error...
            }
        }

        /// <summary>
        /// Get error message from the exception.
        /// </summary>
        /// <param name="error">The exception.</param>
        /// <returns>The error message.</returns>
        private static string GetErrorMessage(Exception error)
        {
            if (error == null)
            {
                return "Error information is not found.";
            }

            string result = error.Message + "\n";
            if (error.InnerException != null)
            {
                result += "\n    " + GetErrorMessage(error.InnerException);
            }

            return result;
        }

        /// <summary>
        /// Get stack trace from the exception.
        /// </summary>
        /// <param name="error">The exception.</param>
        /// <returns>The call stack.</returns>
        private static string GetStackTrace(Exception error)
        {
            if (error == null)
            {
                return "Error information is not found.";
            }

            string result = error.StackTrace + "\n";
            if (error.InnerException != null)
            {
                return GetStackTrace(error.InnerException) + "    " + result;
            }

            return result;
        }

        #region OutputFatalErrorMessage

        /// <summary>
        /// 致命的なエラーが発生したときのメッセージをコンソールに出力します。
        /// </summary>
        /// <param name="message">メッセージ</param>
        private static void OutputFatalErrorMessage(string message)
        {
            // コンソール画面にアタッチ
            bool attached = AttachConsole(AttachParentProcess);

            ConsoleColor originalColor = ConsoleColor.White;

            // 文字色を変更
            try
            {
                originalColor = Console.ForegroundColor;

                Console.ForegroundColor = ConsoleColor.Red;
            }
            catch
            {
            }

            // メッセージを出力
            Console.WriteLine(message);

            // 文字色を復元
            try
            {
                Console.ForegroundColor = originalColor;
            }
            catch
            {
            }

            // アタッチを解除
            if (attached)
            {
                FreeConsole();
            }
        }

        /// <summary>
        /// Win32 API Constant value: ATTACH_PARENT_PROCESS
        /// </summary>
        private const int AttachParentProcess = -1;

        //// <summary>
        //// Win32 API Function: AttachConsole(DWORD)
        //// </summary>
        [DllImport("kernel32.dll")]
        private static extern bool AttachConsole(int processId);

        //// <summary>
        //// Win32 API Function: AttachConsole()
        //// </summary>
        [DllImport("kernel32.dll")]
        private static extern bool AllocConsole();

        //// <summary>
        //// Win32 API Function: FreeConsole()
        //// </summary>
        [DllImport("kernel32.dll")]
        private static extern bool FreeConsole();

        #endregion
    }
}
