﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Windows.Forms;
using System.Xml;
using EffectMaker.Application.OptionPanes;
using EffectMaker.Application.Properties;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.Communicator;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Serialization;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Core;
using EffectMaker.UIControls.Specifics;
using EffectMaker.UIDialogs.MessageDialogs;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.Application
{
    /// <summary>
    /// A form to display option panes.
    /// </summary>
    public partial class OptionsForm : Form
    {
        /// <summary>
        /// Stores the automatically loaded option panes.
        /// </summary>
        private readonly IOptionPane[] optionPanes;


        /// <summary>
        /// オプションウィンドウを最初に開いた時の幅
        /// </summary>
        private int initialWidth;

        /// <summary>
        /// オプションウィンドウを最初に開いた時の高さ
        /// </summary>
        private int initialHeight;

        /// <summary>
        /// Initializes the OptionsForm instance.
        /// </summary>
        public OptionsForm()
        {
            this.InitializeComponent();

            this.btnAccept.Click += this.ButtonAcceptClick;
            this.btnCancel.Click += this.ButtonCancelClick;
            this.trvOptionPaneNodes.AfterSelect += this.TreeViewPaneNodesAfterSelect;

            this.optionPanes = this.LoadOptionPanes();
            this.BuildOptionPanesTree();

            this.FormClosed += (ss, ee) => this.UnloadOptionPanes();


            // ダイアログの非pane領域の高さを求める。
            int offset = this.Height - this.pnlOptionPaneContent.Height;
            // 各paneの高さを元に、スクロールする必要がないダイアログの高さを求める。
            foreach (var pane in this.optionPanes)
            {
                var control = pane as UserControl;
                if (control != null)
                {
                    this.Height = Math.Max(control.Height + offset, this.Height);
                }
            }
            // ダイアログの高さは最低でも650以上であるとする。
            this.Height = Math.Max(this.Height, 650);

            // オプションウィンドウのサイズを記録する設定だったら、最後に閉じたオプションウィンドウのサイズに復元する
            if (OptionStore.RootOptions.Interface.IsResizedOptionWindow == true)
            {
                this.Width = OptionStore.RootOptions.Interface.OptionFormBounds.Width;
                this.Height = OptionStore.RootOptions.Interface.OptionFormBounds.Height;
            }

            // オプションウィンドウを開いた時のサイズを記録しておく.
            this.initialWidth = this.Width;
            this.initialHeight = this.Height;
        }

        /// <summary>
        /// ビューアオプションの送信を行います。
        /// </summary>
        public static void SendViewerConfig()
        {
            int fps = 0;
            string fpsStr = OptionStore.RootOptions.Viewer.FrameRate;
            if (!string.IsNullOrEmpty(fpsStr) && ViewerOptions.FpsDictionary.ContainsKey(fpsStr))
            {
                fps = ViewerOptions.FpsDictionary[fpsStr];
            }

            int reso = 2;
            string resoStr = OptionStore.RootOptions.Viewer.WindowSize;
            if (!string.IsNullOrEmpty(resoStr) && ViewerOptions.WindowSizeDictionary.ContainsKey(resoStr))
            {
                reso = ViewerOptions.WindowSizeDictionary[resoStr];
            }

            ViewerController.Instance.SetOptionSettings(fps, reso, OptionStore.ProjectConfig.LinearMode);
            ViewerController.Instance.Reload();
        }

        /// <summary>
        /// The on load.
        /// </summary>
        /// <param name="e">
        /// The e.
        /// </param>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);
            var top = this.trvOptionPaneNodes.TopNode;
            this.trvOptionPaneNodes.SelectedNode = top;
            this.trvOptionPaneNodes.Select();
        }

        /// <summary>
        /// Called when the OptionsForm is closed.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            base.OnFormClosing(e);

            if (this.DialogResult == DialogResult.OK)
            {
                bool validation = true;

                foreach (IOptionPane pane in this.optionPanes)
                {
                    validation &= pane.OnAccept();
                }

                // 入力に不正がないか
                if (validation)
                {
                    // ビューアオプションの送信
                    OptionsForm.SendViewerConfig();
                }
                else
                {
                    // 入力に不正が会った場合、オプションウィンドウの終了処理を中止する.
                    e.Cancel = true;
                }
            }
            else
            {
                foreach (IOptionPane pane in this.optionPanes)
                {
                    pane.OnCancel();
                }
            }

            // オプションウィンドウのサイズを記録しておく.
            OptionStore.RootOptions.Interface.OptionFormBounds = this.Bounds;

            if (this.initialHeight != this.Bounds.Height || this.initialWidth != this.Bounds.Width)
            {
                OptionStore.RootOptions.Interface.IsResizedOptionWindow = true;
            }
        }

        /// <summary>
        /// Get the display order out of a given instance.
        /// This method looks for the DisplayOrderAttribute.
        /// </summary>
        /// <param name="instance">SingletonInstance to get the display order.</param>
        /// <returns>Returns the display order of the instance, or 0 if not available.</returns>
        private static int GetDisplayOrder(object instance)
        {
            var attr = instance.GetType()
                .GetCustomAttributes(typeof(DisplayOrderAttribute), true)
                .Cast<DisplayOrderAttribute>()
                .FirstOrDefault();

            return attr != null ? attr.Value : 0;
        }

        /// <summary>
        /// Called when the Accept button is clicked.
        /// </summary>
        /// <param name="sender">The called of the event.</param>
        /// <param name="e">The event argument.</param>
        private void ButtonAcceptClick(object sender, EventArgs e)
        {
            bool checkResult = true;
            foreach (var pane in this.optionPanes)
            {
                checkResult &= pane.OnValidationCheck();
            }

            if (!checkResult)
            {
                ThreadSafeMsgBox.Show(
                    Resources.OptionAcceptInvalidPathSetting,
                    Resources.OptionAcceptInvalidPathSettingCaption,
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
                return;
            }

            if (OptionUtil.IsNeedReloadCustomShader
                || OptionUtil.IsNeedReloadCustomAction
                || OptionUtil.IsNeedReloadEmitterExtParams)
            {
                var workspace = WorkspaceRootViewModel.Instance.WorkspaceViewModel.DataModel;
                if (workspace != null && workspace.EmitterSetList.Count != 0)
                {
                    var result = ThreadSafeMsgBox.Show(
                        Resources.OptionAcceptConfirm,
                        Resources.OptionAcceptConfirmCaption,
                        MessageBoxButtons.YesNo,
                        MessageBoxIcon.Warning);
                    if (result == DialogResult.No)
                    {
                        return;
                    }
                }
            }

            this.DialogResult = DialogResult.OK;
            this.Close();
        }

        /// <summary>
        /// Called when the Cancel button is clicked.
        /// </summary>
        /// <param name="sender">The caller of the event.</param>
        /// <param name="e">The event argument.</param>
        private void ButtonCancelClick(object sender, EventArgs e)
        {
            this.DialogResult = DialogResult.Cancel;
            this.Close();
        }

        /// <summary>
        /// Called when an option node is selected.
        /// </summary>
        /// <param name="sender">The caller of the event.</param>
        /// <param name="e">The event argument.</param>
        private void TreeViewPaneNodesAfterSelect(object sender, TreeViewEventArgs e)
        {
            this.pnlOptionPaneContent.AutoScrollPosition = new Point(0, 0);
            this.pnlOptionPaneContent.Controls.Clear();
            var optionPane = e.Node as OptionPaneTreeViewItem;

            if (optionPane != null)
            {
                var control = optionPane.OptionPane as Control;
                if (control != null)
                {
                    control.Width = this.pnlOptionPaneContent.Width;
                    this.pnlOptionPaneContent.Controls.Add(control);
                }

                // カスタムユーザーデータとスペックのパネルは、パネルサイズをオプションウィンドウにピッタリ合わせる
                var customShaderPane = optionPane.OptionPane as CustomShaderOptionPane;
                var customActionPane = optionPane.OptionPane as CustomActionOptionPane;
                var emitterExtParamsPane = optionPane.OptionPane as EmitterExtParamsOptionPane;
                var specOptionPane = optionPane.OptionPane as SpecOptionPane;
                if (customShaderPane != null || customActionPane != null || emitterExtParamsPane != null || specOptionPane != null)
                {
                    control.Anchor = AnchorStyles.Left | AnchorStyles.Top | AnchorStyles.Right | AnchorStyles.Bottom;
                    control.Height = this.pnlOptionPaneContent.Height;
                }
            }
        }

        /// <summary>
        /// Automatically discover available option panes and build up an option node tree.
        /// </summary>
        private void BuildOptionPanesTree()
        {
            OptionPaneTreeViewItem[] nodes = this.optionPanes
                .Select(p => new OptionPaneTreeViewItem(p))
                .ToArray();

            var hasChildren = false;

            var userConfigItem = nodes[0];
            userConfigItem.Name = "UserConfig";
            userConfigItem.Text = Properties.Resources.OptionSectionUserConfig;
            userConfigItem.NodeFont = new Font(FontFamily.GenericSansSerif, 10.0f, FontStyle.Bold);

            var prjConfigItem = nodes[5];
            prjConfigItem.Name = "ProjectConfig";
            prjConfigItem.Text = Properties.Resources.OptionSectionProjectConfig;
            prjConfigItem.NodeFont = new Font(FontFamily.GenericSansSerif, 10.0f, FontStyle.Bold);

            var specConfigItem = nodes[10];
            specConfigItem.Name = "SpecConfig";
            specConfigItem.Text = Properties.Resources.OptionSectionSpecConfig;
            specConfigItem.NodeFont = new Font(FontFamily.GenericSansSerif, 10.0f, FontStyle.Bold);

            this.trvOptionPaneNodes.Nodes.Add(userConfigItem);
            this.trvOptionPaneNodes.Nodes.Add(prjConfigItem);
            this.trvOptionPaneNodes.Nodes.Add(specConfigItem);

            using (new DrawingSuspender(this.trvOptionPaneNodes))
            {
                foreach (OptionPaneTreeViewItem node in nodes)
                {
                    node.OptionPane.DrawTreeView = this.Refresh;

                    if (node.GetHashCode() == userConfigItem.GetHashCode() ||
                        node.GetHashCode() == prjConfigItem.GetHashCode() ||
                        node.GetHashCode() == specConfigItem.GetHashCode())
                    {
                        continue;
                    }

                    if (node.OptionPane.ChildOf == null)
                    {
                        this.trvOptionPaneNodes.Nodes.Add(node);
                    }
                    else
                    {
                        ////OptionPaneTreeViewItem parent = nodes
                        ////    .Single(item => item.OptionPane.Identifier == node.OptionPane.ChildOf);

                        var parent = this.trvOptionPaneNodes.Nodes[node.OptionPane.ChildOf];

                        parent.Nodes.Add(node);
                        hasChildren = true;
                    }
                }

                this.trvOptionPaneNodes.ShowRootLines = hasChildren;

                if (hasChildren)
                {
                    this.trvOptionPaneNodes.ExpandAll();
                }

                foreach (IOptionPane pane in this.optionPanes)
                {
                    pane.OnInitialize();
                    var filePane = pane as ProjectBasicOptionPane;
                    if (filePane != null)
                    {
                        filePane.TriggerProjectSetting = this.NotifyUserSettingChanged;
                        filePane.ApplyProjectSettingForExport = this.ApplyProjectConfigForExport;
                    }

                    var userPane = pane as UserBasicOptionPane;
                    if (userPane != null)
                    {
                        userPane.TriggerProjectSetting = this.NotifyUserSettingChanged;
                    }
                }
            }
        }

        /// <summary>
        /// ユーザー設定の変更を反映します。
        /// </summary>
        private void NotifyUserSettingChanged()
        {
            foreach (var pane in this.optionPanes)
            {
                pane.OnProjectSettingChanged();
            }
        }

        /// <summary>
        /// プロジェクトコンフィグ出力用のインスタンスを更新します。
        /// </summary>
        private void ApplyProjectConfigForExport()
        {
            foreach (var pane in this.optionPanes)
            {
                pane.OnExportProjectSetting();
            }
        }

        /// <summary>
        /// Automatically discover the option panes and create an array of them.
        /// </summary>
        /// <returns>Returns an array of available IOptionPane instance.</returns>
        private IOptionPane[] LoadOptionPanes()
        {
            Assembly asm = Assembly.GetExecutingAssembly();

            var controlType = typeof(Control);
            var optionPaneInterfaceType = typeof(IOptionPane);

            var list = new List<IOptionPane>();

            foreach (Type type in asm.GetTypes())
            {
                bool match = true;

                match &= type.IsInterface == false;
                match &= controlType.IsAssignableFrom(type);
                match &= optionPaneInterfaceType.IsAssignableFrom(type);

                if (match == false)
                {
                    continue;
                }

                IOptionPane instance;

                try
                {
                    instance = (IOptionPane)Activator.CreateInstance(type);
                }
                catch (Exception)
                {
                    //// TODO: reoprt error

                    continue;
                }

                var ctrl = (Control)instance;
                ctrl.Anchor = AnchorStyles.Left | AnchorStyles.Top | AnchorStyles.Right;
                ctrl.Height = ctrl.Controls.Cast<Control>().Max(c => c.Bottom);

                list.Add(instance);
            }

            return list
                .OrderBy(GetDisplayOrder)
                .ThenBy(p => p.DisplayName)
                .ToArray();
        }

        /// <summary>
        /// Unloads all the loaded option panes.
        /// </summary>
        private void UnloadOptionPanes()
        {
            foreach (OptionPaneTreeViewItem node in
                this.trvOptionPaneNodes.Nodes.OfType<OptionPaneTreeViewItem>())
            {
                this.UnloadOptionPane(node);
            }
        }

        /// <summary>
        /// Unload the given option pane.
        /// </summary>
        /// <param name="item">The item to unload.</param>
        private void UnloadOptionPane(OptionPaneTreeViewItem item)
        {
            foreach (OptionPaneTreeViewItem child in
                item.Nodes.OfType<OptionPaneTreeViewItem>())
            {
                this.UnloadOptionPane(child);
            }

            item.Dispose();
        }
    }
}
