﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Application.Properties;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.ProjectConfig;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Specifics;
using EffectMaker.UIControls.Specifics.Behaviors;
using EffectMaker.UIDialogs.MessageDialogs;

namespace EffectMaker.Application.OptionPanes
{
    /// <summary>
    /// File event options.
    /// </summary>
    [DisplayOrder(9)]
    public partial class ProjectFileEventOptionPane : UserControl, IOptionPane
    {
        /// <summary>The list of the combiner definition input controls.</summary>
        private List<EffectCombinerDefInput> combinerDefInputControls =
            new List<EffectCombinerDefInput>();

        /// <summary>
        /// Initializes the FileEventOptionPane instance.
        /// </summary>
        public ProjectFileEventOptionPane()
        {
            this.InitializeComponent();

            foreach (var box in this.Controls.OfType<OptionGroupBox>())
            {
                box.BorderColor = Color.FromArgb(130, 150, 185);
            }

            this.btnPreSaveLoad.Click += this.OnPreSaveLoadButtonClick;
            this.btnPreSaveClear.Click += this.OnPreSaveClearButtonClick;

            this.btnPostSaveLoad.Click += this.OnPostSaveLoadButtonClick;
            this.btnPostSaveClear.Click += this.OnPostSaveClearButtonClick;

            this.btnPreDeleteLoad.Click += this.OnPreDeleteLoadButtonClick;
            this.btnPreDeleteClear.Click += this.OnPreDeleteClearButtonClick;

            this.btnPostDeleteLoad.Click += this.OnPostDeleteLoadButtonClick;
            this.btnPostDeleteClear.Click += this.OnPostDeleteClearButtonClick;

            this.btnPreOpenLoad.Click += this.OnPreOpenLoadButtonClick;
            this.btnPreOpenClear.Click += this.OnPreOpenClearButtonClick;

            OptionUtil.SetCommentLabelState(this.lblProjectFileEventComment);
            this.lblProjectFileEventComment.Width = 512;

            var pairs = new List<Tuple<UIFilePathTextBox, UIClearButton>>
            {
                new Tuple<UIFilePathTextBox, UIClearButton>(this.txtPreSaveCommand, this.btnPreSaveClear),
                new Tuple<UIFilePathTextBox, UIClearButton>(this.txtPostSaveCommand, this.btnPostSaveClear),
                new Tuple<UIFilePathTextBox, UIClearButton>(this.txtPreDeleteCommand, this.btnPreDeleteClear),
                new Tuple<UIFilePathTextBox, UIClearButton>(this.txtPostDeleteCommand, this.btnPostDeleteClear),
                new Tuple<UIFilePathTextBox, UIClearButton>(this.txtPreOpenCommand, this.btnPreOpenClear)
            };
            foreach (var pair in pairs)
            {
                var currentPair = pair;
                currentPair.Item2.Enabled = false;
                currentPair.Item1.BackColor = Color.White;

                // ファイルパスが変更された場合は、ファイルの有無をチェックして、ツリービューを更新する.
                currentPair.Item1.FilePathChanged += (s, e) =>
                {
                    OptionUtil.OnCheckFilePathExisting(s, e);
                    currentPair.Item2.Enabled = !string.IsNullOrEmpty(currentPair.Item1.FilePath);
                    if (this.DrawTreeView != null)
                    {
                        this.DrawTreeView();
                    }
                };
            }
        }

        /// <summary>
        /// オプションウィンドウのTreeViewを再描画するdelegate
        /// </summary>
        public Action DrawTreeView
        {
            get;
            set;
        }

        /// <summary>
        /// Gets the identifier.
        /// </summary>
        public string Identifier
        {
            get { return "ProjectFileEventOptionPane"; }
        }

        /// <summary>
        /// Gets the parent identifier.
        /// </summary>
        public string ChildOf
        {
            get { return "ProjectConfig"; }
        }

        /// <summary>
        /// Gets the displayname.
        /// </summary>
        public string DisplayName
        {
            get { return Properties.Resources.OptionPaneFileEvent; }
        }

        /// <summary>
        /// ユーザ設定の変更をトリガーします。
        /// </summary>
        public Action TriggerUserSetting { get; set; }

        /// <summary>
        /// テキストボックスへのドラッグアンドドロップ時の挙動について設定します.
        /// </summary>
        public void InitializeDragAndDrop()
        {
            // コマンド関連のTextBoxにドラッグアンドドロップできるファイルをbatファイルのみに限定する
            string allowExts = RegexUtility.ExtractExtensions(Properties.Resources.OptionDialogFilterBatFiles);

            // テキストボックスにドラッグ&ドロップ処理を設定するローカル関数
            var setDragDropBehavior = new Action<UIFilePathTextBox>((textBox) =>
            {
                FileDragDropBehavior dragDropBehavior = new FileDragDropBehavior();
                dragDropBehavior.AllowExtensions = allowExts;
                dragDropBehavior.DropResultChanged += (s, e) =>
                {
                    textBox.FilePath = dragDropBehavior.DropResult;
                };

                textBox.Behaviors.Add(dragDropBehavior);
            });

            // テキストボックスにドラッグ&ドロップ処理を設定
            setDragDropBehavior(this.txtPreSaveCommand);
            setDragDropBehavior(this.txtPostSaveCommand);
            setDragDropBehavior(this.txtPreDeleteCommand);
            setDragDropBehavior(this.txtPostDeleteCommand);
            setDragDropBehavior(this.txtPreOpenCommand);
        }

        /// <summary>
        /// Called when initialized.
        /// </summary>
        public void OnInitialize()
        {
            FileEventProjectConfig options = OptionStore.ProjectConfig.FileEventProject;

            this.txtPreSaveCommand.FilePath = options.PreSaveCommandFilePath;
            this.txtPostSaveCommand.FilePath = options.PostSaveCommandFilePath;
            this.txtPreDeleteCommand.FilePath = options.PreDeleteCommandFilePath;
            this.txtPostDeleteCommand.FilePath = options.PostDeleteCommandFilePath;
            this.txtPreOpenCommand.FilePath = options.PreOpenCommandFilePath;

            // ドラッグアンドドロップ時の設定
            this.InitializeDragAndDrop();
        }

        /// <summary>
        /// Called when option window is terminated.
        /// </summary>
        public void OnTerminate()
        {
        }

        /// <summary>
        /// Called when the Accept button is clicked.
        /// </summary>
        /// <returns>入力に不正がなければtrue,入力が不正であればfalse.</returns>
        public bool OnAccept()
        {
            FileEventProjectConfig options = OptionStore.ProjectConfig.FileEventProject;
            FileEventProjectConfig temp = OptionUtil.TempConfig.FileEventProject;

            options.PreSaveCommandFilePath = this.txtPreSaveCommand.FilePath;
            temp.PreSaveCommandFilePath = this.txtPreSaveCommand.FilePath;
            options.PostSaveCommandFilePath = this.txtPostSaveCommand.FilePath;
            temp.PostSaveCommandFilePath = this.txtPostSaveCommand.FilePath;
            options.PreDeleteCommandFilePath = this.txtPreDeleteCommand.FilePath;
            temp.PreDeleteCommandFilePath = this.txtPreDeleteCommand.FilePath;
            options.PostDeleteCommandFilePath = this.txtPostDeleteCommand.FilePath;
            temp.PostDeleteCommandFilePath = this.txtPostDeleteCommand.FilePath;
            options.PreOpenCommandFilePath = this.txtPreOpenCommand.FilePath;
            temp.PreOpenCommandFilePath = this.txtPreOpenCommand.FilePath;

            OptionUtil.MergeFileEvent();

            return true;
        }

        /// <summary>
        /// Called when the Cancel button is clicked.
        /// </summary>
        public void OnCancel()
        {
        }

        /// <summary>
        /// ユーザーコンフィグが読み込まれた際に、その内容にUIを更新します。
        /// </summary>
        public void OnProjectSettingChanged()
        {
            FileEventProjectConfig temp = OptionUtil.TempConfig.FileEventProject;

            this.txtPreSaveCommand.FilePath = temp.PreSaveCommandFilePath;
            this.txtPostSaveCommand.FilePath = temp.PostSaveCommandFilePath;
            this.txtPreDeleteCommand.FilePath = temp.PreDeleteCommandFilePath;
            this.txtPostDeleteCommand.FilePath = temp.PostDeleteCommandFilePath;
            this.txtPreOpenCommand.FilePath = temp.PreOpenCommandFilePath;
        }

        /// <summary>
        /// ファイルパスイベントの設定が有効か無効かを返す。
        /// </summary>
        /// <returns>
        /// 有効ならtrue、無効ならfalse.
        /// </returns>
        public bool OnValidationCheck()
        {
            bool result = true;

            result &= this.txtPreSaveCommand.BackColor == Color.White;
            result &= this.txtPostSaveCommand.BackColor == Color.White;
            result &= this.txtPreDeleteCommand.BackColor == Color.White;
            result &= this.txtPostDeleteCommand.BackColor == Color.White;
            result &= this.txtPreOpenCommand.BackColor == Color.White;

            return result;
        }

        /// <summary>
        /// プロジェクトコンフィグをファイル出力する際に各UIの内容を出力用インスタンスに収集します。
        /// </summary>
        public void OnExportProjectSetting()
        {
            FileEventProjectConfig temp = OptionUtil.TempConfig.FileEventProject;

            temp.PreSaveCommandFilePath = this.txtPreSaveCommand.FilePath;
            temp.PostSaveCommandFilePath = this.txtPostSaveCommand.FilePath;
            temp.PreDeleteCommandFilePath = this.txtPreDeleteCommand.FilePath;
            temp.PostDeleteCommandFilePath = this.txtPostDeleteCommand.FilePath;
            temp.PreOpenCommandFilePath = this.txtPreOpenCommand.FilePath;
        }

        /// <summary>
        /// 保存前コマンドロードボタン
        /// </summary>
        /// <param name="sender">s</param>
        /// <param name="e">e</param>
        private void OnPreSaveLoadButtonClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                CheckFileExists = true,
                CheckPathExists = true,
                Filter = string.Join(
                    "|",
                    Properties.Resources.OptionDialogFilterBatFiles,
                    Properties.Resources.OptionDialogFilterAllFiles),
                Multiselect = false,
                FileName = this.txtPreSaveCommand.FilePath,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.txtPreSaveCommand.FilePath = dlg.FileName;
        }

        /// <summary>
        /// 保存前コマンドクリアボタン
        /// </summary>
        /// <param name="sender">s</param>
        /// <param name="e">e</param>
        private void OnPreSaveClearButtonClick(object sender, EventArgs e)
        {
            this.txtPreSaveCommand.FilePath = string.Empty;
        }

        /// <summary>
        /// 保存後コマンドロードボタン
        /// </summary>
        /// <param name="sender">s</param>
        /// <param name="e">e</param>
        private void OnPostSaveLoadButtonClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                CheckFileExists = true,
                CheckPathExists = true,
                Filter = string.Join(
                    "|",
                    Properties.Resources.OptionDialogFilterBatFiles,
                    Properties.Resources.OptionDialogFilterAllFiles),
                Multiselect = false,
                FileName = this.txtPostSaveCommand.FilePath,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.txtPostSaveCommand.FilePath = dlg.FileName;
        }

        /// <summary>
        /// 保存後コマンドクリアボタン
        /// </summary>
        /// <param name="sender">s</param>
        /// <param name="e">e</param>
        private void OnPostSaveClearButtonClick(object sender, EventArgs e)
        {
            this.txtPostSaveCommand.FilePath = string.Empty;
        }

        /// <summary>
        /// 削除前コマンドロードボタン
        /// </summary>
        /// <param name="sender">s</param>
        /// <param name="e">e</param>
        private void OnPreDeleteLoadButtonClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                CheckFileExists = true,
                CheckPathExists = true,
                Filter = string.Join(
                    "|",
                    Properties.Resources.OptionDialogFilterBatFiles,
                    Properties.Resources.OptionDialogFilterAllFiles),
                Multiselect = false,
                FileName = this.txtPreDeleteCommand.FilePath,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.txtPreDeleteCommand.FilePath = dlg.FileName;
        }

        /// <summary>
        /// 削除前コマンドクリアボタン
        /// </summary>
        /// <param name="sender">s</param>
        /// <param name="e">e</param>
        private void OnPreDeleteClearButtonClick(object sender, EventArgs e)
        {
            this.txtPreDeleteCommand.FilePath = string.Empty;
        }

        /// <summary>
        /// 削除後コマンドロードボタン
        /// </summary>
        /// <param name="sender">s</param>
        /// <param name="e">e</param>
        private void OnPostDeleteLoadButtonClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                CheckFileExists = true,
                CheckPathExists = true,
                Filter = string.Join(
                    "|",
                    Properties.Resources.OptionDialogFilterBatFiles,
                    Properties.Resources.OptionDialogFilterAllFiles),
                Multiselect = false,
                FileName = this.txtPostDeleteCommand.FilePath,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.txtPostDeleteCommand.FilePath = dlg.FileName;
        }

        /// <summary>
        /// 削除後コマンドクリアボタン
        /// </summary>
        /// <param name="sender">s</param>
        /// <param name="e">e</param>
        private void OnPostDeleteClearButtonClick(object sender, EventArgs e)
        {
            this.txtPostDeleteCommand.FilePath = string.Empty;
        }

        /// <summary>
        /// オープン前コマンドロードボタン
        /// </summary>
        /// <param name="sender">s</param>
        /// <param name="e">e</param>
        private void OnPreOpenLoadButtonClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                CheckFileExists = true,
                CheckPathExists = true,
                Filter = string.Join(
                    "|",
                    Properties.Resources.OptionDialogFilterBatFiles,
                    Properties.Resources.OptionDialogFilterAllFiles),
                Multiselect = false,
                FileName = this.txtPreOpenCommand.FilePath,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.txtPreOpenCommand.FilePath = dlg.FileName;
        }

        /// <summary>
        /// オープン前コマンドクリアボタン
        /// </summary>
        /// <param name="sender">s</param>
        /// <param name="e">e</param>
        private void OnPreOpenClearButtonClick(object sender, EventArgs e)
        {
            this.txtPreOpenCommand.FilePath = string.Empty;
        }
    }
}
