﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Application.Properties;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.Specifics.Behaviors;

namespace EffectMaker.Application.OptionPanes
{
    /// <summary>
    /// Interface options.
    /// </summary>
    [DisplayOrder(1)]
    public partial class InterfaceOptionPane : UserControl, IOptionPane
    {
        /// <summary>
        /// Initializes the InterfaceOptionPane instance.
        /// </summary>
        public InterfaceOptionPane()
        {
            this.InitializeComponent();

            this.btnTextEditorBrowse.Click += this.OnTextEditorBrowseButtonClick;
            this.btnTextEditorClear.Click += this.OnTextEditorClearButtonClick;

            this.btnTextEditorClear.Enabled = false;

            // テキストエディタのファイルパスが変更された場合は、ファイルの有無をチェックして、ツリービューを更新する.
            this.txtTextEditor.FilePathChanged += (s, e) =>
            {
                this.btnTextEditorClear.Enabled = !string.IsNullOrEmpty(this.txtTextEditor.FilePath);
                OptionUtil.OnCheckFilePathExisting(s, e);
                if (this.DrawTreeView != null)
                {
                    this.DrawTreeView();
                }
            };

            this.txtTextEditor.BackColor = Color.White;
        }

        /// <summary>
        /// オプションウィンドウのTreeViewを再描画するdelegate
        /// </summary>
        public Action DrawTreeView
        {
            get;
            set;
        }

        /// <summary>
        /// Gets the identifier.
        /// </summary>
        public string Identifier
        {
            get { return "InterfaceOptionPane"; }
        }

        /// <summary>
        /// Gets the parent identifier.
        /// </summary>
        public string ChildOf
        {
            get { return "UserConfig"; }
        }

        /// <summary>
        /// Gets the display name.
        /// </summary>
        public string DisplayName
        {
            get { return Properties.Resources.OptionPaneInterface; }
        }

        /// <summary>
        /// Called when initialized.
        /// </summary>
        public void OnInitialize()
        {
            InterfaceOptions options = OptionStore.RootOptions.Interface;

            this.chkDialogBoxesWinXPStyle.Checked = options.UseWindowsXPStyle;
            this.chkTextBoxClickBehavior.Checked = options.ClickToSelectAllInTextBox;
            this.chkTipsDialog.Checked = options.UseTipsDialog;

            // テキストエディタ関連
            this.txtTextEditor.FilePath = options.TextEditorPath;

            // 使用するテキストエディタのドラッグ&ドロップ処理を設定
            {
                // テキストエディタの実行ファイル以外はドラッグアンドドロップできないようにする
                FileDragDropBehavior dragDropBehavior = new FileDragDropBehavior();
                dragDropBehavior.AllowExtensions = RegexUtility.ExtractExtensions(Properties.Resources.OptionDialogFilterExecutableFiles);
                dragDropBehavior.DropResultChanged += (s, e) =>
                {
                    this.txtTextEditor.FilePath = dragDropBehavior.DropResult;
                };

                this.txtTextEditor.Behaviors.Add(dragDropBehavior);
            }

            OptionUtil.SetCommentLabelState(this.lblDialogBoxesInfo);
        }

        /// <summary>
        /// Called when terminated.
        /// </summary>
        public void OnTerminate()
        {
        }

        /// <summary>
        /// Called when the Accept button is clicked.
        /// </summary>
        /// <returns>入力に不正がなければtrue,入力が不正であればfalse.</returns>
        public bool OnAccept()
        {
            InterfaceOptions options = OptionStore.RootOptions.Interface;

            options.UseWindowsXPStyle = this.chkDialogBoxesWinXPStyle.Checked;
            options.UseTipsDialog = this.chkTipsDialog.Checked;
            options.ClickToSelectAllInTextBox = this.chkTextBoxClickBehavior.Checked;
            options.TextEditorPath = this.txtTextEditor.FilePath;

            return true;
        }

        /// <summary>
        /// Called when the Cancel button is clicked.
        /// </summary>
        public void OnCancel()
        {
        }

        /// <summary>
        /// ユーザーコンフィグが読み込まれた際に、その内容にUIを更新します。
        /// </summary>
        public void OnProjectSettingChanged()
        {
        }

        /// <summary>
        /// テキストエディタのパス設定項目をチェックします。
        /// </summary>
        /// <returns>
        /// 有効ならtrue、無効ならfalse.
        /// </returns>
        public bool OnValidationCheck()
        {
            return this.txtTextEditor.BackColor == Color.White;
        }

        /// <summary>
        /// プロジェクトコンフィグをファイル出力する際に各UIの内容を出力用インスタンスに収集します。
        /// </summary>
        public void OnExportProjectSetting()
        {
        }

        /// <summary>
        /// Called when the TextEditorBrowse Button is clicked.
        /// </summary>
        /// <param name="sender">The TextEditorBrowse Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnTextEditorBrowseButtonClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                CheckFileExists = true,
                CheckPathExists = true,
                Filter = string.Join(
                    "|",
                    Properties.Resources.OptionDialogFilterExecutableFiles,
                    Properties.Resources.OptionDialogFilterAllFiles),
                Multiselect = false,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.txtTextEditor.FilePath = dlg.FileName;
        }

        /// <summary>
        /// Called when the TextEditorClear Button is clicked.
        /// </summary>
        /// <param name="sender">The TextEditorClear Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnTextEditorClearButtonClick(object sender, EventArgs e)
        {
            this.txtTextEditor.FilePath = string.Empty;
        }
    }
}
