﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.DataModel;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Log;
using EffectMaker.UIControls;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Specifics;
using EffectMaker.UIControls.Specifics.Behaviors;
using EffectMaker.UIControls.Specifics.TreeNodes;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.Application
{
    /// <summary>
    /// Class containing the workspace tree view and the controller buttons.
    /// </summary>
    public partial class WorkspaceControl : UIUserControl
    {
        /// <summary>
        /// Stores the workspace root view model.
        /// </summary>
        private WorkspaceRootViewModel root;

        /// <summary>
        /// Initializes the WorkspaceControl instance.
        /// </summary>
        public WorkspaceControl()
        {
            this.InitializeComponent();

            this.AddBinding("AvailableReservedShaders", "AvailableReservedShaders");

            this.trvWorkspace.AddBinding("ItemsSource", "Children");
            this.trvWorkspace.AddBinding("SelectedItem", "SelectedNodeViewModel");
            this.trvWorkspace.AddBehavior(new TreeNodeReorderingBehavior());
            this.trvWorkspace.ItemContainerSelector = new ProjectTreeItemContainerSelector();

            this.Controls.Add(this.trvWorkspace.RenameEditBox);

            // ファイルのドラッグアンドドロップを許可する.
            this.trvWorkspace.AllowDrop = true;

            this.trvWorkspace.DragEnter += (s, e) =>
            {
                if (e.Data.GetDataPresent(DataFormats.FileDrop))
                {
                    e.Effect = DragDropEffects.Copy;
                }
            };

            this.trvWorkspace.DragDrop += (s, e) =>
            {
                string[] files = (string[])e.Data.GetData(DataFormats.FileDrop);

                if (files == null)
                {
                    return;
                }

                if (this.ExecutableHolder == null)
                {
                    return;
                }

                // D&Dされたファイルを開く.
                this.ExecutableHolder.OnFileOpenExecutable.Execute(files);
            };

            this.trvWorkspace.DeleteAction = () =>
            {
                if (this.RootViewModel.CanRemoveSelectedNode)
                {
                    this.RootViewModel.Controller.Remove();
                }
            };
        }

        /// <summary>
        /// Raised when a node get selected or unselected of the project tree view.
        /// </summary>
        public event EventHandler<ValueChangedEventArgs<TreeNode>> SelectedNodeChanged
        {
            add { this.trvWorkspace.SelectedNodeChanged += value; }
            remove { this.trvWorkspace.SelectedNodeChanged -= value; }
        }

        /// <summary>
        /// ビューモデルのルートを取得または設定します。
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public WorkspaceRootViewModel RootViewModel
        {
            get
            {
                return this.root;
            }

            set
            {
                this.root = value;
                this.DataContext = this.root;

                this.InitializeToolBarButtons();
            }
        }

        /// <summary>
        /// ExecutableHolderを取得または設定します。
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public MainFormExecutableHolder ExecutableHolder { get; set; }

        /// <summary>
        /// Enumerates the available reserved shader items.
        /// (each item contains the name, the icon image, and the user data info)
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IEnumerable<Tuple<string, Image, object>> AvailableReservedShaders { get; set; }

        /// <summary>
        /// 名前の変更を行います.
        /// </summary>
        public void Rename()
        {
            this.trvWorkspace.BeginRename();
        }

        /// <summary>
        /// Called at constructor to initialize the toolbar buttons.
        /// </summary>
        private void InitializeToolBarButtons()
        {
            if (this.root == null)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "WorkspaceControl.InitializeToolBarButtons : Root view model is null.");
            }

            WorkspaceController controller = this.root.Controller;

            var toolBarItems = new List<ToolStripItem>();

            this.tsiOpen.Bindings.Add(new UIControls.DataBinding.Binder(this.tsiOpen, "Enabled", "CanLoadEmitterSet"));
            this.tsiOpen.DataContext = this.RootViewModel;

            this.tsiNewEmitter.Bindings.Add(new UIControls.DataBinding.Binder(this.tsiNewEmitter, "Enabled", "CanCreateNewEmitter"));
            this.tsiNewEmitter.DataContext = this.RootViewModel;
            this.tsiNewEmitter.Click += (ss, ee) => controller.CreateNewEmitter();

            this.tsiNewChildEmitter.Bindings.Add(new UIControls.DataBinding.Binder(this.tsiNewChildEmitter, "Enabled", "CanCreateNewChildEmitter"));
            this.tsiNewChildEmitter.DataContext = this.RootViewModel;
            this.tsiNewChildEmitter.Click += (ss, ee) => controller.CreateNewChildEmitter();

            // ランダム
            var tsmiRandomFe1 = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiRandomFe1);
            tsmiRandomFe1.Text = Properties.Resources.MenuItemNameRandomFe1;
            tsmiRandomFe1.Image = Properties.Resources.Icon_Field_Random;
            tsmiRandomFe1.ImageTransparentColor = Color.White;
            tsmiRandomFe1.Click += (s, e) =>
            {
                this.RootViewModel.LastAddNode = SubNodeTypes.RandomFe1;
                this.tsiNewSubNode.DefaultItem = tsmiRandomFe1;
                this.tsiNewSubNode.Image = Properties.Resources.Icon_Field_Random;
                this.RootViewModel.Controller.CreateNewField(FieldTypes.RandomFe1);
            };

            // 新ランダム
            var tsmiRandom = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiRandom);
            tsmiRandom.Text = Properties.Resources.MenuItemNameRandom;
            tsmiRandom.Image = Properties.Resources.Icon_Field_Wave;
            tsmiRandom.ImageTransparentColor = Color.White;
            tsmiRandom.Click += (s, e) =>
            {
                this.RootViewModel.LastAddNode = SubNodeTypes.Random;
                this.tsiNewSubNode.DefaultItem = tsmiRandom;
                this.tsiNewSubNode.Image = Properties.Resources.Icon_Field_Wave;
                this.RootViewModel.Controller.CreateNewField(FieldTypes.Random);
            };

            // 磁力
            var tsmiMagnet = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiMagnet);
            tsmiMagnet.Text = Properties.Resources.MenuItemNameMagness;
            tsmiMagnet.Image = Properties.Resources.Icon_Field_Magness;
            tsmiMagnet.ImageTransparentColor = Color.White;
            tsmiMagnet.Click += (s, e) =>
            {
                this.RootViewModel.LastAddNode = SubNodeTypes.Magnet;
                this.tsiNewSubNode.DefaultItem = tsmiMagnet;
                this.tsiNewSubNode.Image = Properties.Resources.Icon_Field_Magness;
                this.RootViewModel.Controller.CreateNewField(FieldTypes.Magnet);
            };

            // スピン
            var tsmiSpin = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiSpin);
            tsmiSpin.Text = Properties.Resources.MenuItemNameSpin;
            tsmiSpin.Image = Properties.Resources.Icon_Field_Spin;
            tsmiSpin.ImageTransparentColor = Color.White;
            tsmiSpin.Click += (s, e) =>
            {
                this.RootViewModel.LastAddNode = SubNodeTypes.Spin;
                this.tsiNewSubNode.DefaultItem = tsmiSpin;
                this.tsiNewSubNode.Image = Properties.Resources.Icon_Field_Spin;
                this.RootViewModel.Controller.CreateNewField(FieldTypes.Spin);
            };

            // 収束
            var tsmiConverge = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiConverge);
            tsmiConverge.Text = Properties.Resources.MenuItemNameConvergence;
            tsmiConverge.Image = Properties.Resources.Icon_Field_Convergence;
            tsmiConverge.ImageTransparentColor = Color.White;
            tsmiConverge.Click += (s, e) =>
            {
                this.RootViewModel.LastAddNode = SubNodeTypes.Converge;
                this.tsiNewSubNode.DefaultItem = tsmiConverge;
                this.tsiNewSubNode.Image = Properties.Resources.Icon_Field_Convergence;
                this.RootViewModel.Controller.CreateNewField(FieldTypes.Converge);
            };

            // 位置に加算
            var tsmiAddLocation = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiAddLocation);
            tsmiAddLocation.Text = Properties.Resources.MenuItemNameAddPos;
            tsmiAddLocation.Image = Properties.Resources.Icon_Field_AddPos;
            tsmiAddLocation.ImageTransparentColor = Color.White;
            tsmiAddLocation.Click += (s, e) =>
            {
                this.RootViewModel.LastAddNode = SubNodeTypes.AddLocation;
                this.tsiNewSubNode.DefaultItem = tsmiAddLocation;
                this.tsiNewSubNode.Image = Properties.Resources.Icon_Field_AddPos;
                this.RootViewModel.Controller.CreateNewField(FieldTypes.AddLocation);
            };

            // シンプルコリジョン
            var tsmiCollision = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiCollision);
            tsmiCollision.Text = Properties.Resources.MenuItemNameCollision;
            tsmiCollision.Image = Properties.Resources.Icon_Field_Collision;
            tsmiCollision.ImageTransparentColor = Color.White;
            tsmiCollision.Click += (s, e) =>
            {
                this.RootViewModel.LastAddNode = SubNodeTypes.Collision;
                this.tsiNewSubNode.DefaultItem = tsmiCollision;
                this.tsiNewSubNode.Image = Properties.Resources.Icon_Field_Collision;
                this.RootViewModel.Controller.CreateNewField(FieldTypes.Collision);
            };

            // カールノイズ
            var tsmiCurlNoise = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiCurlNoise);
            tsmiCurlNoise.Text = Properties.Resources.MenuItemNameCurlNoise;
            tsmiCurlNoise.Image = Properties.Resources.Icon_Field_CurlNoise;
            tsmiCurlNoise.ImageTransparentColor = Color.White;
            tsmiCurlNoise.Click += (s, e) =>
            {
                this.RootViewModel.LastAddNode = SubNodeTypes.CurlNoise;
                this.tsiNewSubNode.DefaultItem = tsmiCurlNoise;
                this.tsiNewSubNode.Image = Properties.Resources.Icon_Field_CurlNoise;
                this.RootViewModel.Controller.CreateNewField(FieldTypes.CurlNoise);
            };

            // カスタムフィールド
            var tsmiCustomField = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiCustomField);
            tsmiCustomField.Text = Properties.Resources.MenuItemNameCustomField;
            tsmiCustomField.Image = Properties.Resources.Icon_Field_Custom;
            tsmiCustomField.ImageTransparentColor = Color.White;
            tsmiCustomField.Click += (s, e) =>
            {
                this.RootViewModel.LastAddNode = SubNodeTypes.CustomField;
                this.tsiNewSubNode.DefaultItem = tsmiCustomField;
                this.tsiNewSubNode.Image = Properties.Resources.Icon_Field_Custom;
                this.RootViewModel.Controller.CreateNewField(FieldTypes.CustomField);
            };

            // ================= Separator =================
            toolBarItems.Add(new ToolStripSeparator());

            // Reserved shader user data items.
            var reservedShaderMenuItems = new List<UIToolStripMenuItem>();
            if (this.AvailableReservedShaders != null)
            {
                foreach (var item in this.AvailableReservedShaders)
                {
                    var tsmiReservedShader = new UIToolStripMenuItem();
                    toolBarItems.Add(tsmiReservedShader);
                    reservedShaderMenuItems.Add(tsmiReservedShader);
                    tsmiReservedShader.Text = string.Format(Properties.Resources.NodeEmitterPluginText, item.Item1);
                    tsmiReservedShader.Image = item.Item2;
                    tsmiReservedShader.Click += (s, e) =>
                    {
                        this.RootViewModel.LastAddNode = SubNodeTypes.ReservedShader;
                        this.tsiNewSubNode.DefaultItem = tsmiReservedShader;
                        this.tsiNewSubNode.Image = item.Item2;
                        this.RootViewModel.Controller.CreateNewReservedShader(item.Item3);
                    };
                }

                if (reservedShaderMenuItems.Count > 0)
                {
                    // ================= Separator =================
                    toolBarItems.Add(new ToolStripSeparator());
                }
            }

            // カスタムアクション
            var tsmiCustomAction = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiCustomAction);
            tsmiCustomAction.Text = Properties.Resources.MenuItemNameCustomAction;
            tsmiCustomAction.Image = Properties.Resources.Icon_CustomAction;
            tsmiCustomAction.Click += (s, e) =>
            {
                this.RootViewModel.LastAddNode = SubNodeTypes.CustomAction;
                this.tsiNewSubNode.DefaultItem = tsmiCustomAction;
                this.tsiNewSubNode.Image = Properties.Resources.Icon_CustomAction;
                this.RootViewModel.Controller.CreateNewCustomAction();
            };

            // エミッタ拡張パラメータ
            var tsmiEmitterExtParams = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiEmitterExtParams);
            tsmiEmitterExtParams.Text = Properties.Resources.MenuItemNameEmitterExtParams;
            tsmiEmitterExtParams.Image = Properties.Resources.Icon_EmitterExtParams;
            tsmiEmitterExtParams.Click += (s, e) =>
            {
                this.RootViewModel.LastAddNode = SubNodeTypes.EmitterExtParams;
                this.tsiNewSubNode.DefaultItem = tsmiEmitterExtParams;
                this.tsiNewSubNode.Image = Properties.Resources.Icon_EmitterExtParams;
                this.RootViewModel.Controller.CreateNewEmitterExtParams();
            };

            Binder randomFe1Binder = new UIControls.DataBinding.Binder(tsmiRandomFe1, "Enabled", "CanCreateNewRandomFe1Field");
            tsmiRandomFe1.Bindings.Add(randomFe1Binder);
            tsmiRandomFe1.DataContext = this.RootViewModel;
            Binder randomBinder = new UIControls.DataBinding.Binder(tsmiRandom, "Enabled", "CanCreateNewRandomField");
            tsmiRandom.Bindings.Add(randomBinder);
            tsmiRandom.DataContext = this.RootViewModel;
            Binder magnetBinder = new UIControls.DataBinding.Binder(tsmiMagnet, "Enabled", "CanCreateNewMagnetField");
            tsmiMagnet.Bindings.Add(magnetBinder);
            tsmiMagnet.DataContext = this.RootViewModel;
            Binder spinBinder = new UIControls.DataBinding.Binder(tsmiSpin, "Enabled", "CanCreateNewSpinField");
            tsmiSpin.Bindings.Add(spinBinder);
            tsmiSpin.DataContext = this.RootViewModel;
            Binder convergeBinder = new UIControls.DataBinding.Binder(tsmiConverge, "Enabled", "CanCreateNewConvergeField");
            tsmiConverge.Bindings.Add(convergeBinder);
            tsmiConverge.DataContext = this.RootViewModel;
            Binder addLocationBinder = new UIControls.DataBinding.Binder(tsmiAddLocation, "Enabled", "CanCreateNewAddLocationField");
            tsmiAddLocation.Bindings.Add(addLocationBinder);
            tsmiAddLocation.DataContext = this.RootViewModel;
            Binder collisionBinder = new UIControls.DataBinding.Binder(tsmiCollision, "Enabled", "CanCreateNewCollisionField");
            tsmiCollision.Bindings.Add(collisionBinder);
            tsmiCollision.DataContext = this.RootViewModel;
            Binder curlNoiseBinder = new UIControls.DataBinding.Binder(tsmiCurlNoise, "Enabled", "CanCreateNewCurlNoiseField");
            tsmiCurlNoise.Bindings.Add(curlNoiseBinder);
            tsmiCurlNoise.DataContext = this.RootViewModel;
            Binder customFieldBinder = new UIControls.DataBinding.Binder(tsmiCustomField, "Enabled", "CanCreateNewCustomField");
            tsmiCustomField.Bindings.Add(customFieldBinder);
            tsmiCustomField.DataContext = this.RootViewModel;
            Binder customActionBinder = new UIControls.DataBinding.Binder(tsmiCustomAction, "Enabled", "CanCreateNewCustomAction");
            tsmiCustomAction.Bindings.Add(customActionBinder);
            tsmiCustomAction.DataContext = this.RootViewModel;
            Binder emitterExtParamsBinder = new UIControls.DataBinding.Binder(tsmiEmitterExtParams, "Enabled", "CanCreateNewEmitterExtParams");
            tsmiEmitterExtParams.Bindings.Add(emitterExtParamsBinder);
            tsmiEmitterExtParams.DataContext = this.RootViewModel;

            foreach (var item in reservedShaderMenuItems)
            {
                Binder reservedShaderItemBinder = new UIControls.DataBinding.Binder(
                    item,
                    "Enabled",
                    "CanCreateNewReservedShader");

                item.Bindings.Add(reservedShaderItemBinder);
                item.DataContext = this.RootViewModel;
            }

            this.tsiNewSubNode.DropDownItems.AddRange(toolBarItems.ToArray());

            this.tsiNewSubNode.DefaultItem = tsmiRandomFe1;
            this.tsiNewSubNode.Image = Properties.Resources.Icon_Field_Random;
            this.tsiNewSubNode.DrawEnabledButton = false;
            this.tsiNewSubNode.ToolTipText = Properties.Resources.ToolTipTextAddNode;

            tsiNewSubNode.AddBinding("DrawEnabledButton", "CanAddLastNode");
            tsiNewSubNode.AddBinding("EnableButtons", "CanAddAnotherNode");
            tsiNewSubNode.DataContext = this.RootViewModel;
            tsiNewSubNode.OverlayImage = true;
            tsiNewSubNode.AdditionalImage = Properties.Resources.Icon_AnotherNode_Add;
            tsiNewSubNode.ButtonSelectedImage = Properties.Resources.Icon_LeftButton_Selected;
            tsiNewSubNode.ButtonNonSelectedImage = Properties.Resources.Icon_LeftButton_NonSelected;
            tsiNewSubNode.ButtonDownImage = Properties.Resources.Icon_LeftButton_Down;
            tsiNewSubNode.DropDownButtonSelectedImage = Properties.Resources.Icon_DropDownButton_Selected;
            tsiNewSubNode.DropDownButtonImage = Properties.Resources.Icon_DropDownButton;
            tsiNewSubNode.DropDownButtonNonSelectedImage = Properties.Resources.Icon_DropDownButton_NonSelected;
            tsiNewSubNode.DropDownMenuOpenedImage = Properties.Resources.Icon_DropDownMenu_Opened;
            tsiNewSubNode.UseOriginalStyle = true;

            this.tsiNewEmitterSet.Bindings.Add(new UIControls.DataBinding.Binder(this.tsiNewEmitterSet, "Enabled", "CanCreateNewEmitterSet"));
            this.tsiNewEmitterSet.DataContext = this.RootViewModel;
            this.tsiNewEmitterSet.Click += (ss, ee) => controller.CreateNewEmitterSet();

            this.tsiNewPreview.Bindings.Add(new UIControls.DataBinding.Binder(this.tsiNewPreview, "Enabled", "CanCreateNewPreview"));
            this.tsiNewPreview.DataContext = this.RootViewModel;
            this.tsiNewPreview.Click += (ss, ee) => controller.CreateNewPreview();

            this.tsiRemove.Bindings.Add(new UIControls.DataBinding.Binder(
                this.tsiRemove,
                "ToolTipText",
                "RemoveButtonText"));
            this.tsiRemove.Bindings.Add(new UIControls.DataBinding.Binder(this.tsiRemove, "Enabled", "CanRemoveSelectedNode"));
            this.tsiRemove.DataContext = this.RootViewModel;
            this.tsiRemove.Click += (ss, ee) => controller.Remove();

            this.tsiClone.Bindings.Add(new UIControls.DataBinding.Binder(this.tsiClone, "Enabled", "CanDuplicateSelectedNode"));
            this.tsiClone.DataContext = this.RootViewModel;
            this.tsiClone.Click += (ss, ee) => controller.DuplicateSelectedNode();

            this.tsiSaveAll.Bindings.Add(new Binder(this.tsiSaveAll, "Enabled", "CanSaveAnyNode"));
            this.tsiSaveAll.DataContext = this.RootViewModel;
            this.tsiSaveAll.Click += (ss, ee) => this.RootViewModel.OnFileSaveAll(null);

            // ツールチップ設定
            this.tsiOpen.ToolTipText = Properties.Resources.ToolTipTextLoadFile;
            this.tsiSaveAll.ToolTipText = Properties.Resources.ToolTipTextSaveAll;
            this.tsiNewEmitterSet.ToolTipText = Properties.Resources.ToolTipTextNewEmitterSet;
            this.tsiNewEmitter.ToolTipText = Properties.Resources.ToolTipTextNewEmitter;
            this.tsiNewChildEmitter.ToolTipText = Properties.Resources.ToolTipTextNewChildEmitter;
            this.tsiNewPreview.ToolTipText = Properties.Resources.ToolTipTextNewPreview;
            this.tsiRemove.ToolTipText = Properties.Resources.ToolTipTextClose;
            this.tsiClone.ToolTipText = Properties.Resources.ToolTipTextDuplicate;
        }

        /// <summary>
        /// ファイルを開く。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">Event arguments.</param>
        private void OnFileOpen(object sender, EventArgs e)
        {
            if (this.ExecutableHolder == null)
            {
                return;
            }

            this.ExecutableHolder.OnFileOpenExecutable.Execute(null);
        }
    }
}
