﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Options;

namespace EffectMaker.Application
{
    /// <summary>
    /// プレビューコントロール
    /// </summary>
    public partial class ViewerPlaybackControl : UserControl
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ViewerPlaybackControl()
        {
            this.InitializeComponent();

            this.btnBack.Click += this.OnBackButtonClicked;
            this.btnStop.Click += this.OnStopButtonClicked;
            this.btnPlay.Click += this.OnPlayButtonClicked;
            this.btnStep.Click += this.OnStepButtonClicked;
            this.btnFade.Click += this.OnFadeButtonClicked;
            this.btnLoop.Click += this.OnLoopButtonClicked;
            this.txtFrame.TextChanged += this.OnFrameTextChanged;
            this.txtFrame.KeyPress += this.OnFrameTextKeyPress;
            this.txtFrame.Leave += this.OnFrameTextLeave;

            this.btnLoop.Checked = OptionStore.RootOptions.Viewer.LoopMode;
            this.OnLoopButtonClicked(null, null);
            this.txtFrame.Text = OptionStore.RootOptions.Viewer.LoopFrame.ToString("D");
            this.OnLoopButtonClicked(null, null);
        }

        /// <summary>
        /// ループ関係のイベントハンドラです。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        public delegate void LoopEventHandler(object sender, LoopEventArgs e);

        /// <summary>
        /// Backボタンを押されたときに発生します。
        /// </summary>
        public event EventHandler BackButtonClicked;

        /// <summary>
        /// Stopボタンを押されたときに発生します。
        /// </summary>
        public event EventHandler StopButtonClicked;

        /// <summary>
        /// Playボタンを押されたときに発生します。
        /// </summary>
        public event EventHandler PlayButtonClicked;

        /// <summary>
        /// Stepボタンを押されたときに発生します。
        /// </summary>
        public event EventHandler StepButtonClicked;

        /// <summary>
        /// Fadeボタンを押されたときに発生します。
        /// </summary>
        public event EventHandler FadeButtonClicked;

        /// <summary>
        /// Loopボタンを押されたときに発生します。
        /// </summary>
        public event LoopEventHandler LoopButtonClicked;

        /// <summary>
        /// フレーム数が変更されたときに発生します。
        /// </summary>
        public event LoopEventHandler LoopFrameChanged;

        /// <summary>
        /// Backボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnBackButtonClicked(object sender, EventArgs e)
        {
            if (this.BackButtonClicked != null)
            {
                this.BackButtonClicked(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Stopボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnStopButtonClicked(object sender, EventArgs e)
        {
            if (this.StopButtonClicked != null)
            {
                this.StopButtonClicked(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// PlayPauseボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnPlayButtonClicked(object sender, EventArgs e)
        {
            if (this.PlayButtonClicked != null)
            {
                this.PlayButtonClicked(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Stepボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnStepButtonClicked(object sender, EventArgs e)
        {
            if (this.StepButtonClicked != null)
            {
                this.StepButtonClicked(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Fadeボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnFadeButtonClicked(object sender, EventArgs e)
        {
            if (this.FadeButtonClicked != null)
            {
                this.FadeButtonClicked(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Loopボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnLoopButtonClicked(object sender, EventArgs e)
        {
            bool isLoop = this.btnLoop.Checked;
            OptionStore.RootOptions.Viewer.LoopMode = isLoop;

            // アイコンを更新
            this.btnLoop.Image = isLoop ? EffectMaker.Application.Properties.Resources.Icon_PrevCtrl_Loop_On
                : EffectMaker.Application.Properties.Resources.Icon_PrevCtrl_Loop_Off;

            // フレームテキストボックスの入力可/不可を切り替え
            this.txtFrame.Enabled = isLoop;

            if (this.LoopButtonClicked != null)
            {
                int frame;
                bool valid = int.TryParse(this.txtFrame.Text, out frame);

                if (!valid || frame < 1)
                {
                    return;
                }

                this.LoopButtonClicked(this, new LoopEventArgs(isLoop, frame));
                OptionStore.RootOptions.Viewer.LoopFrame = frame;
            }
        }

        /// <summary>
        /// Loopボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnFrameTextChanged(object sender, EventArgs e)
        {
            if (this.LoopFrameChanged != null)
            {
                bool doLoop = this.btnLoop.Checked;
                int frame = 0;
                bool valid = int.TryParse(this.txtFrame.Text, out frame);

                if (!valid || frame < 1)
                {
                    return;
                }

                this.LoopFrameChanged(this, new LoopEventArgs(doLoop, frame));
                OptionStore.RootOptions.Viewer.LoopFrame = frame;
            }
        }

        /// <summary>
        /// 0～9とバックスペース以外の時は、イベントをキャンセルする.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void OnFrameTextKeyPress(object sender, KeyPressEventArgs e)
        {
            if ((e.KeyChar < '0' || '9' < e.KeyChar) && e.KeyChar != '\b')
            {
                e.Handled = true;
            }
        }

        /// <summary>
        /// テキストボックスからフォーカスが外れた時に値を確定する.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">e</param>
        private void OnFrameTextLeave(object sender, EventArgs e)
        {
            int frame = 0;
            bool valid = int.TryParse(this.txtFrame.Text, out frame);

            if (!valid || frame < 1)
            {
                frame = 1;
            }

            this.txtFrame.Text = frame.ToString("D");

            this.LoopFrameChanged(this, new LoopEventArgs(this.btnLoop.Checked, frame));
            OptionStore.RootOptions.Viewer.LoopFrame = frame;
        }

        /// <summary>
        /// ループ設定のイベント情報。
        /// </summary>
        public class LoopEventArgs : EventArgs
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="isLoop">ループのOn/Off</param>
            /// <param name="frame">フレーム数</param>
            public LoopEventArgs(bool isLoop, int frame)
            {
                this.IsLoop = isLoop;
                this.Frame = frame;
            }

            /// <summary>
            /// ループのOn/Off
            /// </summary>
            public bool IsLoop
            {
                get;
                private set;
            }

            /// <summary>
            /// フレーム数
            /// </summary>
            public int Frame
            {
                get;
                private set;
            }
        }
    }
}
