﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.Application
{
    /// <summary>
    /// MainFormExecutableHolder.
    /// </summary>
    public class MainFormExecutableHolder : IBindable
    {
        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>
        /// Backing field for Controls property.
        /// (should be empty, just to satisfy ILogicalTreeElement)
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// メインフォームです.
        /// </summary>
        private MainForm mainForm;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="dataContext">DataContext.</param>
        /// <param name="mainForm">MainForm.</param>
        public MainFormExecutableHolder(ViewModelBase dataContext, MainForm mainForm)
        {
            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);

            // UIから呼び出すIExecutableをバインド
            this.Bindings.Add(new Binder(this, "OnCreateNewEmitterSetExecutable", "OnCreateNewEmitterSetExecutable"));
            this.Bindings.Add(new Binder(this, "OnCloseAllExecutable", "OnCloseAllExecutable"));
            this.Bindings.Add(new Binder(this, "OnCreateNewEmitterExecutable", "OnCreateNewEmitterExecutable"));
            this.Bindings.Add(new Binder(this, "OnCreateNewPreviewExecutable", "OnCreateNewPreviewExecutable"));
            this.Bindings.Add(new Binder(this, "OnRemoveNodeExecutable", "OnRemoveNodeExecutable"));
            this.Bindings.Add(new Binder(this, "OnFileSaveExecutable", "OnFileSaveExecutable"));
            this.Bindings.Add(new Binder(this, "OnFileSaveAllExecutable", "OnFileSaveAllExecutable"));
            this.Bindings.Add(new Binder(this, "OnFileSaveAllAsExecutable", "OnFileSaveAllAsExecutable"));
            this.Bindings.Add(new Binder(this, "OnFileOpenExecutable", "OnFileOpenExecutable"));
            this.Bindings.Add(new Binder(this, "OnWorkspaceSaveExecutable", "OnWorkspaceSaveExecutable"));
            this.Bindings.Add(new Binder(this, "OnWorkspaceOpenExecutable", "OnWorkspaceOpenExecutable"));
            this.Bindings.Add(new Binder(this, "OnLinkEmitterSetExecutable", "OnLinkEmitterSetExecutable"));
            this.Bindings.Add(new Binder(this, "OnUnlinkEmitterSetExecutable", "OnUnlinkEmitterSetExecutable"));
            this.Bindings.Add(new Binder(this, "OnUnlinkAllEmitterSetsExecutable", "OnUnlinkAllEmitterSetsExecutable"));

            // MainDialogsをバインド
            this.Bindings.Add(new Binder(this, "Dialogs", "Dialogs") { Mode = BindingMode.OneWayToSource });
            this.Dialogs = new MainFormDialogs(mainForm);

            // メンバプロパティをバインド
            this.Bindings.Add(new Binder(this, "CanSaveAnyNode", "CanSaveAnyNode"));

            this.DataContext = dataContext;

            this.mainForm = mainForm;
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public ILogicalTreeElement Parent
        {
            get; private set;
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new NullIndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// メインダイアログです。
        /// </summary>
        public MainFormDialogs Dialogs { get; set; }

        /// <summary>
        /// 全てのエミッタセットを閉じるIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnCloseAllExecutable { get; set; }

        /// <summary>
        /// エミッタセットを新規作成するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnCreateNewEmitterSetExecutable { get; set; }

        /// <summary>
        /// エミッタを新規作成するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnCreateNewEmitterExecutable { get; set; }

        /// <summary>
        /// プレビューを新規作成するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnCreateNewPreviewExecutable { get; set; }

        /// <summary>
        /// ノードを削除するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnRemoveNodeExecutable { get; set; }

        /// <summary>
        /// ファイルを保存するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnFileSaveExecutable { get; set; }

        /// <summary>
        /// ファイルを全て上書き保存するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnFileSaveAllExecutable { get; set; }

        /// <summary>
        /// ファイルを全てエクスポートするIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnFileSaveAllAsExecutable { get; set; }

        /// <summary>
        /// ファイルを開くIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnFileOpenExecutable { get; set; }

        /// <summary>
        /// ワークスペースを保存するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnWorkspaceSaveExecutable { get; set; }

        /// <summary>
        /// ワークスペースを読み込むIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnWorkspaceOpenExecutable { get; set; }

        /// <summary>
        /// エミッタセットにリンクを作成するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnLinkEmitterSetExecutable { get; set; }

        /// <summary>
        /// エミッタセットのリンクを削除するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnUnlinkEmitterSetExecutable { get; set; }

        /// <summary>
        /// 全てのエミッタセットのリンクを削除するIExecutableを取得または設定します。
        /// </summary>
        public IExecutable OnUnlinkAllEmitterSetsExecutable { get; set; }

        /// <summary>
        /// 上書き保存可能なノードの有無を取得します。
        /// </summary>
        public bool CanSaveAnyNode
        {
            get
            {
                if (this.mainForm != null)
                {
                    return this.mainForm.SaveAllEnabled;
                }
                else
                {
                    return false;
                }
            }

            set
            {
                if (this.mainForm != null)
                {
                    this.mainForm.SaveAllEnabled = value;
                }
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }
    }
}
