﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Windows.Forms;
using EffectMaker.Application.OptionalMessageBox;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Collections;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Input;
using EffectMaker.UIDialogs.ExportDialog;
using EffectMaker.UIDialogs.MessageDialogs;
using EffectMaker.UILogic.Manager;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.Application
{
    /// <summary>
    /// メインフォームのダイアログです。
    /// </summary>
    public class MainFormDialogs : MainDialogs
    {
        /// <summary>
        /// メインフォームです.
        /// </summary>
        private MainForm mainForm;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="mainForm">MainForm.</param>
        public MainFormDialogs(MainForm mainForm)
        {
            this.mainForm = mainForm;

            // ファイルの保存
            this.OnShowSaveModificationDialogExecutable = new AnonymousExecutable(this.OnShowSaveModificationDialog);
            this.OnShowSaveEmitterSetDialogExecutable = new AnonymousExecutable(this.OnShowSaveEmitterSetDialog);
            this.OnShowSavePreviewDialogExecutable = new AnonymousExecutable(this.OnShowSavePreviewDialog);
            this.OnShowSaveWorkspaceDialogExecutable = new AnonymousExecutable(this.OnShowSaveWorkspaceDialog);
            this.OnShowExportDialogExecutable = new AnonymousExecutable(this.OnShowExportDialog);

            // カーブエディタ呼び出し
            this.OnShowCurveEditorExecutable = new AnonymousExecutable(this.OnShowCurveEditor);
            this.OnConnectCurveEditorExecutable = new AnonymousExecutable(this.OnConnectCurveEditor);
            this.OnDisconnectCurveEditorExecutable = new AnonymousExecutable(this.OnDisconnectCurveEditor);
            this.OnNotifyEmitterToCurveEditorExecutable = new AnonymousExecutable(this.OnNotifyEmitterToCurveEditor);
            this.OnNotifyTabPageToCurveEditorExecutable = new AnonymousExecutable(this.OnNotifyTabPageToCurveEditor);

            // ファイルの読み込み
            this.OnShowFileOpenDialogExecutable = new AnonymousExecutable(this.OnShowFileOpenDialog);
            this.OnShowFileOpenErrorExecutable = new AnonymousExecutable(this.OnShowFileOpenError);
            this.OnShowCustomActionIndexUpdatedDialogExecutable = new AnonymousExecutable(this.OnShowCustomActionIndexUpdatedDialog);
            this.OnShowPreviewFileOpenDialogExecutable = new AnonymousExecutable(this.OnShowPreviewFileOpenDialog);
            this.OnShowWorkspaceFileOpenDialogExecutable = new AnonymousExecutable(this.OnShowWorkspaceFileOpenDialog);
            this.OnShowEsetFileNotFoundDialogExecutable = new AnonymousExecutable(this.OnShowEsetFileNotFoundDialog);
            this.OnShowPreviewFileAlreadyExistsDialogExecutable = new AnonymousExecutable(this.OnShowPreviewFileAlreadyExistsDialog);
            this.OnShowAncientEsetUpdaterDialogExecutable = new AnonymousExecutable(this.OnShowAncientEsetUpdaterDialog);

            // 名前入力
            this.OnShowEmitterSetNameInputDialogExecutable = new AnonymousExecutable(this.OnShowEmitterSetNameInputDialog);
            this.OnShowEmitterNameInputDialogExecutable = new AnonymousExecutable(this.OnShowEmitterNameInputDialog);
            this.OnShowPreviewNameInputDialogExecutable = new AnonymousExecutable(this.OnShowPreviewNameInputDialog);
            this.OnShowWarningInvalidNameExecutable = new AnonymousExecutable(this.OnShowWarningInvalidName);
            this.OnShowWarningNameAlreadyUsedExecutable = new AnonymousExecutable(this.OnShowWarningNameAlreadyUsed);

            // アセット処理
            this.OnShowTextureTypeMismatchExecutable = new AnonymousExecutable(this.OnShowTextureTypeMismatch);
            this.OnShowWarningOnLoadTextureExecutable = new AnonymousExecutable(this.OnShowWarningOnLoadTexture);
            this.OnShowWarningOnLoadPrimitiveExecutable = new AnonymousExecutable(this.OnShowWarningOnLoadPrimitive);
            this.OnShowWarningTextureIsNotAssignedExecutable = new AnonymousExecutable(this.OnShowWarningTextureIsNotAssigned);
            this.OnPromptOverWriteAssetExecutable = new AnonymousExecutable(this.OnPromptOverWriteAsset);
            this.OnShowCopyAssetDialogExecutable = new AnonymousExecutable(this.OnShowCopyAssetDialog);
            this.OnShowAssetPathProblemExecutable = new AnonymousExecutable(this.OnShowAssetPathProblem);
            this.OnShowInvalidEndingTypeExecutable = new AnonymousExecutable(this.OnShowInvalidEndingType);
            this.OnShowInvalidDrawPathDialogExecutable = new AnonymousExecutable(this.OnShowInvalidDrawPathDialog);

            // その他
            this.OnShowErrorDialogExecutable = new AnonymousExecutable(this.OnShowErrorDialog);
            this.OnShowWarningDialogExecutable = new AnonymousExecutable(this.OnShowWarningDialog);
            this.OnShowOkCancelDialogExecutable = new AnonymousExecutable(this.OnShowOkCancelDialog);
        }

        #region Function - ファイルの保存

        /// <summary>
        /// 変更保存ダイアログを表示します。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        public void OnShowSaveModificationDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // メッセージボックスを表示
            DialogResult result = ThreadSafeMsgBox.Show(
                (string)parameters[1],
                (string)parameters[2],
                MessageBoxButtons.YesNoCancel,
                MessageBoxIcon.Question);

            // 結果を記録
            parameters[0] = result;
        }

        /// <summary>
        /// エミッタセット保存ダイアログを表示します。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        public void OnShowSaveEmitterSetDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // ファイル保存ダイアログを作成
            SaveFileDialog dialog = new SaveFileDialog();

            dialog.AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle;
            dialog.FileName = (string)parameters[1];

            var initialDirectory = (string)parameters[2];

            // 初期ディレクトリを設定(指定がなかったら最後に保存したパスを指定)
            dialog.InitialDirectory = string.IsNullOrEmpty(initialDirectory) ?
                ApplicationIOManager.GetLastAccessedDir(LastAccessDirectoryTypes.EmitterSet) :
                initialDirectory;

            // フィルタを設定
            dialog.Filter = IOConstants.EmitterSetDialogFilter;

            // ダイアログを表示
            DialogResult result = dialog.ShowDialog(null);

            if (result != DialogResult.OK)
            {
                return;
            }

            // 初期ディレクトリを更新
            ApplicationIOManager.SetLastAccessedDir(
                LastAccessDirectoryTypes.EmitterSet,
                dialog.FileName);

            // 結果を記録
            parameters[0] = dialog.FileName;
        }

        /// <summary>
        /// プレビュー保存ダイアログを表示します。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        public void OnShowSavePreviewDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // ファイル保存ダイアログを作成
            SaveFileDialog dialog = new SaveFileDialog();

            dialog.AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle;
            dialog.FileName = (string)parameters[1];

            // 初期ディレクトリを設定
            dialog.InitialDirectory = ApplicationIOManager.GetLastAccessedDir(
                LastAccessDirectoryTypes.Preview);

            // フィルタを設定
            dialog.Filter = IOConstants.PreviewDialogFilter;

            // ダイアログを表示
            DialogResult result = dialog.ShowDialog(null);

            if (result != DialogResult.OK)
            {
                return;
            }

            // 初期ディレクトリを更新
            ApplicationIOManager.SetLastAccessedDir(
                LastAccessDirectoryTypes.Preview,
                dialog.FileName);

            // 結果を記録
            parameters[0] = dialog.FileName;
        }

        /// <summary>
        /// ワークスペース保存ダイアログを表示します。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        public void OnShowSaveWorkspaceDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // ファイル保存ダイアログを作成
            SaveFileDialog dialog = new SaveFileDialog();

            dialog.AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle;
            dialog.FileName = (string)parameters[1];

            // 初期ディレクトリを設定
            dialog.InitialDirectory = ApplicationIOManager.GetLastAccessedDir(
                LastAccessDirectoryTypes.Workspace);

            // フィルタを設定
            dialog.Filter = IOConstants.WorkspaceDialogFilter;

            // ダイアログを表示
            DialogResult result = dialog.ShowDialog(null);

            if (result != DialogResult.OK)
            {
                return;
            }

            // 初期ディレクトリを更新
            ApplicationIOManager.SetLastAccessedDir(
                LastAccessDirectoryTypes.Workspace,
                dialog.FileName);

            // 結果を記録
            parameters[0] = dialog.FileName;
        }

        /// <summary>
        /// エクスポートダイアログを表示します。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        public void OnShowExportDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // ファイル保存ダイアログを作成
            var dialog = new ExportDialog((List<string>)parameters[1]);

            // ダイアログを表示
            DialogResult result = dialog.ShowDialog(this.mainForm);

            if (result == DialogResult.OK)
            {
                // ダイアログからの返り値をセット
                parameters[0] = new List<List<FileCopyInfo>>()
                {
                    dialog.EsetSaveList,
                    dialog.PrevSaveList,
                    dialog.FileCopyList,
                };
            }
        }

        #endregion

        #region Function - カーブエディタ呼び出し

        /// <summary>
        /// The on show curve editor.
        /// </summary>
        /// <param name="pamameter">
        /// 使用しません.
        /// </param>
        private void OnShowCurveEditor(object pamameter)
        {
            TheApp.MainForm.ShowCurveEditor();
        }

        /// <summary>
        /// The on connect curve editor.
        /// </summary>
        /// <param name="dataContext">
        /// The data Context.
        /// </param>
        private void OnConnectCurveEditor(object dataContext)
        {
            TheApp.MainForm.ConnectCurveEditor(dataContext);
        }

        /// <summary>
        /// The on disconnect curve editor.
        /// </summary>
        /// <param name="dataContext">
        /// The data Context.
        /// </param>
        private void OnDisconnectCurveEditor(object dataContext)
        {
            TheApp.MainForm.DisconnectCurveEditor(dataContext);
        }

        /// <summary>
        /// 変更されたエミッタのViewModelをカーブエディタウィンドウに通知する
        /// </summary>
        /// <param name="dataContext">
        /// The data Context.
        /// </param>
        private void OnNotifyEmitterToCurveEditor(object dataContext)
        {
            TheApp.MainForm.NotifyEmitterToCurveEditor(dataContext);
        }

        /// <summary>
        /// タブページの変更をカーブエディタウィンドウに通知する
        /// </summary>
        /// <param name="dataContext">
        /// The data Context.
        /// </param>
        private void OnNotifyTabPageToCurveEditor(object dataContext)
        {
            TheApp.MainForm.NotifyTabPageToCurveEditor(dataContext);
        }

        #endregion

        #region Function - ファイルの読み込み

        /// <summary>
        /// ファイルを開くダイアログを表示します。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        private void OnShowFileOpenDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // ファイルを開くダイアログを作成
            OpenFileDialog dialog = new OpenFileDialog();

            dialog.AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle;
            dialog.Multiselect = true;

            // 初期ディレクトリを設定
            dialog.InitialDirectory = ApplicationIOManager.GetLastAccessedDir(
                LastAccessDirectoryTypes.Any);

            // フィルタを設定
            dialog.Filter = IOConstants.OpenFileDialogFilter;

            // ダイアログを表示
            DialogResult result = dialog.ShowDialog(null);

            if (result != DialogResult.OK)
            {
                return;
            }

            // 初期ディレクトリを更新
            ApplicationIOManager.SetLastAccessedDir(
                LastAccessDirectoryTypes.Any,
                dialog.FileName);

            // 結果を記録
            parameters[0] = dialog.FileNames;
        }

        /// <summary>
        /// ファイル保存時のエラーを表示します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : メッセージボックスに表示する説明
        /// [1] : メッセージボックスのキャプション
        /// </param>
        private void OnShowFileOpenError(object parameter)
        {
            object[] parameters = (object[])parameter;

            // メッセージボックスを表示
            ThreadSafeMsgBox.Show(
                (string)parameters[0],
                (string)parameters[1],
                MessageBoxButtons.OK,
                MessageBoxIcon.Warning);
        }

        /// <summary>
        /// カスタムアクションのコールバックIDが書き換えられたことを表示します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : メッセージボックスに表示する説明
        /// [1] : メッセージボックスのキャプション
        /// </param>
        private void OnShowCustomActionIndexUpdatedDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // メッセージボックスを表示
            ThreadSafeMsgBox.Show(
                (string)parameters[0],
                (string)parameters[1],
                MessageBoxButtons.OK,
                MessageBoxIcon.Information);
        }

        /// <summary>
        /// プレビューファイルを開くダイアログを表示します。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        private void OnShowPreviewFileOpenDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // ファイルを開くダイアログを作成
            OpenFileDialog dialog = new OpenFileDialog();

            dialog.AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle;
            dialog.Multiselect = false;

            // 初期ディレクトリを設定
            dialog.InitialDirectory = ApplicationIOManager.GetLastAccessedDir(
                LastAccessDirectoryTypes.Preview);

            // フィルタを設定
            dialog.Filter = IOConstants.PreviewDialogFilter;

            // ダイアログを表示
            DialogResult result = dialog.ShowDialog(null);

            if (result != DialogResult.OK)
            {
                return;
            }

            // 初期ディレクトリを更新
            ApplicationIOManager.SetLastAccessedDir(
                LastAccessDirectoryTypes.Preview,
                dialog.FileName);

            // 結果を記録
            parameters[0] = dialog.FileName;
        }

        /// <summary>
        /// ワークスペースファイルを開くダイアログを表示します。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        private void OnShowWorkspaceFileOpenDialog(object parameter)
        {
            var parameters = (object[])parameter;

            // ファイルを開くダイアログを作成
            var dialog = new OpenFileDialog
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                Multiselect = false,

                // 初期ディレクトリを設定
                InitialDirectory = ApplicationIOManager.GetLastAccessedDir(
                    LastAccessDirectoryTypes.Workspace),

                // フィルタを設定
                Filter = IOConstants.WorkspaceDialogFilter
            };

            // ダイアログを表示
            DialogResult result = dialog.ShowDialog(null);

            if (result != DialogResult.OK)
            {
                return;
            }

            // 初期ディレクトリを更新
            ApplicationIOManager.SetLastAccessedDir(
                LastAccessDirectoryTypes.Workspace,
                dialog.FileName);

            // 結果を記録
            parameters[0] = dialog.FileName;
        }

        /// <summary>
        /// プレビューに関連したエミッタセットファイルが見つからないことを通知します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : メッセージボックスに表示する説明
        /// [1] : メッセージボックスのキャプション
        /// </param>
        private void OnShowEsetFileNotFoundDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // メッセージボックスを表示
            ThreadSafeMsgBox.Show(
                (string)parameters[0],
                (string)parameters[1],
                MessageBoxButtons.OK,
                MessageBoxIcon.Exclamation);
        }

        /// <summary>
        /// 読み込んだファイルと同じ名前のプレビューがエミッタセットにあることを通知します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : メッセージボックスに表示する説明
        /// [1] : メッセージボックスのキャプション
        /// </param>
        private void OnShowPreviewFileAlreadyExistsDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // メッセージボックスを表示
            ThreadSafeMsgBox.Show(
                (string)parameters[0],
                (string)parameters[1],
                MessageBoxButtons.OK,
                MessageBoxIcon.Exclamation);
        }

        /// <summary>
        /// FE1世代のesetアップデート中にプログレスダイアログを表示します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : メッセージ
        /// [1] : キャプション
        /// [2] : アップデートメソッド
        /// </param>
        private void OnShowAncientEsetUpdaterDialog(object parameter)
        {
            var parameters = (object[])parameter;

            // ファイルを開くダイアログを作成
            var dialog = new AncientEsetUpdaterDialog(
                (string)parameters[0],
                (string)parameters[1],
                (Action)parameters[2]);

            // ダイアログを表示してアップデートを実行
            dialog.ShowAndRun();
        }

        #endregion

        #region Function - 名前入力

        /// <summary>
        /// エミッタセット名入力ダイアログを表示します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : DialogResult
        /// [1] : 入力した名前
        /// [2] : キャプション
        /// [3] : デフォルト名
        /// </param>
        private void OnShowEmitterSetNameInputDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // 名前入力ダイアログを作成
            var dialog = new NameAndPresetInputDialog(
                (string)parameters[2],
                (string)parameters[3],
                (List<PresetItem>)parameters[4],
                (IEnumerable<string>)parameters[6]);

            dialog.SelectedPreset = (int)parameters[5];

            // ダイアログを表示
            DialogResult result = dialog.ShowDialog(this.mainForm);

            // 結果を記録
            parameters[0] = result;
            parameters[1] = dialog.InputedName;
            parameters[2] = dialog.SelectedPreset;
        }

        /// <summary>
        /// エミッタ名入力ダイアログを表示します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : DialogResult
        /// [1] : 入力した名前
        /// [2] : キャプション
        /// [3] : デフォルト名
        /// </param>
        private void OnShowEmitterNameInputDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // 名前入力ダイアログを作成
            var dialog = new NameAndPresetInputDialog(
                (string)parameters[2],
                (string)parameters[3],
                (List<PresetItem>)parameters[4],
                (IEnumerable<string>)parameters[6]);

            dialog.SelectedPreset = (int)parameters[5];

            // ダイアログを表示
            DialogResult result = dialog.ShowDialog(this.mainForm);

            // 結果を記録
            parameters[0] = result;
            parameters[1] = dialog.InputedName;
            parameters[2] = dialog.SelectedPreset;
        }

        /// <summary>
        /// プレビュー名入力ダイアログを表示します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : DialogResult
        /// [1] : 入力した名前
        /// [2] : キャプション
        /// [3] : デフォルト名
        /// </param>
        private void OnShowPreviewNameInputDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // 名前入力ダイアログを作成
            var dialog = new NameInputDialog(
                (string)parameters[2],
                (string)parameters[3]);

            // ダイアログを表示
            DialogResult result = dialog.ShowDialog();

            // 結果を記録
            parameters[0] = result;
            parameters[1] = dialog.InputedName;
        }

        /// <summary>
        /// 入力された名前が不正なことを表示します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : メッセージボックスに表示する説明
        /// [1] : メッセージボックスのキャプション
        /// </param>
        private void OnShowWarningInvalidName(object parameter)
        {
            object[] parameters = (object[])parameter;

            // メッセージボックスを表示
            ThreadSafeMsgBox.Show(
                (string)parameters[0],
                (string)parameters[1],
                MessageBoxButtons.OK,
                MessageBoxIcon.Exclamation);
        }

        /// <summary>
        /// 入力された名前が使用済みであることを表示します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : メッセージボックスに表示する説明
        /// [1] : メッセージボックスのキャプション
        /// </param>
        private void OnShowWarningNameAlreadyUsed(object parameter)
        {
            object[] parameters = (object[])parameter;

            // メッセージボックスを表示
            ThreadSafeMsgBox.Show(
                (string)parameters[0],
                (string)parameters[1],
                MessageBoxButtons.OK,
                MessageBoxIcon.Exclamation);
        }

        #endregion

        #region Function - アセット処理

        /// <summary>
        /// テクスチャがエミッタセットの探索範囲外にあることを通知します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : ユーザーの選択
        /// [1] : メッセージボックスに表示する説明
        /// </param>
        private void OnShowWarningOnLoadTexture(object parameter)
        {
            object[] parameters = (object[])parameter;

            DialogResult result = DialogResult.OK;

            // メッセージボックスを表示
            using (OptionalMessageBoxContext context = new OptionalMessageBoxContext("WarningOnLoadTexture"))
            {
                // 詳細ボタンを押した時に、ヘルプを表示する
                EffectMaker.UIDialogs.MessageDialogs.MessageBox.AssignDetailButtonClickHandler(
                    delegate
                    {
                        HelpManager.ShowHelp("104469580");
                    });

                // メッセージボックスをひらく
                result = ThreadSafeMsgBox.ShowEx(
                    (string)parameters[1],
                    MessageBoxButtonsEx.OKDetail,
                    MessageBoxIcon.Warning,
                    MessageBoxDefaultButton.Button1);
            }

            EffectMaker.UIDialogs.MessageDialogs.MessageBox.AssignDetailButtonClickHandler(null);

            // 結果を記録
            parameters[0] = result;
        }

        /// <summary>
        /// プリミティブがエミッタセットの探索範囲外にあることを通知します。
        /// </summary>
        /// <param name="parameter">
        /// Customr parameter.
        /// [0] : ユーザーの選択
        /// [1] : メッセージボックスに表示する説明
        /// </param>
        private void OnShowWarningOnLoadPrimitive(object parameter)
        {
            object[] parameters = (object[])parameter;

            DialogResult result = DialogResult.OK;

            // メッセージボックスを表示
            using (OptionalMessageBoxContext context = new OptionalMessageBoxContext("WarningOnLoadPrimitive"))
            {
                // 詳細ボタンを押した時に、ヘルプを表示する
                EffectMaker.UIDialogs.MessageDialogs.MessageBox.AssignDetailButtonClickHandler(
                    delegate
                    {
                        HelpManager.ShowHelp("104469579");
                    });

                // メッセージボックスをひらく
                result = ThreadSafeMsgBox.ShowEx(
                    (string)parameters[1],
                    MessageBoxButtonsEx.OKDetail,
                    MessageBoxIcon.Warning,
                    MessageBoxDefaultButton.Button1);

                EffectMaker.UIDialogs.MessageDialogs.MessageBox.AssignDetailButtonClickHandler(null);
            }

            // 結果を記録
            parameters[0] = result;
        }

        /// <summary>
        /// テクスチャのリニアモードがオプションと違うことを通知します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : メッセージボックスに表示する説明
        /// </param>
        private void OnShowTextureTypeMismatch(object parameter)
        {
            object[] parameters = (object[])parameter;

            // メッセージボックスを表示
            using (OptionalMessageBoxContext context = new OptionalMessageBoxContext("LinearEditTextureWarning"))
            {
                ThreadSafeMsgBox.ShowEx(
                    (string)parameters[0],
                    MessageBoxButtonsEx.OK,
                    MessageBoxIcon.Warning,
                    MessageBoxDefaultButton.Button1);
            }
        }

        /// <summary>
        /// テクスチャが設定されておらず保存をキャンセルするかどうか選択するダイアログを表示します。
        /// </summary>
        /// <param name="parameter">Custom parameter.
        /// [0] : ユーザーの選択
        /// [1] : メッセージボックスに表示する説明
        /// [2] : キャンセルボタンの表示On/Off
        /// </param>
        private void OnShowWarningTextureIsNotAssigned(object parameter)
        {
            object[] parameters = (object[])parameter;

            bool showCancel = (bool)parameters[2];
            MessageBoxButtonsEx buttons = showCancel ? MessageBoxButtonsEx.YesNoCancel : MessageBoxButtonsEx.YesNo;

            DialogResult result = DialogResult.OK;

            // メッセージボックスを表示
            using (OptionalMessageBoxContext context = new OptionalMessageBoxContext("WarningTextureIsNotAssigned"))
            {
                result = ThreadSafeMsgBox.ShowEx(
                    (string)parameters[1],
                    buttons,
                    MessageBoxIcon.Exclamation,
                    MessageBoxDefaultButton.Button1);
            }

            // 結果を記録
            parameters[0] = result;
        }

        /// <summary>
        /// アセットファイルの上書き確認を行います。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        private void OnPromptOverWriteAsset(object parameter)
        {
            object[] parameters = (object[])parameter;

            // ダイアログを表示して確認を取る
            DialogResult dialogResult =
                ThreadSafeOverwriteAssetDialog.Show(
                (string)parameters[1],
                (DateTime)parameters[2],
                (string)parameters[3],
                (DateTime)parameters[4]);

            // 結果を記録
            parameters[0] = dialogResult;
        }

        /// <summary>
        /// アセットファイルのコピーダイアログを表示します。
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        private void OnShowCopyAssetDialog(object parameter)
        {
            object[] parameters = (object[])parameter;

            // コピーダイアログを作成
            var dialog = new CopyAssetDialog(
                (string)parameters[4],
                (IEnumerable<string>)parameters[5],
                (IEnumerable<string>)parameters[6],
                (IEnumerable<string>)parameters[7],
                (IEnumerable<string>)parameters[8]);

            // コピーダイアログを表示
            DialogResult dialogResult = dialog.ShowDialog(this.mainForm);

            // 結果を記録
            parameters[0] = dialogResult;
            parameters[1] = dialog.TextureDestnationPath;
            parameters[2] = dialog.PrimitiveDestnationPath;
            parameters[3] = dialog.CombinerDestnationPath;
        }

        /// <summary>
        /// アセットファイルの参照エラーを表示します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : ユーザーの選択
        /// [1] : メッセージボックスに表示する説明
        /// [2] : メッセージボックスのキャプション
        /// </param>
        private void OnShowAssetPathProblem(object parameter)
        {
            object[] parameters = (object[])parameter;

            // メッセージボックスを表示
            DialogResult result = ThreadSafeMsgBox.Show(
                (string)parameters[1],
                (string)parameters[2],
                MessageBoxButtons.OK,
                MessageBoxIcon.Exclamation);

            // 結果を記録
            parameters[0] = result;
        }

        /// <summary>
        /// 無効な終了処理タイプの警告を表示します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : ユーザーの選択
        /// [1] : メッセージボックスに表示する説明
        /// [2] : メッセージボックスのキャプション
        /// [3] : キャンセルボタンの表示On/Off
        /// </param>
        private void OnShowInvalidEndingType(object parameter)
        {
            object[] parameters = (object[])parameter;

            bool showCancel = (bool)parameters[3];
            MessageBoxButtons buttons = showCancel ? MessageBoxButtons.YesNoCancel : MessageBoxButtons.YesNo;

            // メッセージボックスを表示
            DialogResult result = ThreadSafeMsgBox.Show(
                (string)parameters[1],
                (string)parameters[2],
                buttons,
                MessageBoxIcon.Exclamation);

            // 結果を記録
            parameters[0] = result;
        }

        /// <summary>
        /// 描画パスに不整合があることを通知するダイアログを表示します。
        /// </summary>
        /// <param name="parameter">
        /// Custom parameter.
        /// [0] : ユーザーの選択
        /// [1] : 他の全ての項目にも適用するかどうか
        /// [2] : 対象のエミッタ名
        /// [3] : キャンセルボタンの表示On/Off
        /// </param>
        private void OnShowInvalidDrawPathDialog(object parameter)
        {
            var parameters = (object[])parameter;

            // ファイルを開くダイアログを作成
            var dialog = new InvalidDrawPathDialog(
                (string)parameters[2], (bool)parameters[3]);

            // ダイアログを表示
            DialogResult result = dialog.ShowDialog(null);

            // 結果を記録
            if (result == DialogResult.Cancel)
            {
                parameters[0] = result;
                parameters[1] = false;
            }
            else
            {
                parameters[0] = result;
                parameters[1] = dialog.IsApplyToAll;
            }
        }

        #endregion

        #region Function - その他

        /// <summary>
        /// エラーダイアログを表示します
        /// </summary>
        /// <param name="parameter">
        /// [0]メッセージ本体
        /// [1]キャプション
        /// </param>
        private void OnShowErrorDialog(object parameter)
        {
            var parameters = (string[])parameter;
            ThreadSafeMsgBox.Show(
                parameters[0],
                parameters[1],
                MessageBoxButtons.OK,
                MessageBoxIcon.Error);
        }

        /// <summary>
        /// 警告ダイアログを表示します
        /// </summary>
        /// <param name="parameter">
        /// [0]メッセージ本体
        /// [1]キャプション
        /// </param>
        private void OnShowWarningDialog(object parameter)
        {
            var parameters = (string[])parameter;
            ThreadSafeMsgBox.Show(
                parameters[0],
                parameters[1],
                MessageBoxButtons.OK,
                MessageBoxIcon.Warning);
        }

        /// <summary>
        /// OkCancelダイアログを表示します
        /// </summary>
        /// <param name="parameter">
        /// [0]メッセージ本体
        /// [1]キャプション
        /// [2](出力)選択結果
        /// </param>
        private void OnShowOkCancelDialog(object parameter)
        {
            var parameters = (string[])parameter;
            var result = ThreadSafeMsgBox.Show(
                parameters[0],
                parameters[1],
                MessageBoxButtons.OKCancel,
                MessageBoxIcon.Warning);
            if (result == DialogResult.OK)
            {
                parameters[2] = "Ok";
            }
            else
            {
                parameters[2] = "Cancel";
            }
        }

        #endregion
    }
}
