﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Runtime.Remoting.Channels;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Application.Properties;
using EffectMaker.Application.TestDialog;
using EffectMaker.BusinessLogic.Commands;
using EffectMaker.BusinessLogic.Connecter;
using EffectMaker.BusinessLogic.DataModelOperation;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.ProjectConfig;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.Communicator;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Debugging.Profiling;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Debug;
using EffectMaker.UIControls.EffectBrowser.Data.Favorites;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Input;
using EffectMaker.UIControls.Layout;
using EffectMaker.UIControls.Specifics.GradationEditor;
using EffectMaker.UIControls.Specifics.TreeNodes;
using EffectMaker.UIDialogs.CurveEditorDialog;
using EffectMaker.UIDialogs.EffectBrowserDialog;
using EffectMaker.UIDialogs.EmbededViewerDialog;
using EffectMaker.UIDialogs.SearchDialog;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.Manager;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.Application
{
    /// <summary>
    /// The main window for the application.
    /// </summary>
    public partial class MainForm : Form, IConnectionStateListener, IMessageListener
    {
        /// <summary>
        /// ビューモデルのルートです。
        /// </summary>
        private WorkspaceRootViewModel rootViewModel;

        /// <summary>
        /// ドックコンテンツです。
        /// </summary>
        private DockContents dockContents;

        /// <summary>
        /// ExecutableHolderです。
        /// </summary>
        private MainFormExecutableHolder executableHolder;

        /// <summary>
        /// メニューに追加したプリセットアイテムを保持するリスト
        /// </summary>
        private List<ToolStripItem> presetMenuItems = new List<ToolStripItem>();

        /// <summary>
        /// 「編集中タブのヘルプ」のToolStripeMenuボタン
        /// </summary>
        private UIToolStripMenuItem tsmiDocumentTab;

        /// <summary>
        /// コンバイナエディタのプロセス
        /// </summary>
        private Process combinerEditorProcess;

        /// <summary>
        /// Constructor.
        /// </summary>
        public MainForm()
        {
            this.InitializeComponent();

            // ログのメニューアイテム名
            this.cmiLogSelectAll.Text = Properties.Resources.LogMenuItemNameSelectAll;
            this.cmiLogCopy.Text = Properties.Resources.LogMenuItemNameCopy;
            this.cmiLogClear.Text = Properties.Resources.LogMenuItemNameClearAll;

            this.mniDebug.Text = "デバッグ";

            // setting icon of menu [Help] - [Version information]
            this.mniHelpAbout.Image = new Icon(Properties.Resources.Icon_EffectMaker, 16, 16).ToBitmap();

            // DO NOT put anything here which involves any logic other than
            // pure UI initialization because the application initialization
            // has not yet been performed.
            // They should be placed in OnLoad method.

            // 暫定として、メインフォームのhandleを渡してしまう
            EffectMaker.DataModelLogic.Utilities.ShaderBinaryHelper.Handle = this.Handle;
        }

        /// <summary>
        /// ログビューを取得します。
        /// </summary>
        public UILogView LogView
        {
            get { return this.uiLogView1; }
        }

        /// <summary>
        /// 全て上書き保存の有効/無効を取得または設定します。
        /// </summary>
        public bool SaveAllEnabled
        {
            get
            {
                return this.btnToolbarSaveAll.Enabled;
            }

            set
            {
                this.btnToolbarSaveAll.Enabled = value;
            }
        }

        /// <summary>
        /// 接続状態が変更されたときの処理.
        /// </summary>
        /// <param name="prevState">前の状態</param>
        /// <param name="currState">現在の状態.</param>
        public void OnConnectionStateChanged(ConnectionStates prevState, ConnectionStates currState)
        {
            TheApp.SyncContext.Send(s =>
            {
                // 非接続状態から接続状態に切り替わった時.
                if (currState == ConnectionStates.Connected)
                {
                    // 再生命令を送る
                    ViewerController.Instance.SendPlay();

                    // ビューア制御メッセージを再転送.
                    OptionsForm.SendViewerConfig();
                }

                if (currState == ConnectionStates.Connected)
                {
                    // [接続]ボタンのイメージとツールチップを変更.
                    this.btnConnectViewer.Image = Properties.Resources.TB_PreviewConnect;
                    this.btnConnectViewer.ToolTipText = Properties.Resources.ToolTipTextDisconnectToViewer;

                    // [切断]メニューアイテムを有効に.
                    this.mniDisconnectViewer.Enabled = true;

                    // メニューアイテムの変更(接続->再転送)
                    this.mniConnectViewer.Image = Properties.Resources.Icon_SendViewer_ReSend;
                    this.mniConnectViewer.Text = Properties.Resources.MenuPreviewResend;
                    this.mniConnectViewer.ToolTipText = Properties.Resources.ToolTipTextResendToViewer;

                    using (new ForceRefreshBinary())
                    {
                        // Send all the emitter sets to the viewer.
                        ViewerMessageHelper.SendEmitterSets(
                            this.rootViewModel.WorkspaceViewModel.DataModel.EmitterSetList);

                        // Send all the models to the viewer.
                        ViewerMessageHelper.SendModels(
                            this.rootViewModel.WorkspaceViewModel.DataModel.ViewerData.ModelList);

                        // Send Viewer data to the viewer.
                        ViewerMessageHelper.SendViewer(
                            this.rootViewModel.WorkspaceViewModel.DataModel.ViewerData);
                    }
                }
                else
                {
                    // [接続ボタン]のイメージとツールチップを変更.
                    this.btnConnectViewer.Image = Properties.Resources.TB_PreviewDisconnect;
                    this.btnConnectViewer.ToolTipText = Properties.Resources.ToolTipTextConnectToViewer;

                    // [切断]メニューアイテムを無効に.
                    this.mniDisconnectViewer.Enabled = false;

                    // メニューアイテムの変更(接続)
                    this.mniConnectViewer.Image = Properties.Resources.TB_PreviewConnect;
                    this.mniConnectViewer.Text = Properties.Resources.MenuPreviewConnect;
                    this.mniConnectViewer.ToolTipText = Properties.Resources.ToolTipTextConnectToViewer;

                    // 全てのエミッタセットのリンクを削除
                    this.executableHolder.OnUnlinkAllEmitterSetsExecutable.Execute(null);
                }

                this.btnConnectViewer.Invalidate();
            },
            null);
        }

        /// <summary>
        /// ビューアからのメッセージ受信時の処理.
        /// </summary>
        /// <param name="msg">メッセージ</param>
        public void OnMessageReceived(EffectMaker.Communicator.Message msg)
        {
            /* ＜仮実装＞
             * あとで，コマンドマネージャを生身で使わない形で実装すること !! */

            TheApp.SyncContext.Send(s =>
            {
                RequestMessage requestMsg = new RequestMessage();

                if (!requestMsg.Deserialize(msg))
                {
                    // 失敗したら終了.
                    return;
                }

                // エディタ操作メッセージであることをチェック.
                if (requestMsg.Type == MessageType.Operation)
                {
                    switch (requestMsg.Operation)
                    {
                        // ファイルを開く.
                        case OperationType.Open:
                            {
                                string[] fileNames = new string[1] { requestMsg.EmitterSetName };
                                this.executableHolder.OnFileOpenExecutable.Execute(fileNames);
                            }

                            break;

                        // ファイルを閉じる.
                        case OperationType.Close:
                            {
                                var target = this.rootViewModel.GetEmitterSet(requestMsg.EmitterSetName);
                                if (target != null)
                                {
                                    CommandManager.Execute(new RemoveEmitterSetCommand(target));
                                }
                            }

                            break;

                        // 新規作成.
                        case OperationType.Create:
                            {
                                string inputName = requestMsg.EmitterSetName;
                                if (string.IsNullOrEmpty(inputName))
                                {
                                    inputName = DefaultNameGenerator.GenerateDefaultEmitterSetName(null);
                                }

                                string emitterName = DefaultNameGenerator.GenerateDefaultEmitterName(null);
                                string previewName = DefaultNameGenerator.GenerateDefaultPreviewName(inputName);

                                CommandManager.Execute(new CreateEmitterSetCommand(this.rootViewModel.WorkspaceViewModel, inputName, emitterName, previewName));
                            }

                            break;

                        // 複製する.
                        case OperationType.Duplicate:
                            {
                                var target = this.rootViewModel.GetEmitterSet(requestMsg.EmitterSetName);
                                if (target != null)
                                {
                                    EffectMaker.Foundation.Interfaces.IExecutable exec = target.NodeDuplicateExecutable;
                                    if (exec == null)
                                    {
                                        return;
                                    }

                                    if (exec.CanExecute(null))
                                    {
                                        exec.Execute(null);
                                    }

                                    var selectedItem = this.rootViewModel.SelectedNodeViewModel as EmitterSetViewModel;
                                    var emitterSet = selectedItem.DataModel;
                                    CommandManager.Execute(new RenameCommand(selectedItem, requestMsg.DuplicateEmitterSetName, emitterSet.Name));
                                }
                            }

                            break;

                        // リンクを作成
                        case OperationType.LinkEset:
                            {
                                string esetName = requestMsg.EmitterSetName;
                                this.executableHolder.OnLinkEmitterSetExecutable.Execute(esetName);
                            }

                            break;

                        // リンクを破棄
                        case OperationType.UnlinkEset:
                            {
                                string esetName = requestMsg.EmitterSetName;
                                this.executableHolder.OnUnlinkEmitterSetExecutable.Execute(esetName);
                            }

                            break;
                    }
                }
            },
            null);
        }

        /// <summary>
        /// ファイルを開きます。
        /// </summary>
        /// <param name="paths">ファイルリスト</param>
        public void LoadFiles(List<string> paths)
        {
            this.executableHolder.OnFileOpenExecutable.Execute(paths.ToArray());
        }

        /// <summary>
        /// The show curve editor.
        /// </summary>
        public void ShowCurveEditor()
        {
            this.dockContents.ShowCurveEditorDialog();
        }

        /// <summary>
        /// The connect curve editor.
        /// </summary>
        /// <param name="dataContext">
        /// The data context.
        /// </param>
        public void ConnectCurveEditor(object dataContext)
        {
            CurveEditorDialog curveEditor = this.dockContents.GetCurveEditorDialog(true);
            curveEditor.AddDataContext(dataContext);
        }

        /// <summary>
        /// The disconnect curve editor.
        /// </summary>
        /// <param name="dataContext">
        /// The data context.
        /// </param>
        public void DisconnectCurveEditor(object dataContext)
        {
            CurveEditorDialog curveEditor = this.dockContents.GetCurveEditorDialog(true);
            curveEditor.RemoveDataContext(dataContext);
        }

        /// <summary>
        /// The notify curve editor.
        /// </summary>
        /// <param name="dataContext">
        /// The data context.
        /// </param>
        public void NotifyEmitterToCurveEditor(object dataContext)
        {
            CurveEditorDialog curveEditor = this.dockContents.GetCurveEditorDialog(true);
            curveEditor.NotifyEmitter(dataContext);
        }

        /// <summary>
        /// The notify changing tab page to curve editor.
        /// </summary>
        /// <param name="parameter">
        /// The dummy parameter.
        /// </param>
        public void NotifyTabPageToCurveEditor(object parameter)
        {
            CurveEditorDialog curveEditor = this.dockContents.GetCurveEditorDialog(true);
            curveEditor.NotifyTabPage(parameter);
        }

        /// <summary>
        /// Called when the form is completely loaded.
        /// </summary>
        /// <param name="e">The load event argument.</param>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            Logger.Log(LogLevels.Debug, "Beginning MainForm.Onload...");
            var timer = new ProfileTimer("MainForm.OnLoad");

            using (new ProfileTimer("Initialize view models"))
            {
                // ビューモデルを取得
                this.rootViewModel = TheApp.RootViewModel;

                // ExecutableHolderを作成
                this.executableHolder = new MainFormExecutableHolder(this.rootViewModel, this);
            }

            using (new ProfileTimer("Initialize property dialog"))
            {
                // ドックコンテンツを初期化
                this.dockContents = new DockContents(
                    this,
                    this.dcpPropertyDialogContainer,
                    this.rootViewModel,
                    this.executableHolder);

                this.dockContents.LoadDockInfo();

                // プロパティダイアログを表示
                this.dockContents.ShowPropertyDialog();
            }

            // テクスチャダイアログ
            using (new ProfileTimer("Initialize texture viewer dialog"))
            {
                // 自動作成フラグを使ってテクスチャダイアログを作成
                this.dockContents.GetTextureViewerDialog(true);
            }

            using (new ProfileTimer("Initialize menu commands"))
            {
                // メニューコマンド初期化
                this.InitializeMenuCommand();
            }

            using (new ProfileTimer("Initialize tool bar help buttons"))
            {
                // ツールバーのヘルプボタンの初期化
                this.InitializeToolBarHelp();
            }

            using (new ProfileTimer("Initialize panels layout items"))
            {
                // パネルレイアウトの初期化
                this.InitializePanesLayoutItems();
            }

            using (new ProfileTimer("Initialize viewer connect targets"))
            {
                // スペックターゲットの初期化.
                this.InitializeSpecTarget();
            }

            // メニューにバインドを追加
            using (new ProfileTimer("Add binding to menu items"))
            {
                this.mniFileOpen.AddBinding("Enabled", "CanLoadEmitterSet");
                this.mniFileOpen.DataContext = this.rootViewModel;

                this.mniFileNewEmitterSet.AddBinding("Enabled", "CanCreateNewEmitterSet");
                this.mniFileNewEmitterSet.DataContext = this.rootViewModel;

                this.mniFileNewEmitter.AddBinding("Enabled", "CanCreateNewEmitter");
                this.mniFileNewEmitter.DataContext = this.rootViewModel;

                this.mniFileNewPreview.AddBinding("Enabled", "CanCreateNewPreview");
                this.mniFileNewPreview.DataContext = this.rootViewModel;

                this.mniFileClose.AddBinding("Enabled", "CanCloseSelectedNode");
                this.mniFileClose.DataContext = this.rootViewModel;

                this.mniFileCloseAll.AddBinding("Enabled", "CanCloseAll");
                this.mniFileCloseAll.DataContext = this.rootViewModel;

                this.mniFileSave.AddBinding("Enabled", "CanSaveSelectedNode");
                this.mniFileSave.DataContext = this.rootViewModel;

                this.mniFileSaveAs.AddBinding("Enabled", "CanSaveAsSelectedNode");
                this.mniFileSaveAs.DataContext = this.rootViewModel;

                this.mniFileSaveAll.AddBinding("Enabled", "CanSaveAnyNode");
                this.mniFileSaveAll.DataContext = this.rootViewModel;

                this.mniFileExportAll.AddBinding("Enabled", "CanRemoveAnyNode");
                this.mniFileExportAll.DataContext = this.rootViewModel;

                this.mniFileRename.AddBinding("Enabled", "CanRenameSelectedNode");
                this.mniFileRename.DataContext = this.rootViewModel;

                this.mniDuplicate.AddBinding("Enabled", "CanDuplicateSelectedNode");
                this.mniDuplicate.DataContext = this.rootViewModel;

                this.mniCopy.AddBinding("Enabled", "CanCopySelectedNode");
                this.mniCopy.DataContext = this.rootViewModel;

                this.mniPaste.AddBinding("Enabled", "CanPasteNodeOrPasteEmitter");
                this.mniPaste.DataContext = this.rootViewModel;

                this.mniDelete.AddBinding("Enabled", "CanDeleteSelectedNode");
                this.mniDelete.DataContext = this.rootViewModel;

                this.mniEnableConvert.AddBinding("Enabled", "CanSwitchConvertFlag");
                this.mniEnableConvert.DataContext = this.rootViewModel;

                // 貼り付けメニューに関する追加処理
                this.mniEdit.DropDownOpening += (ss, ee) =>
                {
                    // Enabledへのバインダーをアップデート
                    // (コピー後に無効な文字列がクリップされた時対策)
                    var binder = this.mniPaste.Bindings.FirstOrDefault(
                        x => x.ElementPropertyName == "Enabled");
                    if (binder != null)
                    {
                        binder.UpdateElement();
                    }

                    // 「エミッタを貼り付け」と「内容を貼り付け」を切り替え
                    if (this.rootViewModel.SelectedNodeViewModel is EmitterSetViewModel)
                    {
                        this.mniPaste.Text = Properties.Resources.MenuEditPasteEmitter;
                    }
                    else
                    {
                        this.mniPaste.Text = Properties.Resources.MenuEditPaste;
                    }
                };
            }

            using (new ProfileTimer("Assign event handlers"))
            {
                this.btnToolbarExportAll.Click += this.OnFileSaveAllAs;

                this.btnToolbarLoadWorkspace.Click += this.OnWorkspaceOpen;
                this.btnToolbarSaveWorkspace.Click += this.OnWorkspaceSave;
                this.btnToolbarSaveWorkspaceAs.Click += this.OnWorkspaceSaveAs;

                this.btnToolbarViewProperty.Click += this.OnShowPropertyWindow;
                this.btnToolbarEffectBrowser.Click += this.OnShowEffectBrowser;
                this.btnToolbarCurveWindow.Click += this.OnShowCurveWindow;
                this.btnToolbarViewHistory.Click += this.OnShowHistory;

                this.btnToolbarCaptureThumbnail.Click += this.OnCaptureThumbnail;

                this.mniFileExit.Click += this.FileExit_Click;

                this.btnToolbarSaveAll.Enabled = false;
                this.btnToolbarExportAll.Enabled = false;

                this.btnToolbarExportAll.AddBinding("Enabled", "CanRemoveAnyNode");
                this.btnToolbarExportAll.DataContext = this.rootViewModel;

                this.btnToolbarCombinerEditor.Click += this.OnExecuteCombinerEditor;

                this.mniViewToolBar.Checked = true;
                this.mniViewToolBar.Click += (ss, ee) =>
                {
                    this.mniViewToolBar.Checked = !this.mniViewToolBar.Checked;
                    this.tspMainToolBar.Visible = this.mniViewToolBar.Checked;
                };

                PropertyDialog propertyDialog = this.dockContents.GetPropertyDialog(true);
                propertyDialog.PropertyPanel.SelectedIndexChanged += (ss, ee) => this.UpdateToolBarHelp();
                propertyDialog.OnWorkspaceSelectedNodeChangedAdditionalAction = this.UpdateToolBarHelp;
            }

            using (new ProfileTimer("Adjust size and location for MainForm"))
            {
                Rectangle bounds = OptionStore.RootOptions.Interface.MainFormBounds;
                if (bounds.IsEmpty == true)
                {
                    bounds = this.Bounds;
                }

                this.Bounds = this.AdjustBoundsToVisibleScreenArea(bounds);

                // Handle size and location change event and save them to the config.
                this.SizeChanged += (s, args) =>
                {
                    OptionStore.RootOptions.Interface.MainFormBounds = this.Bounds;
                };

                this.LocationChanged += (s, args) =>
                {
                    OptionStore.RootOptions.Interface.MainFormBounds = this.Bounds;
                };
            }

            using (new ProfileTimer("Set up tool bar"))
            {
                // 接続ボタンのイメージとツールチップをセット.
                this.btnConnectViewer.Image = Properties.Resources.TB_PreviewDisconnect;
                this.btnConnectViewer.ToolTipText = Properties.Resources.ToolTipTextConnectToViewer;

                this.mniConnectViewer.Image = Properties.Resources.TB_PreviewConnect;
                this.mniConnectViewer.Text = Properties.Resources.MenuPreviewConnect;
                this.mniConnectViewer.ToolTipText = Properties.Resources.ToolTipTextConnectToViewer;

                // 接続先をオプションから設定
                this.cbViewerConnection.SelectedIndex = Math.Min(
                    this.cbViewerConnection.Items.Count - 1,
                    (int)OptionStore.RootOptions.Viewer.ConnectionTarget);

                // ログビューにアイコン(イメージリスト)をセット.
                this.uiLogView1.ImageList = this.imageListLogView;
                this.uiLogView1.ContextMenuStrip = this.ctxMenuLog;

                // タイトルウィンドウのタイトルをセット
                this.UpdateTitleBarInternal();

                // メニューバーの情報をセット
                this.UpdateMenuItemVisible();

                // コマンドボタンを無効にし、イベントを登録する.
                this.btnToolbarUndo.Enabled = false;
                this.btnToolbarRedo.Enabled = false;
                this.mniEditUndo.Enabled = false;
                this.mniEditRedo.Enabled = false;
            }

            using (new ProfileTimer("Miscellaneous initializations"))
            {
                CommandManager.CommandExecuted += this.CommandManagerCommandChanged;
                CommandManager.ActiveTargetChanged += this.CommandManagerActiveTargetChanged;

                // リニアモード対応
                OptionStore.OptionChanged += (ss, ee) => ColorUtility.IsGammaCorrectionEnabled = OptionStore.ProjectConfig.LinearMode;
                ColorUtility.Gamma = 2.2;

                // オプションを適応させる
                OptionStore.TriggerOptionChangedEvent(null);

                // オプション変更時にタイトルバーを更新する
                OptionStore.OptionChanged += (ss, ee) => this.UpdateTitleBarInternal();

                // オプション変更時にメニューバーの情報を更新する
                OptionStore.OptionChanged += (ss, ee) => this.UpdateMenuItemVisible();

                // サブウィンドウのショートカットを設定
                this.SetShortcutBipass();

                // テキストボックス内で本来のショートカットキーを有効にするための処理
                UIUserControl.EnableMenuShortcut = () =>
                {
                    // コピペ関連ステートの復帰
                    WorkspaceRootViewModel.Instance.BlockingUpdateCopyPasteState = false;
                    WorkspaceRootViewModel.Instance.RaiseCopyPastePropertyChanged();
                    this.mniCopy.ShortcutKeys = Keys.Control | Keys.C;
                    this.mniPaste.ShortcutKeys = Keys.Control | Keys.V;
                };
                UIUserControl.DisableMenuShortcut = () =>
                {
                    // コピペ関連ステートを一時無効化
                    WorkspaceRootViewModel.Instance.BlockingUpdateCopyPasteState = true;
                    this.mniCopy.ShortcutKeys = Keys.None;
                    this.mniCopy.ShortcutKeyDisplayString = "Ctrl+C";
                    this.mniPaste.ShortcutKeys = Keys.None;
                    this.mniPaste.ShortcutKeyDisplayString = "Ctrl+V";
                };

                // テクスチャのキャッシュサイズを設定する
                EffectMaker.BusinessLogic.Manager.TextureManager.Instance.SetMaxCapacity(
                    OptionStore.RootOptions.Details.MaximumTextureCacheSize * 1024 * 1024);

                // オプションで設定したワークスペースを、起動時に読み込ませる
                if (File.Exists(OptionStore.RootOptions.Basic.WorkspaceFilesAutoLoad))
                {
                    this.rootViewModel.OnWorkspaceOpenCore(OptionStore.RootOptions.Basic.WorkspaceFilesAutoLoad);
                }

                // メッセージマネージャのスレッドを起動する.
                MessageManager.Instance.StartThread();
            }

            timer.Stop();
        }

        /// <summary>
        /// ツールバーにあるToolStripeSplitButtonヘルプの状態を更新する.
        /// </summary>
        private void UpdateToolBarHelp()
        {
            PropertyDialog dialog = this.dockContents.GetPropertyDialog(true);
            IDocumentLinker tab = dialog.GetCurrentPageLink();
            if (tab != null && tab.DocumentId.IsNullOrEmpty() == false)
            {
                // 選択されているタブに、ドキュメントのリンクがない場合は、tsmiDocumentTabをdisableにする
                tsmiDocumentTab.Enabled = true;
                if (this.uiToolStripSplitButtonToolBarHelp.DefaultItem == tsmiDocumentTab)
                {
                    this.uiToolStripSplitButtonToolBarHelp.DrawEnabledButton = true;
                    this.uiToolStripSplitButtonToolBarHelp.EnableButtons = true;
                }
            }
            else
            {
                // 選択されているタブにドキュメントのリンクが有る場合は、tsmiDocumentTabでヘルプを開けるようにする
                tsmiDocumentTab.Enabled = false;
                if (this.uiToolStripSplitButtonToolBarHelp.DefaultItem == tsmiDocumentTab)
                {
                    this.uiToolStripSplitButtonToolBarHelp.DrawEnabledButton = false;
                    this.uiToolStripSplitButtonToolBarHelp.EnableButtons = true;
                }
            }
            this.uiToolStripSplitButtonToolBarHelp.Invalidate();

        }

        /// <summary>
        /// 終了コマンドがクリックされた.
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event.</param>
        private void FileExit_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        /// <summary>
        /// ショートカットのバイパスをまとめて設定
        /// </summary>
        private void SetShortcutBipass()
        {
            this.PushShortcut(this.mniFileOpen);
            this.PushShortcut(this.mniFileClose);
            this.PushShortcut(this.mniFileCloseAll);

            this.PushShortcut(this.mniFileSave);
            this.PushShortcut(this.mniFileSaveAs);
            this.PushShortcut(this.mniFileSaveAll);

            this.PushShortcut(this.mniFileExit);

            this.PushShortcut(this.mniConnectViewer);

            this.PushShortcut(this.mniViewProperty);
            this.PushShortcut(this.mniEffectBrowser);
            this.PushShortcut(this.mniCurveWindow);
            this.PushShortcut(this.mniViewHistory);

            this.PushShortcut(this.mniEditUndo);
            this.PushShortcut(this.mniEditRedo);
        }

        /// <summary>
        /// ショートカットのバイパスを設定します。
        /// </summary>
        /// <param name="menu">対応するメニュー</param>
        private void PushShortcut(UIToolStripMenuItem menu)
        {
            GlobalKeyEventHandler.Instance.PushShortcutAction(
                menu.ShortcutKeys,
                () =>
                {
                    if (menu.Enabled)
                    {
                        menu.PerformClick();
                    }
                });
        }

        /// <summary>
        /// コマンドのアクティブターゲットが変更された時に呼び出される.
        /// </summary>
        /// <param name="e">event.</param>
        void CommandManagerActiveTargetChanged(CommandManager.CommandActiveTargetEventArgs e)
        {
            CommandStack cmdStack = CommandManager.GetActiveStack();
            if (cmdStack == null)
            {
                this.btnToolbarUndo.Enabled = this.mniEditUndo.Enabled = false;
                this.btnToolbarRedo.Enabled = this.mniEditRedo.Enabled = false;
            }
            else if (cmdStack != null)
            {
                this.btnToolbarUndo.Enabled = this.mniEditUndo.Enabled =
                    cmdStack.UndoBuffer.FirstOrDefault() != null;
                this.btnToolbarRedo.Enabled = this.mniEditRedo.Enabled =
                    cmdStack.RedoBuffer.FirstOrDefault() != null;
            }
        }

        /// <summary>
        /// コマンドが実行された後に呼び出される.
        /// </summary>
        /// <param name="e">event.</param>
        void CommandManagerCommandChanged(CommandManager.CommandEventArgs e)
        {
            CommandStack cmdStack = CommandManager.GetActiveStack();
            if (cmdStack != null)
            {
                this.btnToolbarUndo.Enabled = this.mniEditUndo.Enabled =
                    cmdStack.UndoBuffer.FirstOrDefault() != null;
                this.btnToolbarRedo.Enabled = this.mniEditRedo.Enabled =
                    cmdStack.RedoBuffer.FirstOrDefault() != null;
            }
        }

        /// <summary>
        /// タイトルバーの文字列を変更します。
        /// </summary>
        private void UpdateTitleBarInternal()
        {
            System.Reflection.Assembly assembly = System.Reflection.Assembly.GetExecutingAssembly();
            FileVersionInfo info = FileVersionInfo.GetVersionInfo(assembly.Location);

            // タイトルの表示を変更します。
            string title = string.Format(
                "{0} {1} {2}",
                ////  ツール名表示。
                System.Windows.Forms.Application.ProductName,
                ////  バージョン番号。
                info.ProductMajorPart + "." + info.ProductMinorPart + "." + info.ProductBuildPart,
                //// リニア編集モード
                OptionStore.ProjectConfig.LinearMode ? Properties.Resources.LinearEditModeOn : string.Empty);

            this.Text = title;
        }

        /// <summary>
        /// メニューバーアイテムの表示状態を変更します。
        /// </summary>
        private void UpdateMenuItemVisible()
        {
            // コンバイナエディタの表示状態の変更
            if (this.btnToolbarCombinerEditor.Visible != OptionStore.ProjectConfig.IsEftCombinerEditorEnabled)
            {
                this.btnToolbarCombinerEditor.Visible = OptionStore.ProjectConfig.IsEftCombinerEditorEnabled;
                this.mniCombinerEditor.Visible = OptionStore.ProjectConfig.IsEftCombinerEditorEnabled;

                // ショートカットキー無効化のため、Enabled状態も変更する。
                this.btnToolbarCombinerEditor.Enabled = OptionStore.ProjectConfig.IsEftCombinerEditorEnabled;
                this.mniCombinerEditor.Enabled = OptionStore.ProjectConfig.IsEftCombinerEditorEnabled;
            }
        }

        #region メニューコマンド

        /// <summary>
        /// メニューコマンド初期化。
        /// </summary>
        private void InitializeMenuCommand()
        {
#if DEBUG

            // デバックメニューを表示にします。
            this.mniDebug.Enabled = this.mniDebug.Visible = true;

#endif // DEBUG

            // メインメニュー
            this.mniFile.Text = Properties.Resources.MainMenuFile;
            this.mniEdit.Text = Properties.Resources.MainMenuEdit;
            this.mniView.Text = Properties.Resources.MainMenuView;
            this.mniPreview.Text = Properties.Resources.MainMenuPreview;
            this.mniTool.Text = Properties.Resources.MainMenuTool;
            this.mniHelp.Text = Properties.Resources.MainMenuHelp;
            this.mniHelp.DropDownOpening += (sender, args) =>
            {
                this.mniHelpHint.Enabled = !TipsDialog.Showing;
                this.mniHelpSampleFolder.Enabled = Directory.Exists(FavoritesManager.SampleFolderPath);
            };

            // ファイル関連
            this.mniFileNew.Text = Properties.Resources.MenuFileNew;
            this.mniFileNewEmitterSet.Text = Properties.Resources.MenuFileNewEmitterSet;
            this.mniFileNewEmitter.Text = Properties.Resources.MenuFileNewEmitter;
            this.mniFileNewPreview.Text = Properties.Resources.MenuFileNewPreview;
            this.mniFileOpen.Text = Properties.Resources.MenuFileOpen;
            this.mniFileRecentFilesRoot.Text = Properties.Resources.MenuFileRecentFiles;
            this.mniFileClose.Text = Properties.Resources.MenuFileClose;
            this.mniFileCloseAll.Text = Properties.Resources.MenuFileCloseAll;
            this.mniFileSave.Text = Properties.Resources.MenuFileSave;
            this.mniFileSaveAs.Text = Properties.Resources.MenuFileSaveAs;
            this.mniFileSaveAll.Text = Properties.Resources.MenuFileSaveAll;
            this.mniFileExportAll.Text = Properties.Resources.MenuFileExportAll;
            this.mniFileLoadWorkspace.Text = Properties.Resources.MenuFileLoadWorkspace;
            this.mniFileSaveWorkspace.Text = Properties.Resources.MenuFileSaveWorkspace;
            this.mniFileSaveWorkspaceAs.Text = Properties.Resources.MenuFileSaveWorkspaceAs;
            this.mniFileRename.Text = Properties.Resources.MenuFileRename;
            this.mniFileExit.Text = Properties.Resources.MenuFileExit;

            // 編集関連
            this.mniEditUndo.Text = Properties.Resources.MenuEditUndo;
            this.mniEditRedo.Text = Properties.Resources.MenuEditRedo;
            this.mniCopy.Text = Properties.Resources.MenuEditCopy;
            this.mniPaste.Text = Properties.Resources.MenuEditPaste;
            this.mniDuplicate.Text = Properties.Resources.MenuEditDuplicate;
            this.mniDelete.Text = Properties.Resources.MenuEditDelete;
            this.mniEnableConvert.Text = Properties.Resources.MenuEditEnableConvert;

            // 階層化の解除をショートカットを効かせるために隠しメニューとして追加
            var mniUnparentEmitter = new UIToolStripMenuItem();
            this.mniEdit.Children.Add(mniUnparentEmitter);
            mniUnparentEmitter.Visible = false;
            mniUnparentEmitter.ShortcutKeys = Keys.Control | Keys.Shift | Keys.P;
            mniUnparentEmitter.Click += (s, e) =>
            {
                WorkspaceRootViewModel.Instance.WorkspaceViewModel.OnUnparentEmitter();
            };

            // 表示関連
            this.mniViewProperty.Text = Properties.Resources.MenuViewProperty;
            this.mniEffectBrowser.Text = Properties.Resources.MenuViewEffectBrowser;
            this.mniCurveWindow.Text = Properties.Resources.MenuViewCurveEditor;
            this.mniViewHistory.Text = Properties.Resources.MenuViewHistory;
            this.mniViewSearchAndBatchEdit.Text = Properties.Resources.MenuViewSearchAndBatchEdit;
            this.mniViewToolBar.Text = Properties.Resources.MenuViewToolBar;

            // 接続、切断アイテム
            this.mniConnectViewer.Image = Properties.Resources.TB_PreviewConnect;
            this.mniConnectViewer.Text = Properties.Resources.MenuPreviewConnect;
            this.mniConnectViewer.ToolTipText = Properties.Resources.ToolTipTextConnectToViewer;

            this.mniDisconnectViewer.Image = Properties.Resources.TB_PreviewDisconnect;
            this.mniDisconnectViewer.Text = Properties.Resources.MenuPreviewDisconnect;
            this.mniDisconnectViewer.ToolTipText = Properties.Resources.ToolTipTextDisconnectToViewer;
            this.mniDisconnectViewer.Enabled = false;

            // ビューア起動メニューの設定
            this.mniLaunchPCViewer.Image = Properties.Resources.Icon_StartViewer;
            this.mniLaunchPCViewer.Text = Properties.Resources.MenuPreviewLaunchPCViewer;
            this.mniLaunchPCViewer.Click += (sender, args) =>
            {
                Process viewerProcess = ViewerRunner.LaunchPCViewer();

                if (viewerProcess != null)
                {
                    EmbededViewerDialog viewerDialog = this.dockContents.ShowEmbededViewerDialog();
                    viewerDialog.ViewerProcess = viewerProcess;
                }
            };
            this.mniLaunchTargetViewer.Image = Properties.Resources.Icon_StartViewer;
            this.mniLaunchTargetViewer.Text = Properties.Resources.MenuPreviewLaunchTargetViewer;
            this.mniLaunchTargetViewer.Click += this.OnLaunchTargetViewer;
            Action updateLaunchMenuAction = () =>
            {
                this.mniLaunchPCViewer.Enabled = !string.IsNullOrEmpty(SpecManager.CurrentSpec.PCViewerPath);
                this.mniLaunchTargetViewer.Enabled = !string.IsNullOrEmpty(SpecManager.CurrentSpec.TargetViewerPath);
            };
            SpecManager.CurrentSpecChanged += (sender, args) => updateLaunchMenuAction();
            updateLaunchMenuAction();

            // ツールバーのビューア起動ボタンの設定
            this.btnToolBarLaunchViewer.Click += (sender, args) =>
            {
                int index = this.cbViewerConnection.SelectedIndex;
                if (index == 0 && !string.IsNullOrEmpty(SpecManager.CurrentSpec.PCViewerPath))
                {
                    Process viewerProcess = ViewerRunner.LaunchPCViewer();

                    if (viewerProcess != null)
                    {
                        EmbededViewerDialog viewerDialog = this.dockContents.ShowEmbededViewerDialog();
                        viewerDialog.ViewerProcess = viewerProcess;
                    }
                }
                else if (index == 1 && !string.IsNullOrEmpty(SpecManager.CurrentSpec.TargetViewerPath))
                {
                    this.OnLaunchTargetViewer(sender, args);
                }
            };
            Action updateLaunchButtonAction = () =>
            {
                int index = this.cbViewerConnection.SelectedIndex;
                if (index == 0 && !string.IsNullOrEmpty(SpecManager.CurrentSpec.PCViewerPath))
                {
                    this.btnToolBarLaunchViewer.Enabled = true;
                    this.btnToolBarLaunchViewer.ToolTipText = Properties.Resources.MenuPreviewLaunchPCViewer;
                }
                else if (index == 1 && !string.IsNullOrEmpty(SpecManager.CurrentSpec.TargetViewerPath))
                {
                    this.btnToolBarLaunchViewer.Enabled = true;
                    this.btnToolBarLaunchViewer.ToolTipText = Properties.Resources.MenuPreviewLaunchTargetViewer;
                }
                else
                {
                    this.btnToolBarLaunchViewer.Enabled = false;
                    this.btnToolBarLaunchViewer.ToolTipText = Properties.Resources.ToolTipNotAvailableViewer;
                }
            };
            SpecManager.CurrentSpecChanged += (sender, args) => updateLaunchButtonAction();
            this.cbViewerConnection.SelectedIndexChanged += (sender, args) => updateLaunchButtonAction();
            updateLaunchButtonAction();

            this.mniCaptureThumbnail.Text = Properties.Resources.MenuCaptureThumbnail;

            // ツール関連
            this.mniToolOption.Text = Properties.Resources.MenuToolOption;

            // ヘルプ関連
            this.mniHelpHelp.Text = Properties.Resources.MenuHelpHelp;
            this.mniHelpNews.Text = Properties.Resources.MenuHelpNews;
            this.mniHelpHint.Text = Properties.Resources.MenuHelpHint;
            this.mniHelpTroubleshooting.Text = Properties.Resources.MenuHelpTroubleshooting;
            this.mniHelpAbout.Text = Properties.Resources.MenuHelpAbout;
            this.mniCombinerEditor.Text = Properties.Resources.MenuHelpCombinerEditor;
            this.mniHelpSampleFolder.Text = Properties.Resources.MenuHelpSampleFolder;
            this.mniHelpExecutableFolder.Text = Properties.Resources.MenuHelpExecutableFolder;
            this.mniHelpAutoBackupFolder.Text = Properties.Resources.MenuHelpAutoBackupFolder;

            // ツールチップ関連
            this.btnToolbarLoadFile.ToolTipText = Properties.Resources.ToolTipTextLoadFile;
            this.btnToolbarSaveAll.ToolTipText = Properties.Resources.ToolTipTextSaveAll;
            this.btnToolbarExportAll.ToolTipText = Properties.Resources.ToolTipTextExportAll;
            this.btnToolbarUndo.ToolTipText = Properties.Resources.ToolTipTextUndo;
            this.btnToolbarRedo.ToolTipText = Properties.Resources.ToolTipTextRedo;
            this.btnToolbarCombinerEditor.ToolTipText = Properties.Resources.ToolTipTextCombinerEditor;
            this.btnToolbarViewProperty.ToolTipText = Properties.Resources.ToolTipTextViewProperty;
            this.btnToolbarEffectBrowser.ToolTipText = Properties.Resources.ToolTipTextEffectBrowser;
            this.btnToolbarCurveWindow.ToolTipText = Properties.Resources.ToolTipTextCurveEditor;
            this.btnToolbarViewHistory.ToolTipText = Properties.Resources.ToolTipTextViewHistory;
            this.btnToolbarCaptureThumbnail.ToolTipText = Properties.Resources.ToolTipTextCaptureThumbnail;
            this.btnToolbarLoadWorkspace.ToolTipText = Properties.Resources.ToolTipTextLoadWorkspace;
            this.btnToolbarSaveWorkspace.ToolTipText = Properties.Resources.ToolTipTextSaveWorkspace;
            this.btnToolbarSaveWorkspaceAs.ToolTipText = Properties.Resources.ToolTipTextSaveWorkspaceAs;
            this.lblPlatform.ToolTipText = Properties.Resources.ToolTipTextPlatformLabel;

            this.lblPlatform.Text = Properties.Resources.LabelPlatform;
        }

        #endregion

        /// <summary>
        /// Handle SelectedIndexChanged event for the viewer connection target combo box.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnConnectionTargetSelectedIndexChanged(object sender, EventArgs e)
        {
            this.dummyFocusButton.Select();

            var workspace = this.rootViewModel.WorkspaceViewModel.DataModel;

            // 切断処理.
            ViewerMessageHelper.RequestDeleteEmitterSets(workspace.EmitterSetList);
            ViewerMessageHelper.RequestDeleteModels(workspace.ViewerData.ModelList);

            MessageManager.Instance.Disconnect();

            uint id = (uint)this.cbViewerConnection.SelectedIndex;
            switch (id)
            {
                // PCビューア
                case 0:
                    this.cbViewerConnection.ToolTipText = Properties.Resources.ToolTipTextConnectionTargetPC;
                    break;

                // 実機ビューア.
                case 1:
                    this.cbViewerConnection.ToolTipText = Properties.Resources.ToolTipTextConnectionTargetConsole;
                    break;
            }

            OptionStore.RootOptions.Viewer.ConnectionTarget = id;
            BinaryConversionUtility.ForProtocol.IsLittleEndian =
                !SpecManager.CurrentSpec.Connections[(int)id].IsBigEndian;

            MessageManager.Instance.SetActiveConnecterById(id);
            if (MessageManager.Instance.IsAutoConnectEnabled == true)
            {
                Task.Run(() => MessageManager.Instance.Connect());
            }
        }


        /// <summary>
        ///
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnSpecTargetSelectedIndexChanged(object sender, EventArgs e)
        {
            this.dummyFocusButton.Select();

            if (ViewerController.Instance.IsConnected)
            {
                var workspace = this.rootViewModel.WorkspaceViewModel.DataModel;

                // 切断処理.
                ViewerMessageHelper.RequestDeleteEmitterSets(workspace.EmitterSetList);
                ViewerMessageHelper.RequestDeleteModels(workspace.ViewerData.ModelList);

                MessageManager.Instance.Disconnect();
            }

            // スペックを選択するComboBoxのツールチップも、変更しておく
            this.cbSpecSelection.ToolTipText = Properties.Resources.ToolTipTextSpecComboBox +
                SpecManager.SpecDefinitions[this.cbSpecSelection.SelectedIndex].Name;

            using (new ForceRefreshBinary())
            {
                SpecManager.SelectedIndex = this.cbSpecSelection.SelectedIndex;
                this.InitializeConnectionTarget();

                if (ViewerController.Instance.IsConnected)
                {
                    using (new ProfileTimer("Send binary data"))
                    {
                        // 再生を再開
                        ViewerController.Instance.SendPlay();

                        // Send Viewer option to the viewer.
                        OptionsForm.SendViewerConfig();

                        // Send all the emitter sets to the viewer.
                        ViewerMessageHelper.SendEmitterSets(
                            this.rootViewModel.WorkspaceViewModel.DataModel.EmitterSetList);

                        // Send all the models to the viewer.
                        ViewerMessageHelper.SendModels(
                            this.rootViewModel.WorkspaceViewModel.DataModel.ViewerData.ModelList);

                        // Send Viewer data to the viewer.
                        ViewerMessageHelper.SendViewer(
                            this.rootViewModel.WorkspaceViewModel.DataModel.ViewerData);
                    }
                }
            }

            OptionStore.RootOptions.Viewer.SpecTarget = SpecManager.CurrentSpec.Name;
        }

        /// <summary>
        /// パネルレイアウトのコンボボックスアイテムを初期化します.
        /// </summary>
        private void InitializePanesLayoutItems()
        {
            this.cbPanesLayout.Items.Add(Properties.Resources.PanelLayoutSingle);
            this.cbPanesLayout.Items.Add(Properties.Resources.PanelLayoutSideBySide);
            this.cbPanesLayout.SelectedIndex = 1;
            this.cbPanesLayout.ToolTipText = Properties.Resources.ToolTipTextLayoutSideBySide;
        }

        /// <summary>
        /// 接続ターゲットの初期化を行います.
        /// </summary>
        /// <returns>初期化に成功したらtrueを返却します.</returns>
        private bool InitializeSpecTarget()
        {
            // 状態通知インタフェースを登録.
            ViewerController.Instance.AddStateNotifier(0, this);

            // メッセージ通知インタフェースを登録.
            MessageManager.Instance.AddMessageNotifier(this);

            foreach (var spec in SpecManager.SpecDefinitions)
            {
                this.cbSpecSelection.Items.Add(spec.Name);
            }

            // 文字列で保存されていたスペックを復元（見つからなかったら先頭のもの）
            var selectedSpec = SpecManager.SpecDefinitions.FirstOrDefault(s => s.Name == OptionStore.RootOptions.Viewer.SpecTarget);
            this.cbSpecSelection.SelectedIndex = selectedSpec == null ? 0 : SpecManager.SpecDefinitions.IndexOf(selectedSpec);

            // ComboBoxのツールチップもセットしておく
            this.cbSpecSelection.ToolTipText = Properties.Resources.ToolTipTextSpecComboBox +
                SpecManager.SpecDefinitions[this.cbSpecSelection.SelectedIndex].Name;

            return true;
        }

        /// <summary>
        /// 接続ターゲットの初期化を行います.
        /// </summary>
        /// <returns>初期化に成功したらtrueを返却します.</returns>
        private bool InitializeConnectionTarget()
        {
            MessageManager.Instance.ClearConnecters();
            this.cbViewerConnection.Items.Clear();

            // 接続先を登録
            uint index = 0;
            foreach (var connection in SpecManager.CurrentSpec.Connections)
            {
                MessageManager.Instance.AddConnecter(index, new TcpConnecter(connection));
                this.cbViewerConnection.Items.Add(index == 0 ? Resources.ConnectToPC : Resources.ConnectToConsole);

                ++index;
            }

            // アクティブコネクターを設定.
            MessageManager.Instance.SetActiveConnecterById(0);

            // コンボボックスの選択項目を設定.
            this.cbViewerConnection.ToolTipText = Properties.Resources.ToolTipTextConnectionTargetPC;
            this.cbViewerConnection.SelectedIndex = Math.Min(
                this.cbViewerConnection.Items.Count - 1, (int)OptionStore.RootOptions.Viewer.ConnectionTarget);

            // 初回接続処理.
            if (MessageManager.Instance.IsAutoConnectEnabled)
            {
                Task.Run(() => MessageManager.Instance.Connect());
            }

            // 正常終了.
            return true;
        }

        /// <summary>
        /// ツールバーにあるヘルプボタンの初期化を行います。
        /// </summary>
        private void InitializeToolBarHelp()
        {
            var toolBarItems = new List<ToolStripItem>();

            var tsmiDocumentMainWin = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiDocumentMainWin);
            tsmiDocumentMainWin.Text = Properties.Resources.MenuItemNameDocumentMainWin;
            tsmiDocumentMainWin.Image = Properties.Resources.Icon_Document_MainWin;
            tsmiDocumentMainWin.ImageTransparentColor = Color.White;
            tsmiDocumentMainWin.Click += (s, e) =>
            {
                this.uiToolStripSplitButtonToolBarHelp.DrawEnabledButton = true;
                this.uiToolStripSplitButtonToolBarHelp.DefaultItem = tsmiDocumentMainWin;
                this.uiToolStripSplitButtonToolBarHelp.Image = Properties.Resources.Icon_Document_MainWin;
                HelpManager.ShowHelp("83956054", "");
            };

            var tsmiDocumentPropertyWin = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiDocumentPropertyWin);
            tsmiDocumentPropertyWin.Text = Properties.Resources.MenuItemNameDocumentPropertyWin;
            tsmiDocumentPropertyWin.Image = Properties.Resources.Icon_Document_PropertyWin;
            tsmiDocumentPropertyWin.ImageTransparentColor = Color.White;
            tsmiDocumentPropertyWin.Click += (s, e) =>
            {
                this.uiToolStripSplitButtonToolBarHelp.DrawEnabledButton = true;
                this.uiToolStripSplitButtonToolBarHelp.DefaultItem = tsmiDocumentPropertyWin;
                this.uiToolStripSplitButtonToolBarHelp.Image = Properties.Resources.Icon_Document_PropertyWin;
                HelpManager.ShowHelp("92116275", "");
            };

            tsmiDocumentTab = new UIToolStripMenuItem();
            toolBarItems.Add(tsmiDocumentTab);
            tsmiDocumentTab.Text = Properties.Resources.MenuItemNameDocumentTab;
            tsmiDocumentTab.Image = Properties.Resources.Icon_Document_Tab;
            tsmiDocumentTab.ImageTransparentColor = Color.White;
            tsmiDocumentTab.Enabled = false;
            tsmiDocumentTab.Click += (s, e) =>
            {
                this.uiToolStripSplitButtonToolBarHelp.DefaultItem = tsmiDocumentTab;
                this.uiToolStripSplitButtonToolBarHelp.Image = Properties.Resources.Icon_Document_Tab;
                PropertyDialog propertyDialog = this.dockContents.GetPropertyDialog(true);
                IDocumentLinker selectedTab = propertyDialog.GetCurrentPageLink();
                if (selectedTab != null && selectedTab.DocumentId.IsNullOrEmpty() == false)
                {
                    this.uiToolStripSplitButtonToolBarHelp.DrawEnabledButton = true;
                    HelpManager.ShowHelp(selectedTab.DocumentId, "");
                }
            };

            this.uiToolStripSplitButtonToolBarHelp.DropDownItems.AddRange(toolBarItems.ToArray());

            this.uiToolStripSplitButtonToolBarHelp.DefaultItem = tsmiDocumentMainWin;
            this.uiToolStripSplitButtonToolBarHelp.Image = Properties.Resources.Icon_Document_MainWin;
            this.uiToolStripSplitButtonToolBarHelp.ToolTipText = Properties.Resources.ToolTipTextToolStripSplitButtonHelp;

            this.uiToolStripSplitButtonToolBarHelp.ButtonSelectedImage = Properties.Resources.Icon_LeftButton_Selected;
            this.uiToolStripSplitButtonToolBarHelp.ButtonNonSelectedImage = Properties.Resources.Icon_LeftButton_NonSelected;
            this.uiToolStripSplitButtonToolBarHelp.ButtonDownImage = Properties.Resources.Icon_LeftButton_Down;
            this.uiToolStripSplitButtonToolBarHelp.DropDownButtonSelectedImage = Properties.Resources.Icon_DropDownButton_Selected;
            this.uiToolStripSplitButtonToolBarHelp.DropDownButtonImage = Properties.Resources.Icon_DropDownButton;
            this.uiToolStripSplitButtonToolBarHelp.DropDownButtonNonSelectedImage = Properties.Resources.Icon_DropDownButton_NonSelected;
            this.uiToolStripSplitButtonToolBarHelp.DropDownMenuOpenedImage = Properties.Resources.Icon_DropDownMenu_Opened;
            this.uiToolStripSplitButtonToolBarHelp.UseOriginalStyle = true;
            this.uiToolStripSplitButtonToolBarHelp.DrawEnabledButton = true;
            this.uiToolStripSplitButtonToolBarHelp.EnableButtons = true;
        }

        /// <summary>
        /// 元に戻す処理.
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnUndo(object sender, EventArgs e)
        {
            CommandManager.Undo();
        }

        /// <summary>
        /// やり直し処理.
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnRedo(object sender, EventArgs e)
        {
            CommandManager.Redo();
        }

        /// <summary>
        /// Called when the Copy menu is clicked.
        /// </summary>
        /// <param name="sender">Event called.</param>
        /// <param name="e">Event argument.</param>
        private void OnCopy(object sender, EventArgs e)
        {
            this.rootViewModel.Controller.CopySelectedNode();
        }

        /// <summary>
        /// Called when the Paste menu is clicked.
        /// </summary>
        /// <param name="sender">Event called.</param>
        /// <param name="e">Event argument.</param>
        private void OnPaste(object sender, EventArgs e)
        {
            if (this.rootViewModel.CanPasteEmitter)
            {
                var esetVM = this.rootViewModel.SelectedNodeViewModel as EmitterSetViewModel;
                if (esetVM != null)
                {
                    var exec = esetVM.NodePasteEmitterExecutable;
                    if (exec != null)
                    {
                        exec.Execute(null);
                    }
                }
            }
            else if (this.rootViewModel.CanPasteSelectedNode)
            {
                this.rootViewModel.Controller.PasteSelectedNode();
            }
        }

        /// <summary>
        /// エミッタセットを新規作成。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnFileNewEmitterSet(object sender, EventArgs e)
        {
            this.executableHolder.OnCreateNewEmitterSetExecutable.Execute(null);
        }

        /// <summary>
        /// エミッタを新規作成。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnFileNewEmitter(object sender, EventArgs e)
        {
            this.executableHolder.OnCreateNewEmitterExecutable.Execute(null);
        }

        /// <summary>
        /// 閉じる。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnFileClose(object sender, EventArgs e)
        {
            LayoutEngineBase.SuspendLayout();
            this.executableHolder.OnRemoveNodeExecutable.Execute(null);
            LayoutEngineBase.ResumeLayout();
        }

        /// <summary>
        /// 全て閉じる
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnFileCloseAll(object sender, EventArgs e)
        {
            object[] parameters = new object[] { null, false };

            LayoutEngineBase.SuspendLayout();
            this.executableHolder.OnCloseAllExecutable.Execute(parameters);
            LayoutEngineBase.ResumeLayout();
        }

        /// <summary>
        /// プレビューを新規作成。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnFileNewPreview(object sender, EventArgs e)
        {
            this.executableHolder.OnCreateNewPreviewExecutable.Execute(null);
        }

        /// <summary>
        /// ファイルを開く。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnFileOpen(object sender, EventArgs e)
        {
            this.executableHolder.OnFileOpenExecutable.Execute(null);
        }

        /// <summary>
        /// 保存。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnFileSave(object sender, EventArgs e)
        {
            this.executableHolder.OnFileSaveExecutable.Execute(false);
        }

        /// <summary>
        /// 別名で保存。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnFileSaveAs(object sender, EventArgs e)
        {
            this.executableHolder.OnFileSaveExecutable.Execute(true);
        }

        /// <summary>
        /// 全て上書き保存。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnFileSaveAll(object sender, EventArgs e)
        {
            this.executableHolder.OnFileSaveAllExecutable.Execute(null);
        }

        /// <summary>
        /// 全てエクスポート。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnFileSaveAllAs(object sender, EventArgs e)
        {
            this.executableHolder.OnFileSaveAllAsExecutable.Execute(null);
        }

        /// <summary>
        /// 名前を変更。
        /// </summary>
        /// <param name="sender">The sender object</param>
        /// <param name="e">event arguments</param>
        private void OnRename(object sender, EventArgs e)
        {
            PropertyDialog dialog = this.dockContents.GetPropertyDialog(true);
            dialog.Rename();
        }

        /// <summary>
        /// 伊藤テスト処理
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event.</param>
        private void MniItoTest_Click(object sender, EventArgs e)
        {
            ////using (var dialog = new ColorPickerTest())
            using (var dialog = new GradationEditorTest())
            {
                dialog.ShowDialog(this);
            }
        }

        /// <summary>
        /// パネルレイアウトを変更したときの処理を行います.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The event arguments.</param>
        private void PanesLayoutSelectedIndexChanged(object sender, EventArgs e)
        {
            this.dummyFocusButton.Select();

            bool oldLayout = OptionStore.RootOptions.Interface.UseSingleColumnLayout;
            bool newLayout;

            // 選択されたレイアウトを取得
            if (this.cbPanesLayout.SelectedIndex == 0)
            {
                newLayout = true;
                this.cbPanesLayout.ToolTipText = Properties.Resources.ToolTipTextLayoutSingle;
            }
            else
            {
                newLayout = false;
                this.cbPanesLayout.ToolTipText = Properties.Resources.ToolTipTextLayoutSideBySide;
            }

            if (oldLayout == newLayout)
            {
                return;
            }

            // オプション変更イベントを発行
            var oldOptions = (RootOptions)OptionStore.RootOptions.Clone();

            OptionStore.RootOptions.Interface.UseSingleColumnLayout = newLayout;
            OptionStore.TriggerOptionChangedEvent(null);

            var newOptions = (RootOptions)OptionStore.RootOptions.Clone();

            // オプション変更コマンドを発行
            CommandManager.Execute(new SetOptionCommand(oldOptions, newOptions, false));
        }

        /// <summary>
        /// 接続、もしくは再転送を行います.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnConnectOrSendBinaryButtonClick(object sender, EventArgs e)
        {
            if (this.IsEnableResendMessage() == false)
            {
                return;
            }

            if (!MessageManager.Instance.IsConnected)
            {
                // 接続されていなければ接続する.
                Task.Run(() => MessageManager.Instance.Connect());
                return;
            }
            else if (ViewerController.Instance.IsConnected)
            {
                using (new ProfileTimer("Send binary data"))
                {
                    // 再生を再開
                    ViewerController.Instance.SendPlay();

                    // Send Viewer option to the viewer.
                    OptionsForm.SendViewerConfig();

                    // Send all the emitter sets to the viewer.
                    ViewerMessageHelper.SendEmitterSets(
                        this.rootViewModel.WorkspaceViewModel.DataModel.EmitterSetList);

                    // Send all the models to the viewer.
                    ViewerMessageHelper.SendModels(
                        this.rootViewModel.WorkspaceViewModel.DataModel.ViewerData.ModelList);

                    // Send Viewer data to the viewer.
                    ViewerMessageHelper.SendViewer(
                        this.rootViewModel.WorkspaceViewModel.DataModel.ViewerData);
                }
            }
        }

        /// <summary>
        /// 接続・切断を行います.
        /// </summary>
        /// <param name="sender">送り元</param>
        /// <param name="e">イベント引数</param>
        private void OnConnectOrDisconnect(object sender, EventArgs e)
        {
            if (MessageManager.Instance.IsConnected == false)
            {
                // 接続されていなければ接続する.
                Task.Run(() =>
                {
                    if (MessageManager.Instance.CheckTargetExistence())
                    {
                        if (!MessageManager.Instance.Connect())
                        {
                            // 接続エラー
                            // 通信スレッドからメインスレッドに同期してイベントを実行
                            TheApp.SyncContext.Send(() =>
                            {
                                WorkspaceRootViewModel.Instance.Dialogs.ShowFailedViewerConnectionDialog();
                            });
                        }
                    }
                    else
                    {
                        // 接続先が無い
                        // 通信スレッドからメインスレッドに同期してイベントを実行
                        TheApp.SyncContext.Send(() =>
                        {
                            WorkspaceRootViewModel.Instance.Dialogs.ShowTargetManagerNotFoundDialog();
                        });
                    }
                });
            }
            else if (ViewerController.Instance.IsConnected)
            {
                ViewerMessageHelper.RequestDeleteEmitterSets(
                    this.rootViewModel.WorkspaceViewModel.DataModel.EmitterSetList);

                ViewerMessageHelper.RequestDeleteModels(
                    this.rootViewModel.WorkspaceViewModel.DataModel.ViewerData.ModelList);

                MessageManager.Instance.Disconnect();
            }
        }

        /// <summary>
        /// 切断を行います.
        /// </summary>
        /// <param name="sender">送り元</param>
        /// <param name="e">イベント引数</param>
        private void OnDisconnectButtonClick(object sender, EventArgs e)
        {
            if (ViewerController.Instance.IsConnected == true)
            {
                ViewerMessageHelper.RequestDeleteEmitterSets(
                    this.rootViewModel.WorkspaceViewModel.DataModel.EmitterSetList);
            }

            if (MessageManager.Instance.IsConnected == true)
            {
                MessageManager.Instance.Disconnect();
            }
        }

        /// <summary>
        /// コンバイナエディタ起動イベントを処理します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報.</param>
        private void OnExecuteCombinerEditor(object sender, EventArgs e)
        {
            // すでに起動してあれば、アクティブにする
            if (this.combinerEditorProcess != null)
            {
                Microsoft.VisualBasic.Interaction.AppActivate(this.combinerEditorProcess.Id);

                return;
            }

            // Nintendo SDK 構成のときのexeファイルパス
            string combinerPathNdi = IOConstants.ToolsDirectoryPath + @"\Graphics\EffectCombinerEditor\EffectCombinerEditor.exe";

            // EPD 向け構成のときのexeファイルパス
            string combinerPathEpd = IOConstants.ExecutableFolderPath + @"\CombinerEditor\EffectCombinerEditor.exe";

            string combinerPath = null;

            // コンバイナエディタのexeを探す
            if (File.Exists(combinerPathNdi))
            {
                combinerPath = combinerPathNdi;
            }
            else if (File.Exists(combinerPathEpd))
            {
                combinerPath = combinerPathEpd;
            }

            // exeファイルが見つからなければログビューにエラーを出して終了
            if (combinerPath == null)
            {
                Logger.Log("LogView", LogLevels.Warning, Properties.Resources.WarningEffectCombinerEditorNotFound);

                return;
            }

            // コンバイナエディタを起動する
            try
            {
                this.combinerEditorProcess = Process.Start(combinerPath);
                this.combinerEditorProcess.EnableRaisingEvents = true;
                this.combinerEditorProcess.Exited += (s, ee) => { this.combinerEditorProcess = null; };

                if (this.combinerEditorProcess == null)
                {
                    Logger.Log("LogView", LogLevels.Warning, Properties.Resources.WarningFailedLaunchEffectCombinerEditor);
                }
            }
            catch (System.Exception ex)
            {
                Logger.Log("LogView", LogLevels.Warning, Properties.Resources.WarningFailedLaunchEffectCombinerEditor);
                Logger.Log("LogView", LogLevels.Warning, ex.Message);
            }
        }

        /// <summary>
        /// カーブエディタダイアログを表示します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnShowCurveWindow(object sender, EventArgs e)
        {
            this.dockContents.ShowCurveEditorDialog();
        }

        /// <summary>
        /// 履歴ダイアログを表示します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnShowHistory(object sender, EventArgs e)
        {
            this.dockContents.ShowCommandHistoryDialog();
        }

        /// <summary>
        /// サムネールを撮影します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnCaptureThumbnail(object sender, EventArgs e)
        {
            EmitterSetViewModel emitterSetViewModel = HierarchyViewModel.GetParent<EmitterSetViewModel>(this.rootViewModel.SelectedNodeViewModel as ViewModelBase);

            // エミッタセットが選択されているかチェック
            if (emitterSetViewModel == null)
            {
                return;
            }

            string esetFilePath = emitterSetViewModel.FilePath;

            // エミッタセットのパスが決まっているかチェック
            if (string.IsNullOrEmpty(esetFilePath))
            {
                return;
            }

            string thumbsDirPath = Path.Combine(esetFilePath, ".NwThumbs");

            // .NwThumbs フォルダを作成
            IOUtility.SafeCreateDirectory(thumbsDirPath);

            string imageFilePath = Path.Combine(thumbsDirPath, emitterSetViewModel.FileName + IOConstants.EmitterSetFileExtension + ".png");

            // ビューアにキャプチャ命令を送信
            ViewerController.Instance.SendCapture(imageFilePath);
        }

        /// <summary>
        /// 検索ウィンドウを表示します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnShowSearchAndBatchEditWindow(object sender, EventArgs e)
        {
            this.dockContents.ShowSearchDialog();
        }

        /// <summary>
        /// テクスチャダイアログを表示します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnShowTexture(object sender, EventArgs e)
        {
            this.dockContents.ShowTextureViewerDialog();
        }

        /// <summary>
        /// ヘルプドキュメントを表示します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnShowHelp(object sender, EventArgs e)
        {
            HelpManager.ShowHelp();
        }

        /// <summary>
        /// ニュースを表示します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnShowNews(object sender, EventArgs e)
        {
            HelpManager.ShowNews();
        }

        /// <summary>
        /// トラブルシューティングを表示します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnShowTroubleshooting(object sender, EventArgs e)
        {
            HelpManager.ShowTroubleshooting();
        }

        /// <summary>
        /// オプションダイアログを表示する.
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Event argument.</param>
        private void OnShowOptionDialog(object sender, EventArgs e)
        {
            using (var optionForm = new OptionsForm())
            {
                var result = optionForm.ShowDialog(this);
                if (result == DialogResult.OK)
                {
                    OptionStore.TriggerOptionChangedEvent(null);
                }
            }
        }

        /// <summary>
        /// バージョン情報ダイアログを表示する.
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Event Argument.</param>
        private void OnShowAbout(object sender, EventArgs e)
        {
            using (var dlg = new AboutDialog.AboutDialog())
            {
                dlg.ShowDialog(this);
            }
        }

        /// <summary>
        /// ログビューが開くときに呼び出されるイベント.
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event arguments.</param>
        private void OnLogViewCtxMenuOpening(
            object sender, System.ComponentModel.CancelEventArgs e)
        {
            this.cmiLogSelectAll.Enabled = this.uiLogView1.NumMessages > 0;
            this.cmiLogCopy.Enabled = this.uiLogView1.NumSelectedMessages > 0;
            this.cmiLogClear.Enabled = this.uiLogView1.NumMessages > 0;
        }

        /// <summary>
        /// ログビューコンテキストメニュー：すべて選択.
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event arguments.</param>
        private void OnClickLogViewSelectAll(object sender, EventArgs e)
        {
            // すべて選択.
            this.uiLogView1.SelectAll();
        }

        /// <summary>
        /// ログビューコンテキストメニュー：コピー.
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event arguments.</param>
        private void OnClickLogViewCopy(object sender, EventArgs e)
        {
            // コピー.
            string text = this.uiLogView1.GetSelectedText();
            Clipboard.SetDataObject(text, true);
        }

        /// <summary>
        /// ログビューコンテキストメニュー：クリア.
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event arguments.</param>
        private void OnClickLogViewClear(object sender, EventArgs e)
        {
            // 全てクリアする.
            this.uiLogView1.Clear();
        }

        /// <summary>
        /// コントロールツリービューアを表示します。
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Event argument.</param>
        private void OnShowControlTreeViewer(object sender, EventArgs e)
        {
            if (OptionStore.RootOptions.Interface.ShowControlTreeView == true)
            {
                // Determine the default location and size for the control tree viewer.
                Rectangle bounds = OptionStore.RootOptions.Interface.ControlTreeViewBounds;
                if (bounds.IsEmpty == true)
                {
                    // Generate default bounds.
                    Screen screen = Screen.PrimaryScreen;
                    bounds.Width = (int)(screen.WorkingArea.Width * 0.9f);
                    bounds.Height = (int)(screen.WorkingArea.Height * 0.9f);
                    bounds.X = screen.WorkingArea.X +
                        ((screen.WorkingArea.Width - bounds.Width) / 2);
                    bounds.Y = screen.WorkingArea.Y +
                        ((screen.WorkingArea.Height - bounds.Height) / 2);
                }

                // Create the control tree viewer and show it.
                var controlTreeViewer = new ControlTreeViewer();
                controlTreeViewer.ShowWithBounds(this, bounds);

                // Handle size and location change event and save them to the config.
                controlTreeViewer.SizeChanged += (s, args) =>
                {
                    OptionStore.RootOptions.Interface.ControlTreeViewBounds = controlTreeViewer.Bounds;
                };

                controlTreeViewer.LocationChanged += (s, args) =>
                {
                    OptionStore.RootOptions.Interface.ControlTreeViewBounds = controlTreeViewer.Bounds;
                };
            }
        }

        /// <summary>
        /// プロパティウィンドウの表示.
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event.</param>
        private void OnShowPropertyWindow(object sender, EventArgs e)
        {
            this.dockContents.ShowPropertyDialog();
        }

        /// <summary>
        /// エフェクトブラウザウィンドウの表示.
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event.</param>
        private void OnShowEffectBrowser(object sender, EventArgs e)
        {
            this.dockContents.ShowEffectBrowserDialog();
        }

        /// <summary>
        /// テストダイアログ(竹内)
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Event argument.</param>
        private void OnTakeuchiDialogTest(object sender, EventArgs e)
        {
            var ofd = new OpenFileDialog();
            if (ofd.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            var udd2 = new EffectMaker.Udd2Converter.Udd2Converter();
            udd2.LoadDefinition(ofd.FileName);
            udd2.ConvertToFile(Path.GetDirectoryName(ofd.FileName));
        }

        /// <summary>
        /// テスト(Tanaka)
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Event argument.</param>
        private void OnTanakaTest(object sender, EventArgs e)
        {
            this.dockContents.ShowEmbededViewerDialog();
        }

        /// <summary>
        /// テストダイアログ(Chez)
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Event argument.</param>
        private void OnChezDialogTest(object sender, EventArgs e)
        {
        }

        /// <summary>
        /// テストダイアログ(平林)
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Event argument.</param>
        private void OnHirabayashiDialogTest(object sender, EventArgs e)
        {
            var testDialog = new HirabayashiTestDialog();

            testDialog.Show(this);
        }

        /// <summary>
        /// 再転送が有効かを調べる.
        /// </summary>
        /// <returns>再転送が有効ならtrueを返す。</returns>
        private bool IsEnableResendMessage()
        {
            // 切断状態か、エミッターセットが1つ以上ある場合
            if (ViewerController.Instance.IsConnected)
            {
                var list = this.rootViewModel.WorkspaceViewModel.DataModel.EmitterSetList;

                return list.Any();
            }
            else
            {
                return true;
            }
        }

        /// <summary>
        /// プレビューメニューが開く前に呼び出されるイベント.
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event.</param>
        private void OnPreviewDropDownOpening(object sender, EventArgs e)
        {
            // 切断状態か、エミッターセットが1つ以上ある場合
            this.mniConnectViewer.Enabled = this.IsEnableResendMessage();
        }

        /// <summary>
        /// プレビューメニューが閉じたときに呼び出されるイベント.
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event.</param>
        private void OnPreviewDropDownClosed(object sender, EventArgs e)
        {
            // この処理は、接続中でエミッターが作成されていない時、再転送(F5)が無効だが、
            // その後にエミッターを追加すると、メニューアイテムが無効になっているため、
            // F5キーを押すことができない状態になることを、回避する事を目的とする。
            this.mniConnectViewer.Enabled = true;
        }

        /// <summary>
        /// フォームを閉じるときに呼び出されるイベント.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The event argument</param>
        private void OnFormClosing(object sender, FormClosingEventArgs e)
        {
            LayoutEngineBase.SuspendLayout();

            try
            {
                this.rootViewModel.IsExiting = true;

                // 未保存のファイルを保存する
                object[] parameters = new object[] { null, TheApp.fatalErrorHandled };

                this.executableHolder.OnCloseAllExecutable.Execute(parameters);

                // 処理されないファイルがあったとき、Closeイベントをキャンセルする
                if ((bool)parameters[0] == false)
                {
                    this.rootViewModel.IsExiting = false;
                    e.Cancel = true;
                    return;
                }

                PropertyDialog propertyDialog = this.dockContents.GetPropertyDialog(true);
                propertyDialog.DetachSelectedNodeChanged();

                // 接続状態の場合.
                if (ViewerController.Instance.IsConnected == true)
                {
                    // 削除メッセージを送る.
                    ViewerMessageHelper.RequestDeleteEmitterSets(
                        this.rootViewModel.WorkspaceViewModel.DataModel.EmitterSetList);

                    ViewerMessageHelper.RequestDeleteModels(
                        this.rootViewModel.WorkspaceViewModel.DataModel.ViewerData.ModelList);
                }

                if (MessageManager.Instance.IsConnected == true)
                {
                    // 切断.
                    MessageManager.Instance.Disconnect();
                }

                // エフェクトブラウザの設定を保存
                EffectBrowserDialog effectBrowser = this.dockContents.GetEffectBrowserDialog(false);
                if (effectBrowser != null)
                {
                    effectBrowser.SaveConfig();
                }

                // ドッキングウィンドウの状態を保存
                this.dockContents.SaveDockInfo();

                // ビューア埋め込みウィンドウを閉じてビューアを終了させる
                // (アプリを終了するとき OnClosing や OnClosed が呼ばれないための対応)
                this.dockContents.GetEmbededViewerDialog(false)?.Close();
            }
            finally
            {
                LayoutEngineBase.ResumeLayout();
            }
        }

        /// <summary>
        /// コンボボックスが閉じた際にフォーカスを外す（これでホイール完全無効化）
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void OnDropDownClosed(object sender, EventArgs e)
        {
            this.dummyFocusButton.Select();
        }

        /// <summary>
        /// 複製する.
        /// </summary>
        /// <param name="sender">sender.</param>
        /// <param name="e">event.</param>
        private void OnDuplicate(object sender, EventArgs e)
        {
            this.rootViewModel.Controller.DuplicateSelectedNode();
        }

        /// <summary>
        /// 削除する.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnDelete(object sender, EventArgs e)
        {
            this.rootViewModel.Controller.Remove();
        }

        /// <summary>
        /// 機能の有効 / 無効.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnEnableConvert(object sender, EventArgs e)
        {
            this.rootViewModel.Controller.SwitchEnableConvert();
        }

        /// <summary>
        /// ワークスペースを開く.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnWorkspaceOpen(object sender, EventArgs e)
        {
            string dummy = string.Empty;
            this.executableHolder.OnWorkspaceOpenExecutable.Execute(dummy);
        }

        /// <summary>
        /// ワークスペースを上書き保存.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnWorkspaceSave(object sender, EventArgs e)
        {
            this.executableHolder.OnWorkspaceSaveExecutable.Execute(false);
        }

        /// <summary>
        /// ワークスペースを名前を付けて保存.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnWorkspaceSaveAs(object sender, EventArgs e)
        {
            this.executableHolder.OnWorkspaceSaveExecutable.Execute(true);
        }

        /// <summary>
        /// 「最近使用したファイル」メニューを開くときに呼び出されるイベント.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnRecentFilesOpening(object sender, EventArgs e)
        {
            FileEventOptions filePathsOptions = OptionStore.RootOptions.FileEvent;

            this.mniFileRecentFilesRoot.Controls.Clear();

            // 最近使用したファイルがないとき、
            if (filePathsOptions.LastLoadedFileList.Count == 0)
            {
                UIToolStripMenuItem item = new UIToolStripMenuItem()
                {
                    Text = Properties.Resources.MenuFileRecentFilesNoFiles,
                    Enabled = false
                };
                this.mniFileRecentFilesRoot.Controls.Add(item);

                return;
            }

            // 最近使用したファイルを追加
            for (int i = filePathsOptions.LastLoadedFileList.Count - 1; i >= 0; --i)
            {
                string path = filePathsOptions.LastLoadedFileList[i];
                string text = Path.GetFileName(path);
                bool enabled = File.Exists(path);

                if (enabled == false)
                {
                    text += Properties.Resources.MenuFileRecentFilesNotFound;
                }

                UIToolStripMenuItem fileItem = new UIToolStripMenuItem()
                {
                    Text = text,
                    ToolTipText = path,
                    Enabled = enabled
                };
                fileItem.Click += this.OnRecentFilesItemClicked;
                this.mniFileRecentFilesRoot.DropDownItems.Add(fileItem);
            }

            // セパレータを追加
            ToolStripSeparator separator = new ToolStripSeparator();
            this.mniFileRecentFilesRoot.DropDownItems.Add(separator);

            // クリアメニューを追加
            UIToolStripMenuItem clearItem = new UIToolStripMenuItem()
            {
                Text = Properties.Resources.MenuFileRecentFilesClear
            };
            clearItem.Click += this.OnRecentFilesClearClicked;
            this.mniFileRecentFilesRoot.Controls.Add(clearItem);
        }

        /// <summary>
        /// 最近使用したファイルのメニューアイテムをクリックされたときに呼び出されるイベント.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnRecentFilesItemClicked(object sender, EventArgs e)
        {
            var senderMenu = sender as UIToolStripMenuItem;
            Debug.Assert(senderMenu != null, "senderが不正.");

            // esetファイルを開く
            string[] fileNames = new string[1] { senderMenu.ToolTipText };
            this.executableHolder.OnFileOpenExecutable.Execute(fileNames);
        }

        /// <summary>
        /// 「履歴のクリア」メニューをクリックされたときに呼び出されるイベント.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnRecentFilesClearClicked(object sender, EventArgs e)
        {
            OptionStore.RootOptions.FileEvent.ClearLastLoadedFileList();
        }

        /// <summary>
        /// 新規作成メニューを開く時にプリセットリストを登録する
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnPresetDropDownOpening(object sender, EventArgs e)
        {
            PresetUtility.SetupPresetMenu(
                this.mniFileNew,
                this.presetMenuItems,
                true,
                PresetMode.EmitterSetAndEmitter);
        }

        /// <summary>
        /// The on open with explorer.
        /// </summary>
        /// <param name="sender">
        /// The sender.
        /// </param>
        /// <param name="e">
        /// The e.
        /// </param>
        private void OnOpenWithExplorer(object sender, EventArgs e)
        {
            var esetVM = this.rootViewModel.SelectedNodeViewModel as EmitterSetViewModel;
            if (esetVM != null)
            {
                var exec = esetVM.NodeOpenExlorerExecutable;
                if (exec != null)
                {
                    exec.Execute(null);
                    return;
                }
            }

            var prevVM = this.rootViewModel.SelectedNodeViewModel as PreviewViewModel;
            if (prevVM != null)
            {
                var exec = prevVM.NodeOpenExlorerExecutable;
                if (exec != null)
                {
                    exec.Execute(null);
                    return;
                }
            }
        }

        /// <summary>
        /// ヒントを表示します。
        /// </summary>
        /// <param name="sender">使用しません</param>
        /// <param name="e">使用しません</param>
        private void OnShowHint(object sender, EventArgs e)
        {
            if (TipsDialog.Showing)
            {
                return;
            }

            var hintDialog = new TipsDialog(false);
            hintDialog.AdjustPosition(this.DesktopBounds);
            hintDialog.Show(this);
        }

        /// <summary>
        /// EXEファイルの位置を開く
        /// </summary>
        /// <param name="sender">使用しません</param>
        /// <param name="e">使用しません</param>
        private void OnOpenExecutableFolder(object sender, EventArgs e)
        {
            string path = IOConstants.ExecutableFolderPath + @"\" + Path.GetFileName(IOConstants.ExecutablePath);
            string commandline = "/e, /select,\"" + path + "\"";
            System.Diagnostics.Process.Start("EXPLORER.EXE", commandline);
        }

        /// <summary>
        /// サンプルフォルダを開く
        /// </summary>
        /// <param name="sender">使用しません</param>
        /// <param name="e">使用しません</param>
        private void OnOpenSampleFolder(object sender, EventArgs e)
        {
            string path = FavoritesManager.SampleFolderPath;
            System.Diagnostics.Process.Start("EXPLORER.EXE", path);
        }

        /// <summary>
        /// オートバックアップフォルダを開く
        /// </summary>
        /// <param name="sender">使用しません</param>
        /// <param name="e">使用しません</param>
        private void OnOpenAutoBackupFolder(object sender, EventArgs e)
        {
            string path = AppData.AutoBackupManager.AutoBackupFolderPath;
            System.Diagnostics.Process.Start("EXPLORER.EXE", path);
        }

        /// <summary>
        /// 実機ビューアを実行する
        /// </summary>
        /// <param name="sender">使用しません</param>
        /// <param name="e">使用しません</param>
        private void OnLaunchTargetViewer(object sender, EventArgs e)
        {
            string prevToolTip = this.btnToolBarLaunchViewer.ToolTipText;
            string prevMenuTip = this.mniLaunchTargetViewer.ToolTipText;

            // メニューを一時的に無効化して、ツールチップテキストを差し替える
            this.btnToolBarLaunchViewer.Enabled = false;
            this.btnToolBarLaunchViewer.ToolTipText = Resources.TargetViewerIsLaunching;
            this.mniLaunchTargetViewer.Enabled = false;
            this.mniLaunchTargetViewer.ToolTipText = Resources.TargetViewerIsLaunching;

            Action revertStatus = () =>
            {
                this.btnToolBarLaunchViewer.Enabled = true;
                this.btnToolBarLaunchViewer.ToolTipText = prevToolTip;
                this.mniLaunchTargetViewer.Enabled = true;
                this.mniLaunchTargetViewer.ToolTipText = prevMenuTip;
            };

            var result = ViewerRunner.LaunchTargetViewer();
            if (result == null)
            {
                // 起動に失敗したら即戻す
                revertStatus();
            }
            else
            {
                Task.Run(() =>
                {
                    // 10秒のタイムアウト付きで待って、プロセス終了後（起動完了後）に戻す
                    result.WaitForExit(100000);
                    TheApp.SyncContext.Send(() => revertStatus());
                });
            }
        }
    }
}
