﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Application.CommandLine
{
    /// <summary>
    /// ヘッダーオプションを処理します。
    /// </summary>
    public class EsetHeaderProcessor : CommandLineProcessorBase
    {
        /// <summary>
        /// 出力ヘッダファイル名です。
        /// </summary>
        private string esetHeaderFile;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="workingData">作業データ</param>
        /// <param name="esetHeaderFile">出力ヘッダファイル名</param>
        public EsetHeaderProcessor(CommandLineWorkingData workingData, string esetHeaderFile)
            : base(workingData)
        {
            this.esetHeaderFile = esetHeaderFile;
        }

        /// <summary>
        /// エラーログを取得します。
        /// </summary>
        public override string ErrorReport { get { return string.Empty; } }

        /// <summary>
        /// 設定処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Setup()
        {
            // ヘッダファイルオプションが指定されているのに、バイナリ出力オプションが指定されていないときはエラーで返す
            if (this.WorkingData.Converter.EmitterIndexes == null || this.WorkingData.Converter.EmitterSetIndexes == null)
            {
                Logger.Log("Console", LogLevels.Error, Properties.Resources.ConsoleMsgBinaryNotFound);
                return false;
            }

            if (string.IsNullOrEmpty(this.WorkingData.EsetHeaderFilePath) == false)
            {
                // ここには来ないはずなのでエラー出力しない
                return false;
            }

            // エミッタセットヘッダファイルの絶対パスを取得
            this.WorkingData.EsetHeaderFilePath =
                PathUtility.ToAbsolutePath(this.esetHeaderFile, TheApp.WorkingFolder);

            // 出力ディレクトリが存在するかチェック
            string dirPath = PathUtility.GetDirectoryName(this.WorkingData.EsetHeaderFilePath);

            if (Directory.Exists(dirPath) == false)
            {
                Logger.Log(
                    "Console",
                    LogLevels.Error,
                    Properties.Resources.ConsoleMsgHeaderOutputDirNotFound,
                    this.WorkingData.EsetHeaderFilePath);

                return false;
            }

            return true;
        }

        /// <summary>
        /// コマンドライン処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Process()
        {
            string headerContents = string.Empty;
            string projectName = this.WorkingData.Converter.ProjectName;
            var emitterSetIndexes = this.WorkingData.Converter.EmitterSetIndexes;
            var emitterIndexes = this.WorkingData.Converter.EmitterIndexes;

            // ヘッダ部分書き込み
            headerContents += "/*****************************************************************************\n";
            headerContents += "        " + Properties.Resources.CommentHeadeFileForRuntime + "\n";
            headerContents += "*****************************************************************************/\n";
            headerContents += "#ifndef _Converter_" + projectName + "_Included_\n";
            headerContents += "#define _Converter_" + projectName + "_Included_\n\n";

            // 名前空間
            headerContents += "namespace " + projectName + "\n";
            headerContents += "{\n";

            // エミッタセット
            headerContents += "    //\n";
            headerContents += "    //    EmitterSet\n";
            headerContents += "    //\n";

            for (int i = 0; i < emitterSetIndexes.Count; i++)
            {
                // エミッタセット名がかぶっていないかのチェック
                bool collision = false;
                for (int j = 0; j < emitterSetIndexes.Count; j++)
                {
                    if (i != j && emitterSetIndexes[i].Key == emitterSetIndexes[j].Key)
                    {
                        collision = true;
                        break;
                    }
                }

                if (collision == true)
                {
                    // エミッタセット名がだぶっていたら変数をコメントアウトする
                    headerContents += "    //const int " + emitterSetIndexes[i].Key.PadRight(40) + " = " + emitterSetIndexes[i].Value.ToString() + ";    " +
                        Properties.Resources.CommentMultipleDefinitionHeaderFileForRuntime + "\n";
                }
                else
                {
                    // エミッタセット名がだぶっていなかったら、ヘッダファイルに変数を書き込む
                    headerContents += "    const int " + emitterSetIndexes[i].Key.PadRight(40) + " = " + emitterSetIndexes[i].Value.ToString() + ";\n";
                }
            }

            // エミッタ
            headerContents += "\n\n    //\n";
            headerContents += "    //    Emitter\n";
            headerContents += "    //\n";
            foreach (KeyValuePair<string, int> emitterIndex in emitterIndexes)
            {
                headerContents += "    const int " + emitterIndex.Key.PadRight(40) + " = " + emitterIndex.Value.ToString() + ";\n";
            }

            headerContents += "} // namespace " + projectName + "\n";
            headerContents += "#endif\n";

            // ヘッダの中身が空っぽだったら終了
            if (string.IsNullOrEmpty(headerContents) == true)
            {
                return false;
            }

            // 書き込みをするためのファイルストリーム
            FileStream file = new FileStream(this.WorkingData.EsetHeaderFilePath, FileMode.Create);

            byte[] buffer = System.Text.Encoding.GetEncoding("Shift_JIS").GetBytes(headerContents);

            file.Write(buffer, 0, buffer.Length);

            // ファイルを閉じる
            file.Close();
            return true;
        }

        /// <summary>
        /// 書き換えた設定などを復元します。
        /// </summary>
        public override void Cleanup()
        {
        }
    }
}
