﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Xml;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Application.CommandLine
{
    /// <summary>
    /// エミッタリストファイルです。
    /// </summary>
    public class ConvertListFile
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ConvertListFile()
        {
        }

        /// <summary>
        /// コンバートリストファイルをロードします。
        /// </summary>
        /// <param name="path">コンバートリストファイルのパス</param>
        public bool Load(string path)
        {
            // 絶対パスを取得
            string absoluteFilePath = PathUtility.ToAbsolutePath(path, TheApp.WorkingFolder);

            // ファイルが存在するかチェック
            if (File.Exists(absoluteFilePath) == false)
            {
                Logger.Log(
                    "Console",
                    LogLevels.Error,
                    Properties.Resources.ConsoleMsgConvertParamFileNotFound,
                    absoluteFilePath);

                return false;
            }

            // 作業フォルダを取得
            string workingFolder = Path.GetDirectoryName(absoluteFilePath);

            try
            {
                // コンバートリストファイルを開く
                XmlDocument xml = new XmlDocument();
                xml.Load(absoluteFilePath);

                Func<string, string, bool> eval = (strA, strB) =>
                    string.Compare(strA, strB, StringComparison.OrdinalIgnoreCase) == 0;

                // EffectMakerCommandLineConverterArgノードを取得
                XmlElement rootNode = xml.DocumentElement;

                if (rootNode == null || !eval(rootNode.Name, "EffectMakerCommandLineConverterArg"))
                {
                    Logger.Log(
                        "Console",
                        LogLevels.Error,
                        Properties.Resources.ConsoleMsgFailParseConvertParamFile,
                        absoluteFilePath);

                    return false;
                }

                bool findBinaryNameNode = false;
                bool findEmitterSetListNode = false;
                bool findName = false;

                string binaryName = null;
                List<string> esetPathList = null;
                string projectName = null;

                // 子ノードをパース
                foreach (XmlNode child in rootNode.ChildNodes)
                {
                    if (eval(child.Name, "BinaryName"))
                    {
                        if (findBinaryNameNode)
                        {
                            Logger.Log(
                                "Console",
                                LogLevels.Error,
                                Properties.Resources.ConsoleMsgFailParseConvertParamFile,
                                absoluteFilePath);

                            return false;
                        }

                        // バイナリファイル名を取得
                        binaryName = child.InnerText;

                        findBinaryNameNode = true;
                    }
                    else if (eval(child.Name, "EmitterSetList"))
                    {
                        if (findEmitterSetListNode)
                        {
                            Logger.Log(
                                "Console",
                                LogLevels.Error,
                                Properties.Resources.ConsoleMsgFailParseConvertParamFile,
                                absoluteFilePath);

                            return false;
                        }

                        // エミッタセットリストを取得
                        esetPathList = this.ParseEmitterSetListNode(child as XmlElement, workingFolder);

                        findEmitterSetListNode = true;
                    }
                    else if (eval(child.Name, "name"))
                    {
                        if (findName)
                        {
                            Logger.Log(
                                "Console",
                                LogLevels.Error,
                                Properties.Resources.ConsoleMsgFailParseConvertParamFile,
                                absoluteFilePath);

                            return false;
                        }

                        // ヘッダファイルのnamespace名を取得
                        projectName = child.InnerText;

                        findName = true;
                    }
                }

                if (binaryName == null)
                {
                    binaryName = string.Empty;
                }

                if (esetPathList == null)
                {
                    Logger.Log(
                        "Console",
                        LogLevels.Error,
                        Properties.Resources.ConsoleMsgFailParseConvertParamFile,
                        absoluteFilePath);

                    return false;
                }

                // 読み込みが完了したらプロパティをまとめてセットする
                this.FilePath = path;
                this.AbsoluteFilePath = absoluteFilePath;
                this.WorkingFolder = workingFolder;

                this.BinaryName = binaryName;
                this.EsetPathList = esetPathList;
                this.ProjectName = projectName;
            }
            catch (Exception e)
            {
                Console.WriteLine("Failed parsing the emitter set list XML file.");
                Console.WriteLine("Error message : " + e.Message);

                return false;
            }

            return true;
        }

        /// <summary>
        /// EmitterSetListノードをパースします。
        /// </summary>
        /// <param name="esetListNode">EmitterSetListノード</param>
        /// <returns>エミッタセットファイルリストを返します。</returns>
        private List<string> ParseEmitterSetListNode(XmlElement esetListNode, string workingFolder)
        {
            Func<string, string, bool> eval = (strA, strB) =>
                string.Compare(strA, strB, StringComparison.OrdinalIgnoreCase) == 0;

            var esetFileList = new List<string>();

            foreach (XmlNode child in esetListNode.ChildNodes)
            {
                if (eval(child.Name, "file"))
                {
                    // 指定されたファイルパスをリストに登録
                    string filePath = PathUtility.ToAbsolutePath(
                        child.InnerText,
                        workingFolder);

                    if (Path.IsPathRooted(filePath) == false ||
                        Path.GetExtension(filePath).ToLower() != IOConstants.EmitterSetFileExtension ||
                        File.Exists(filePath) == false)
                    {
                        Logger.Log(
                            "Console",
                            LogLevels.Error,
                            Properties.Resources.ConsoleMsgEsetFileNotFound,
                            filePath);

                        continue;
                    }

                    esetFileList.Add(filePath);
                }
                else if (eval(child.Name, "dir"))
                {
                    string dirPath = PathUtility.ToAbsolutePath(
                        child.InnerText,
                        workingFolder);

                    if (Path.IsPathRooted(dirPath) == false ||
                        Directory.Exists(dirPath) == false)
                    {
                        Logger.Log(
                            "Console",
                            LogLevels.Error,
                            Properties.Resources.ConsoleMsgEsetDirNotFound,
                            dirPath);

                        continue;
                    }

                    // ディレクトリの情報を取得
                    DirectoryInfo dirInfo = new DirectoryInfo(dirPath);

                    FileInfo[] fileInfoList = dirInfo.GetFiles(
                        "*" + IOConstants.EmitterSetFileExtension,
                        SearchOption.AllDirectories);

                    // ファイル名順でソート
                    Array.Sort<FileInfo>(fileInfoList, delegate(FileInfo a, FileInfo b)
                    {
                        return System.IO.Path.GetFileNameWithoutExtension(a.Name).CompareTo(System.IO.Path.GetFileNameWithoutExtension(b.Name));
                    });

                    // ディレクトリ内のファイルをリストに追加
                    foreach (FileInfo info in fileInfoList)
                    {
                        if (info.Exists == false)
                        {
                            continue;
                        }

                        esetFileList.Add(info.FullName);
                    }
                }
            }

            bool distinct = false;

            // 重複するファイルがないか調べる
            for (int i = 0; i < esetFileList.Count; i++)
            {
                for (int j = i+1; j < esetFileList.Count; j++)
                {
                    if (esetFileList[i] == esetFileList[j])
                    {
                        distinct = true;

                        Logger.Log(
                            "Console",
                            LogLevels.Error,
                            Properties.Resources.ConsoleMsgEsetDuplicated,
                            esetFileList[i]);

                        break;
                    }
                }
            }

            if (esetFileList.Count == 0 || distinct)
            {
                return null;
            }

            return esetFileList;
        }

        /// <summary>
        /// コンバートリストファイルのファイルパスを取得します。
        /// </summary>
        public string FilePath { get; private set; }

        /// <summary>
        /// コンバートリストファイルの絶対パスを取得します。
        /// </summary>
        public string AbsoluteFilePath { get; private set; }

        /// <summary>
        /// 作業フォルダを取得します。
        /// </summary>
        public string WorkingFolder { get; private set; }

        /// <summary>
        /// バイナリヘッダに追加するバイナリ名を取得します。
        /// </summary>
        public string BinaryName { get; private set; }

        /// <summary>
        /// コンバートするエミッタセットファイルパスのリストを取得します。
        /// </summary>
        public List<string> EsetPathList { get; private set; }

        /// <summary>
        /// プロジェクト名を取得します。
        /// </summary>
        public string ProjectName { get; private set; }
    }
}
