﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

//// #define DEBUG_AUTO_BACKUP_MANAGER

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Threading;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.Application.AutoBackup
{
    /// <summary>
    /// オートバックアップを行うクラスです。
    /// esetファイル専用処理です。
    /// </summary>
    public class AutoBackupManager
    {
        /// <summary>
        /// 処理対象のルートビューモデル
        /// </summary>
        private WorkspaceRootViewModel rootViewModel;

        /// <summary>
        /// ワークスペースビューモデル
        /// </summary>
        private WorkspaceViewModel workspaceViewModel;

        /// <summary>
        /// オートバックアップファイルのリスト
        /// </summary>
        private LinkedList<BackupInfo> backupList = new LinkedList<BackupInfo>();

        /// <summary>
        /// スレッド
        /// </summary>
        private Thread thread;

        /// <summary>
        /// スレッド終了フラグ
        /// </summary>
        private bool threadFinishing;

        /// <summary>
        /// オートバックアップが有効かどうか
        /// </summary>
        private bool isAutoBackupActive;

        /// <summary>
        /// オートバックアップを行う間隔
        /// </summary>
        private double autoBackupInterval;

        /// <summary>
        /// 保管期限の過ぎたバックアップファイルを削除するかどうか
        /// </summary>
        private bool deleteExpiredFilesEnabled;

        /// <summary>
        /// バックアップファイルの保管期限
        /// </summary>
        private double deleteDaysLater;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public AutoBackupManager()
        {
            // 意図しないファイルが消されることのないように適当な文字列を入れておく
            this.AutoBackupFolderPath = "<Uninitialized>";
        }

        /// <summary>
        /// ファイナライザです。
        /// </summary>
        ~AutoBackupManager()
        {
            Debug.Assert(this.rootViewModel == null);
            Debug.Assert(this.workspaceViewModel == null);
            Debug.Assert(this.thread == null);
            Debug.Assert(this.backupList.First == null);
        }

        /// <summary>
        /// オートバックアップフォルダのパスを取得します。
        /// </summary>
        public string AutoBackupFolderPath { get; private set; }

        /// <summary>
        /// 初期化処理を行います。
        /// </summary>
        /// <param name="root">処理対象のルートビューモデル</param>
        public bool Initialize(WorkspaceRootViewModel root)
        {
            Debug.Assert(root != null);

            // バックアップフォルダのパスを設定
            this.AutoBackupFolderPath = Path.Combine(IOConstants.ExecutableFolderPath, "AutoBackup");

            // バックアップフォルダを作成する
            bool resCreate = IOUtility.SafeCreateTemporaryDirectory(this.AutoBackupFolderPath, IOUtility.TemporaryDirectoryUsage.ForGlobal);

            if (resCreate == false)
            {
                Logger.Log(LogLevels.Error, Properties.Resources.ErrorFailedCreatingAutoBackupFolder);
            }

            // オプションから値を取得
            this.UpdateSettings();

            // オプションにイベントハンドラを登録
            OptionStore.OptionChanged += this.OnOptionChanged;

            // 保管期限の過ぎたバックアップファイルを削除
            if (this.deleteExpiredFilesEnabled)
            {
                this.DeleteAutoBackupFiles(true);
            }

            // スレッドを作成
            this.thread = new Thread(this.ThreadProcess);
            this.thread.Name = "EffectMaker.AutoBackupManagerThread";

            this.thread.Start();

            this.rootViewModel = root;
            this.workspaceViewModel = this.rootViewModel.WorkspaceViewModel;

            // ビューモデル追加/削除のイベントハンドラを登録
            if (this.workspaceViewModel != null)
            {
                this.workspaceViewModel.ChildrenAdded += this.OnWorkspaceChildrenAdded;
                this.workspaceViewModel.ChildrenRemoved += this.OnWorkspaceChildrenRemoved;
            }

            // ワークスペースビューモデル変更のイベントハンドラを登録
            this.rootViewModel.WorkspaceViewModelChanged += this.OnWorkspaceViewModelChanged;

            #if DEBUG_AUTO_BACKUP_MANAGER
                Logger.Log(LogLevels.Information, "AutoBackupManager.Initialize: AutoBackupManager is initialized.");
            #endif

            return true;
        }

        /// <summary>
        /// 破棄処理を行います。
        /// </summary>
        public void Release()
        {
            // ワークスペースビューモデル変更のイベントハンドラを削除
            this.rootViewModel.WorkspaceViewModelChanged -= this.OnWorkspaceViewModelChanged;

            // ビューモデル追加/削除のイベントハンドラを削除
            if (this.workspaceViewModel != null)
            {
                this.workspaceViewModel.ChildrenAdded -= this.OnWorkspaceChildrenAdded;
                this.workspaceViewModel.ChildrenRemoved -= this.OnWorkspaceChildrenRemoved;
            }

            // 起動中のスレッドを停止して破棄する
            if (this.thread != null)
            {
                if (this.thread.IsAlive)
                {
                    this.threadFinishing = true;
                    this.thread.Join(3939);
                }

                this.thread = null;
            }

            // オートバックアップリストをクリア
            lock (this.backupList)
            {
                this.backupList.Clear();
            }

            // オプションに登録したイベントハンドラを削除
            OptionStore.OptionChanged -= this.OnOptionChanged;

            this.workspaceViewModel = null;
            this.rootViewModel = null;
            this.AutoBackupFolderPath = "<Uninitialized>";

            #if DEBUG_AUTO_BACKUP_MANAGER
                Logger.Log(LogLevels.Information, "AutoBackupManager.Release: AutoBackupManager is released.");
            #endif
        }

        /// <summary>
        /// オートバックアップのスレッド処理です。
        /// </summary>
        public void ThreadProcess()
        {
            // メインループ
            while (this.threadFinishing == false)
            {
                if (this.isAutoBackupActive == false)
                {
                    Thread.Sleep(500);
                    continue;
                }

                // バックアップリストをロック
                lock (this.backupList)
                {
                    // バックアップリストの先頭にあるファイルを処理
                    if (this.backupList.First != null)
                    {
                        BackupInfo info = this.backupList.First.Value;

                        bool shouldAutoBackup = false;

                        // タイムスタンプからオートバックアップするかどうか判定
                        {
                            // 前回保存したときからの経過時間を取得
                            TimeSpan elapsed = DateTime.Now.ToUniversalTime() - info.LastAutoSaveTime;

                            // 経過時間が設定値を超えていたときオートバックアップする
                            if (elapsed.TotalSeconds >= this.autoBackupInterval)
                            {
                                shouldAutoBackup = true;
                            }
                        }

                        if (shouldAutoBackup)
                        {
                            // オートバックアップを実行
                            bool resBackup = this.DoAutoBackup(info);
                            info.LastAutoSaveTime = DateTime.Now.ToUniversalTime();

                            // バックアップしたファイルをリストの終端に移動
                            this.backupList.RemoveFirst();
                            this.backupList.AddLast(info);

                            #if DEBUG_AUTO_BACKUP_MANAGER
                                string fileName = info.ViewModel.FileName + IOConstants.EmitterSetFileExtension;
                                string filePath = Path.Combine(this.AutoBackupFolderPath, fileName);

                                if (resBackup) {
                                    Logger.Log(LogLevels.Information, "AutoBackupManager.ThreadLoop: Save the File {0}.", filePath);
                                }
                                else {
                                    Logger.Log(LogLevels.Information, "AutoBackupManager.ThreadLoop: Failed to save the file {0}.", filePath);
                                }
                            #endif
                        }
                    }
                }

                Thread.Sleep(50);
            }

            this.threadFinishing = false;
        }

        /// <summary>
        /// バックアップファイルを削除します.
        /// </summary>
        /// <param name="checkTimestamp">保管期限を過ぎたファイルのみ削除するときtrue</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool DeleteAutoBackupFiles(bool checkTimestamp)
        {
            // バックアップフォルダの有無をチェック
            if (Directory.Exists(this.AutoBackupFolderPath) == false)
            {
                return false;
            }

            string[] paths;

            // 保存先フォルダ直下のファイルリストを取得
            try
            {
                paths = Directory.GetFiles(this.AutoBackupFolderPath);
            }
            catch
            {
                return false;
            }

            // それぞれのファイルについて処理
            foreach (string path in paths)
            {
                try
                {
                    // ファイルの保管時間を取得
                    DateTime lastAccessedTime = File.GetLastAccessTimeUtc(path);
                    TimeSpan elapsed = DateTime.Now.ToUniversalTime() - lastAccessedTime;

                    // 保管期限の過ぎたファイルを削除
                    if (checkTimestamp == false || elapsed.TotalDays >= this.deleteDaysLater)
                    {
                        File.Delete(path);

                        #if DEBUG_AUTO_BACKUP_MANAGER
                            Logger.Log(LogLevels.Information, "AutoBackupManager.DeleteExpiredFiles: Delete the file {0}.", Path.GetFileName(path));
                        #endif
                    }
                }
                catch
                {
                    continue;
                }
            }

            return true;
        }

        /// <summary>
        /// オートバックアップを行います。
        /// </summary>
        /// <param name="info">ファイル情報</param>
        /// <returns>処理が成功したときtrueを返します。</returns>
        private bool DoAutoBackup(BackupInfo info)
        {
            Debug.Assert(info != null && info.ViewModel != null);

            // バックアップフォルダの有無をチェック
            if (Directory.Exists(this.AutoBackupFolderPath) == false)
            {
                return false;
            }

            // ワークスペースビューモデルを取得
            var workspace = ViewModelBase.GetParent<WorkspaceViewModel>(info.ViewModel);

            if (workspace == null)
            {
                return false;
            }

            // ファイルを保存
            try
            {
                string fileName = info.ViewModel.FileName + IOConstants.EmitterSetFileExtension;
                string filePath = Path.Combine(this.AutoBackupFolderPath, fileName);

                workspace.BackupEmitterSetFile(info.ViewModel, filePath);
            }
            catch (Exception e)
            {
                Logger.Log(LogLevels.Error, Properties.Resources.ErrorFailedWritingAutoBackupFile, this.GetTargetFileName(info), e.Message);

                return false;
            }

            return true;
        }

        /// <summary>
        /// バックアップ対象の名前を取得します。
        /// 例外を握りつぶすので安全に呼び出せます。
        /// </summary>
        /// <param name="info">オートバックアップ情報</param>
        /// <returns>バックアップ対象のファイル名。ビューモデルの状態などの問題で取得できない場合はその旨を示す文字列を返します。</returns>
        private string GetTargetFileName(BackupInfo info)
        {
            string result;

            try
            {
                result = info.ViewModel.FileName;
            }
            catch
            {
                result = "<Can't get the name>";
            }

            return result;
        }

        /// <summary>
        /// オプションを変更したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnOptionChanged(object sender, EventArgs e)
        {
            this.UpdateSettings();
        }

        /// <summary>
        /// オートバックアップの設定を更新します。
        /// </summary>
        private void UpdateSettings()
        {
            // オプションから設定内容をコピー
            this.isAutoBackupActive = OptionStore.RootOptions.Details.IsAutoBackupActive;
            this.autoBackupInterval = OptionStore.RootOptions.Details.BackupInterval * 60;

            this.deleteExpiredFilesEnabled = OptionStore.RootOptions.Details.DeleteOldBackupFilesAtStartup;
            this.deleteDaysLater = OptionStore.RootOptions.Details.DeleteDaysLater;
        }

        /// <summary>
        /// ワークスペースビューモデルが変更されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="workspace">ワークスペースビューモデル</param>
        private void OnWorkspaceViewModelChanged(object sender, WorkspaceViewModel workspace)
        {
            if (this.workspaceViewModel != null)
            {
                this.workspaceViewModel.ChildrenAdded -= this.OnWorkspaceChildrenAdded;
                this.workspaceViewModel.ChildrenRemoved -= this.OnWorkspaceChildrenRemoved;
            }

            lock (this.backupList)
            {
                this.backupList.Clear();
            }

            this.workspaceViewModel = workspace;

            if (this.workspaceViewModel != null)
            {
                this.workspaceViewModel.ChildrenAdded += this.OnWorkspaceChildrenAdded;
                this.workspaceViewModel.ChildrenRemoved += this.OnWorkspaceChildrenRemoved;
            }
        }

        /// <summary>
        /// ビューモデルがワークスペースビューモデルに追加されたときの処理を行います。
        /// エミッタセットが追加されたときバックアップリストに追加します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="child">ツリーに追加されたビューモデル</param>
        private void OnWorkspaceChildrenAdded(object sender, IHierarchyObject child)
        {
            var emitterSet = child as EmitterSetViewModel;
            if (emitterSet == null) return;

            // バックアップリストをロック
            lock (this.backupList)
            {
                // エミッタセットが登録済みでないかチェック
                BackupInfo info = this.backupList.FirstOrDefault(item => item.ViewModel == emitterSet);

                if (info != null)
                {
                    return;
                }

                // オートバックアップファイルを登録
                info = new BackupInfo(emitterSet, DateTime.Now.ToUniversalTime());
                this.backupList.AddFirst(info);

                #if DEBUG_AUTO_BACKUP_MANAGER
                    Logger.Log(LogLevels.Information, "AutoBackupManager.OnViewModelAdded: Registered emitter set {0}.", info.ViewModel.FileName);
                #endif
            }
        }

        /// <summary>
        /// ビューモデルがワークスペースビューモデルから削除されたときの処理を行います。
        /// エミッタセットが削除されたときバックアップリストから削除します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="child">ツリーから削除されたビューモデル</param>
        private void OnWorkspaceChildrenRemoved(object sender, IHierarchyObject child)
        {
            var emitterSet = child as EmitterSetViewModel;
            if (emitterSet == null) return;

            // バックアップリストをロック
            lock (this.backupList)
            {
                // 登録されたエミッタセットを取得
                BackupInfo info = this.backupList.FirstOrDefault(item => item.ViewModel == emitterSet);

                // 登録を解除
                if (info != null)
                {
                    this.backupList.Remove(info);

                    #if DEBUG_AUTO_BACKUP_MANAGER
                        Logger.Log(LogLevels.Information, "AutoBackupManager.OnViewModelRemoved: Unregistered emitter set {0}.", info.ViewModel.FileName);
                    #endif
                }
            }
        }

        /// <summary>
        /// バックアップファイルの情報です。
        /// </summary>
        private class BackupInfo
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="viewModel">保存対象のビューモデル</param>
            /// <param name="saveTime">最終保存時間</param>
            public BackupInfo(EmitterSetViewModel viewModel, DateTime saveTime)
            {
                this.ViewModel = viewModel;
                this.LastAutoSaveTime = saveTime;
            }

            /// <summary>
            /// 保存対象のビューモデルを取得します。
            /// </summary>
            public EmitterSetViewModel ViewModel { get; private set; }

            /// <summary>
            /// 最終保存時間を取得または設定します。
            /// </summary>
            public DateTime LastAutoSaveTime { get; set; }
        }
    }
}
