﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Workflow.Core
{
    public class InputPlug<T> : PlugBase<T>
    {
        public T Value { get; private set; }

        internal void SetValue(T value, bool forcePropagate = false)
        {
            if (EqualityComparer<T>.Default.Equals(Value, value))
            {
                if (forcePropagate)
                    Propagate();
                return;
            }

            var oldValue = Value;

            OnValueChanging(new ValueChangedEventArgs<T>(oldValue, value));

            Value = value;
            Propagate();

            OnValueChanged(new ValueChangedEventArgs<T>(oldValue, value));
        }

        private void Propagate()
        {
            if (WorkflowItem != null)
                WorkflowItem.OnInputsInvalidated();
        }

        protected virtual void OnValueChanging(ValueChangedEventArgs<T> e)
        {
            var handler = ValueChanging;
            if (handler != null)
                handler(this, e);
        }

        protected virtual void OnValueChanged(ValueChangedEventArgs<T> e)
        {
            var handler = ValueChanged;
            if (handler != null)
                handler(this, e);
        }

        public event EventHandler<ValueChangedEventArgs<T>> ValueChanging;
        public event EventHandler<ValueChangedEventArgs<T>> ValueChanged;

        public OutputPlug<T> RemoteOutputPlug { get; private set; }

        internal void SetRemoteOutput(OutputPlug<T> output)
        {
            if (RemoteOutputPlug == output)
                return;

            var oldRemoteOutput = RemoteOutputPlug;

            OnRemoteOutputChanging(new OutputPlugChangedEventArgs<T>(oldRemoteOutput, RemoteOutputPlug));

            RemoteOutputPlug = output;
            if (RemoteOutputPlug != null)
                SetValue(RemoteOutputPlug.Value, true);
            else
                SetValue(default(T));

            OnRemoteOutputChanged(new OutputPlugChangedEventArgs<T>(oldRemoteOutput, RemoteOutputPlug));
        }

        protected virtual void OnRemoteOutputChanging(OutputPlugChangedEventArgs<T> e)
        {
            var hander = OutputPlugChanging;
            if (hander != null)
                hander(this, e);
        }

        protected virtual void OnRemoteOutputChanged(OutputPlugChangedEventArgs<T> e)
        {
            var hander = OutputPlugChanged;
            if (hander != null)
                hander(this, e);
        }

        public event EventHandler<OutputPlugChangedEventArgs<T>> OutputPlugChanging;
        public event EventHandler<OutputPlugChangedEventArgs<T>> OutputPlugChanged;

        public WorkflowItem<T> RemoteWorkflowItem
        {
            get
            {
                return RemoteOutputPlug != null ? RemoteOutputPlug.WorkflowItem : null;
            }
        }
    }
}
