﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Workflow.Core
{
    public static class ConnectionManager
    {
        public static event EventHandler BeginWiringTransaction;
        public static event EventHandler EndWiringTransaction;

        internal static void OnBeginWiringTransaction()
        {
            var handler = BeginWiringTransaction;
            if (handler != null)
                handler(null, EventArgs.Empty);
        }

        internal static void OnEndWiringTransaction()
        {
            var handler = EndWiringTransaction;
            if (handler != null)
                handler(null, EventArgs.Empty);
        }

        /// <summary>
        /// Checks whether creating a link between the two given plugs could produce a circular reference or not.
        /// </summary>
        /// <param name="outputPlug">OutputPlug to link.</param>
        /// <param name="inputPlug">InputPlug to link.</param>
        /// <returns>Returns true if the two given plugs are producing a circular reference, false otherwise.</returns>
        public static bool CheckCircularReference<T>(OutputPlug<T> outputPlug, InputPlug<T> inputPlug)
        {
            WorkflowItem<T>[] circularRoute;
            return CheckCircularReference(outputPlug, inputPlug, out circularRoute);
        }

        /// <summary>
        /// Checks whether creating a link between the two given plugs could produce a circular reference or not.
        /// </summary>
        /// <param name="outputPlug">OutputPlug to link.</param>
        /// <param name="inputPlug">InputPlug to link.</param>
        /// <param name="circularRoute">Contains the WorkflowItems that are part of the circular reference, if any.
        /// Set to null if there is no circular reference.</param>
        /// <returns>Returns true if the two given plugs are producing a circular reference, false otherwise.</returns>
        public static bool CheckCircularReference<T>(OutputPlug<T> outputPlug, InputPlug<T> inputPlug, out WorkflowItem<T>[] circularRoute)
        {
            if (outputPlug == null)
                throw new ArgumentNullException("outputPlug");
            if (inputPlug == null)
                throw new ArgumentNullException("inputPlug");

            if (inputPlug.WorkflowItem == null || inputPlug.WorkflowItem == null)
            {
                circularRoute = null;
                return false;
            }

            return CheckCircularReference(outputPlug.WorkflowItem, inputPlug.WorkflowItem, out circularRoute);
        }

        /// <summary>
        /// Checks whether creating a link between the two given WorkflowItems could produce a circular reference or not.
        /// </summary>
        /// <param name="sourceWorkflowItem">Source WorkflowItem to link.</param>
        /// <param name="targetWorkflowItem">Target WorkflowItem to link.</param>
        /// <returns>Returns true if the two given WorkflowItems are producing a circular reference, false otherwise.</returns>
        public static bool CheckCircularReference<T>(WorkflowItem<T> sourceWorkflowItem, WorkflowItem<T> targetWorkflowItem)
        {
            WorkflowItem<T>[] circularRoute;
            return CheckCircularReference(sourceWorkflowItem, targetWorkflowItem, out circularRoute);
        }

        /// <summary>
        /// Checks whether creating a link between the two given WorkflowItems could produce a circular reference or not.
        /// </summary>
        /// <param name="sourceWorkflowItem">Source WorkflowItem to link.</param>
        /// <param name="targetWorkflowItem">Target WorkflowItem to link.</param>
        /// <param name="circularRoute">Contains the WorkflowItems that are part of the circular reference, if any.
        /// Set to null if there is no circular reference.</param>
        /// <returns>Returns true if the two given WorkflowItems are producing a circular reference, false otherwise.</returns>
        public static bool CheckCircularReference<T>(WorkflowItem<T> sourceWorkflowItem, WorkflowItem<T> targetWorkflowItem, out WorkflowItem<T>[] circularRoute)
        {
            if (sourceWorkflowItem == null)
                throw new ArgumentNullException("sourceWorkflowItem");
            if (targetWorkflowItem == null)
                throw new ArgumentNullException("targetWorkflowItem");

            var knownWorkflowItems = new HashSet<WorkflowItem<T>>(Utility.UniquelyIdentifiableEqualityComparer)
            {
                sourceWorkflowItem,
            };

            var stack = new Stack<WorkflowItem<T>>();
            stack.Push(targetWorkflowItem);

            while (stack.Count > 0)
            {
                var stackWorkflowItem = stack.Pop();

                if (knownWorkflowItems.Add(stackWorkflowItem) == false)
                {
                    if (stackWorkflowItem.Equals(sourceWorkflowItem))
                    {
                        circularRoute = knownWorkflowItems.ToArray();
                        return true;
                    }
                }

                foreach (var intermediateTargetWorkflowItem in stackWorkflowItem.GetOutputWorkflowItems())
                    stack.Push(intermediateTargetWorkflowItem);
            }

            circularRoute = null;
            return false;
        }

        public static bool AreConnected<T>(OutputPlug<T> outputPlug, InputPlug<T> inputPlug)
        {
            if (outputPlug == null)
                throw new ArgumentNullException("outputPlug");
            if (inputPlug == null)
                throw new ArgumentNullException("inputPlug");

            if (inputPlug.RemoteOutputPlug != null)
                return inputPlug.RemoteOutputPlug.Identifier == outputPlug.Identifier;

            return false;
        }

        public static void Connect<T>(OutputPlug<T> outputPlug, InputPlug<T> inputPlug, bool autoDisconnect = true)
        {
            if (outputPlug == null)
                throw new ArgumentNullException("outputPlug");
            if (inputPlug == null)
                throw new ArgumentNullException("inputPlug");

            if (inputPlug.WorkflowItem != null &&
                outputPlug.WorkflowItem != null &&
                inputPlug.WorkflowItem.Identifier == outputPlug.WorkflowItem.Identifier)
            {
                // tries to connect input with output of the same workflow item
                Utility.ThrowSelfConnectionException(inputPlug, outputPlug);
            }

            if (inputPlug.RemoteOutputPlug != null)
            {
                if (autoDisconnect == false)
                    Utility.ThrowInputAlreadyPluggedException(inputPlug);
            }

            OnBeginWiringTransaction();

            try
            {
                if (inputPlug.RemoteOutputPlug != null)
                {
                    if (autoDisconnect)
                        inputPlug.RemoteOutputPlug.RemoveInputPlug(inputPlug);
                }

                outputPlug.AddInputPlug(inputPlug);
                inputPlug.SetRemoteOutput(outputPlug);
            }
            finally
            {
                OnEndWiringTransaction();
            }
        }

        public static bool TryConnect<T>(OutputPlug<T> outputPlug, InputPlug<T> inputPlug, bool autoDisconnect = true)
        {
            try
            {
                Connect(outputPlug, inputPlug, autoDisconnect);
                return true;
            }
            catch
            {
                return false;
            }
        }

        public static void Disconnect<T>(InputPlug<T> inputPlug)
        {
            if (inputPlug == null)
                throw new ArgumentNullException("inputPlug");

            if (inputPlug.RemoteOutputPlug == null)
                return;

            OnBeginWiringTransaction();

            try
            {
                inputPlug.RemoteOutputPlug.RemoveInputPlug(inputPlug);
                inputPlug.SetRemoteOutput(null);
            }
            finally
            {
                OnEndWiringTransaction();
            }
        }

        public static void Disconnect<T>(OutputPlug<T> outputPlug)
        {
            if (outputPlug == null)
                throw new ArgumentNullException("outputPlug");

            OnBeginWiringTransaction();

            try
            {
                foreach (var inputPlug in outputPlug.RemoteInputPlugs)
                    Disconnect(inputPlug);
            }
            finally
            {
                OnEndWiringTransaction();
            }
        }

        public static void Disconnect<T>(WorkflowItem<T> workflowItem)
        {
            if (workflowItem == null)
                throw new ArgumentNullException("workflowItem");

            OnBeginWiringTransaction();

            try
            {
                foreach (var inputPlug in workflowItem.InputPlugs)
                    Disconnect(inputPlug);

                foreach (var outputPlug in workflowItem.OutputPlugs)
                    Disconnect(outputPlug);
            }
            finally
            {
                OnEndWiringTransaction();
            }
        }
    }
}
