﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace ShaderTyping
{
    public enum ParameterDirection
    {
        Input,
        Output,
        Reference,
    }

    /// <summary>
    /// Most basic definition of a shader language function parameter: a name, a type, and a direction.
    /// </summary>
    public class ParameterDefinition : IEquatable<ParameterDefinition>
    {
        public ParameterDefinition(string name,
            ShaderTypeDefinition type,
            ParameterDirection direction)
        {
            if (name == null)
                throw new ArgumentNullException("name");
            if (string.IsNullOrWhiteSpace(name))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_PARAMETER, "name"), "name");
            if (type == null)
                throw new ArgumentNullException("type");
            if (Enum.IsDefined(typeof(ParameterDirection), direction) == false)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_PARAMETER, "direction"), "direction");

            Name = name;
            Type = type;
            Direction = direction;
        }

        public string Name { get; private set; }
        public ShaderTypeDefinition Type { get; private set; }
        public ParameterDirection Direction { get; private set; }

        public virtual bool IsAnOutput
        {
            get
            {
                return Direction == ParameterDirection.Output ||
                    Direction == ParameterDirection.Reference;
            }
        }

        public virtual bool IsAnInput
        {
            get
            {
                return Direction == ParameterDirection.Input ||
                    Direction == ParameterDirection.Reference;
            }
        }

        public override string ToString()
        {
            var dir = "";

            switch (Direction)
            {
                case ParameterDirection.Input: dir = "in"; break;
                case ParameterDirection.Output: dir = "out"; break;
                case ParameterDirection.Reference: dir = "ref"; break;
            }

            return string.Format("[{0}] {1} {2}", dir, Type, Name);
        }

        #region Equals

        public override bool Equals(object obj)
        {
            if (obj is ParameterDefinition)
                return Equals((ParameterDefinition)obj);

            return false;
        }

        public virtual bool Equals(ParameterDefinition other)
        {
            if ((object)other == null)
                return false;

            return Name == other.Name &&
                Type == other.Type &&
                Direction == other.Direction;
        }

        public override int GetHashCode()
        {
            return (Name + Type + Direction).GetHashCode();
        }

        public static bool operator ==(ParameterDefinition t1, ParameterDefinition t2)
        {
            if ((object)t1 == null)
                return (object)t2 == null;

            return t1.Equals(t2);
        }

        public static bool operator !=(ParameterDefinition t1, ParameterDefinition t2)
        {
            return !(t1 == t2);
        }

        #endregion
    }
}
