﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using EffectCombiner.Core.Extensions;
using ShaderTyping;

namespace ShaderGenerator.GLSL
{
    public class FunctionCall
    {
        public FunctionCall(FunctionDefinition function, IGrouping<string, PlugValue>[] inputs, PlugValue[] outputs)
        {
            if (inputs.Length != function.Parameters.Count(x => x.IsAnInput))
                throw new ArgumentException(Messages.EXCEPTION_INVALID_INPUT_PLUG_COUNT);

            Function = function;
            Inputs = inputs;
            Outputs = outputs;
        }

        public FunctionDefinition Function { get; private set; }
        public IGrouping<string, PlugValue>[] Inputs { get; private set; }
        public PlugValue[] Outputs { get; private set; }

        public string GetCallCode(string indent = null)
        {
            if (indent == null)
                indent = string.Empty;

            var variables = ExtractInputVariables();
            var arguments = CreateArgumentsAndAddOutputVariables(variables);

            return VariablesDeclarationsCode(indent, variables) +
                FunctionCallCode(indent, arguments);
        }

        private List<Tuple<Variable, Variable.Assignment[]>> ExtractInputVariables()
        {
            var variables = new List<Tuple<Variable, Variable.Assignment[]>>();

            foreach (var input in Inputs)
            {
                if (input.MoreThanOne() || input.First().BaseTarget != input.First().Target)
                {
                    var parameter = Function.Parameters.First(x => x.Name == input.Key);
                    if (parameter == null)
                    {
                        throw new Exception(string.Format("No parameter called {0} was found in function {1}", input.Key, Function.Name));
                    }
                    var type = parameter.Type;
                    var name = input.First().BaseTarget;
                    var assignments = input.Select(x => new Variable.Assignment(x.Target, x.Expression)).ToArray();
                    variables.Add(Tuple.Create(new Variable(type, name), assignments));
                }
            }
            return variables;
        }

        private List<string> CreateArgumentsAndAddOutputVariables(List<Tuple<Variable, Variable.Assignment[]>> variables)
        {
            var arguments = new List<string>();

            bool hasNoReturnValue = Function.ReturnType.IsVoid;
            int i = 0;
            int o = (hasNoReturnValue ? 0 : 1);
            foreach (var parameter in Function.Parameters)
            {
                var type = parameter.Type;
                switch (parameter.Direction)
                {
                    case ParameterDirection.Input:
                        {
                            var input = Inputs[i];
                            bool deferred = input.First().BaseTarget != input.First().Target;
                            var expression = deferred ? input.First().BaseTarget : input.First().Expression;
                            arguments.Add(expression);
                            ++i;
                            break;
                        }
                    case ParameterDirection.Output:
                        {
                            var varName = Outputs[o].Expression;
                            variables.Add(Tuple.Create(new Variable(type, varName), new Variable.Assignment[0]));
                            arguments.Add(varName);
                            ++o;
                            break;
                        }
                    case ParameterDirection.Reference:
                        {
                            var input = Inputs[i];
                            bool deferred = input.First().BaseTarget != input.First().Target;
                            var expression = deferred ? input.First().BaseTarget : input.First().Expression;

                            var varName = Outputs[o].Expression;
                            var assignment = new Variable.Assignment(varName, expression);

                            var variable = variables.Any(x => x.Item1.Name == varName);
                            if (variable == false)
                            {
                                var assignments = new Variable.Assignment[] { assignment };
                                variables.Add(Tuple.Create(new Variable(type, varName), assignments));
                            }

                            arguments.Add(varName);
                            ++i;
                            ++o;
                        }
                        break;
                }
            }
            return arguments;
        }

        private static string VariablesDeclarationsCode(string indent, List<Tuple<Variable, Variable.Assignment[]>> variables)
        {
            var declarations = string.Empty;
            foreach (var variable in variables)
            {
                if (variable.Item2.Length == 1 && variable.Item2[0].Target == variable.Item1.Name)
                {
                    declarations += indent + String.Format("{0} {1} = {2};", variable.Item1.Type, variable.Item1.Name, variable.Item2[0].Expression) + Environment.NewLine;
                }
                else
                {
                    declarations += indent + String.Format("{0} {1};", variable.Item1.Type, variable.Item1.Name) + Environment.NewLine;
                    foreach (var assignment in variable.Item2)
                    {
                        declarations += indent + String.Format("{0} = {1};", assignment.Target, assignment.Expression) + Environment.NewLine;
                    }
                }
            }
            return declarations;
        }

        private string FunctionCallCode(string indent, List<string> arguments)
        {
            bool hasNoReturnValue = Function.ReturnType.IsVoid;
            var returnOutput = (hasNoReturnValue ? string.Empty : string.Format("{0} {1} = ", Outputs[0].Type, Outputs[0].Expression));
            var functionCall = indent + returnOutput + string.Format("{0}({1});", Function.Name, string.Join(", ", arguments)) + Environment.NewLine;
            return functionCall;
        }
    }
}
