﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Runtime.InteropServices;
using System.Diagnostics;
using System.Reflection;

namespace ShaderGenerator.CafeCompiler
{
    public enum CompilationResultCode
    {
        Success = 0,
        UnsupportedGpu = 1,
        UnsupportedShaderType = 2,
        InputFileError = 3,
        OutputFileError = 4,
        BadParameter = 5,
        CompilationFailed = 6,
        InitializationFailed = 7,
        LoadingDllFailed = 8,
    }

    public enum CompilationOutcome
    {
        CompilationSucceeded,
        CompilationFailed,
        CouldNotRunCompiler
    }

    public static class CompilationResultCodeExtensions
    {
        public static string ToMessage(this CompilationResultCode code)
        {
            if (Enum.IsDefined(typeof(CompilationResultCode), code) == false)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "code"), "code");

            switch (code)
            {
                case CompilationResultCode.Success: return Messages.COMPILATION_RESULT_SUCCESS;
                case CompilationResultCode.UnsupportedGpu: return Messages.COMPILATION_RESULT_UNSUPPORTED_GPU;
                case CompilationResultCode.UnsupportedShaderType: return Messages.COMPILATION_RESULT_UNSUPPORTED_SHADER_TYPE;
                case CompilationResultCode.InputFileError: return Messages.COMPILATION_RESULT_INPUT_FILE_ERROR;
                case CompilationResultCode.OutputFileError: return Messages.COMPILATION_RESULT_OUTPUT_FILE_ERROR;
                case CompilationResultCode.BadParameter: return Messages.COMPILATION_RESULT_BAD_PARAMETER;
                case CompilationResultCode.CompilationFailed: return Messages.COMPILATION_RESULT_COMPILATION_FAILED;
                case CompilationResultCode.InitializationFailed: return Messages.COMPILATION_RESULT_INITIALIZATION_FAILED;
                case CompilationResultCode.LoadingDllFailed: return Messages.COMPILATION_RESULT_LOADING_DLL_FAILED;
            }
            throw new ArgumentException(Messages.COMPILATION_RESULT_UNKNOWN_RESULT_CODE);
        }

        public static CompilationOutcome ToOutcome(this CompilationResultCode code)
        {
            if (Enum.IsDefined(typeof(CompilationResultCode), code) == false)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "code"), "code");

            switch (code)
            {
                case CompilationResultCode.Success: return CompilationOutcome.CompilationSucceeded;
                case CompilationResultCode.UnsupportedGpu: return CompilationOutcome.CompilationFailed;
                case CompilationResultCode.UnsupportedShaderType: return CompilationOutcome.CompilationFailed;
                case CompilationResultCode.InputFileError: return CompilationOutcome.CouldNotRunCompiler;
                case CompilationResultCode.OutputFileError: return CompilationOutcome.CouldNotRunCompiler;
                case CompilationResultCode.BadParameter: return CompilationOutcome.CouldNotRunCompiler;
                case CompilationResultCode.CompilationFailed: return CompilationOutcome.CompilationFailed;
                case CompilationResultCode.InitializationFailed: return CompilationOutcome.CouldNotRunCompiler;
                case CompilationResultCode.LoadingDllFailed: return CompilationOutcome.CouldNotRunCompiler;
            }
            throw new ArgumentException(Messages.COMPILATION_RESULT_UNKNOWN_RESULT_CODE);
        }
    }

    public class Compiler
    {
        public CompilationResultCode CompileInMemory(string pixelShader, out string compiledOutput, bool forceUniformBlock, out IEnumerable<string> errors)
        {
            compiledOutput = null;
            var inputTempFile = Path.GetTempFileName();
            var outputTempFile = Path.GetTempFileName();

            try
            {
                File.WriteAllText(inputTempFile, pixelShader);
                var result = CompileFiles(inputTempFile, outputTempFile, forceUniformBlock, out errors);
                compiledOutput = File.ReadAllText(outputTempFile);

                return result;
            }
            finally
            {
                try
                {
                    File.Delete(inputTempFile);
                }
                catch { }

                try
                {
                    File.Delete(outputTempFile);
                }
                catch { }
            }
        }

        public CompilationResultCode CompileFiles(string pixelShaderFile, string compiledOutputFile, bool forceUniformBlock, out IEnumerable<string> errors)
        {
            if (string.IsNullOrWhiteSpace(pixelShaderFile))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "pixelShaderFile"), "pixelShaderFile");
            if (string.IsNullOrWhiteSpace(compiledOutputFile))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "compiledOutputFile"), "compiledOutputFile");

            errors = null;

            var dumpFile = string.Format("{0}\\{1}",
                Directory.GetCurrentDirectory(),
                PixelShaderDumpFilename);

            try
            {
                File.Delete(dumpFile);
            }
            catch { }

            var arguments = new List<string>
            {
                string.Format("-p \"{0}\"", pixelShaderFile),
                string.Format("-o \"{0}\"", compiledOutputFile),
                "-d",
            };

            if (forceUniformBlock)
                arguments.Add("-force_uniformblock");

            var startInfo = new ProcessStartInfo
            {
                Arguments = string.Join(" ", arguments.ToArray()),
                CreateNoWindow = true,
                FileName = CompilerFullFilename,
                UseShellExecute = false,
                WindowStyle = ProcessWindowStyle.Hidden,
            };

            var p = new Process { StartInfo = startInfo };

            p.Start();
            p.WaitForExit();

            if (File.Exists(dumpFile))
            {
                try
                {
                    errors = File.ReadAllLines(dumpFile)
                        .Where(l => l.StartsWith("ERROR: "))
                        .ToArray();
                }
                catch (IOException)
                {
                    errors = null;
                }
            }

            return (CompilationResultCode)p.ExitCode;
        }

        #region Static members

        private const string Win32Path = "win32";
        private const string Win64Path = "win64";
        private const string CompilerLibraryRelativePath = "system\\bin";
        private const string CompilerFilename = "gshCompile";
        private const string CompilerExtension = ".exe";

        private const string PixelShaderDumpFilename = "glslps.txt";

        public static bool IsAvailable { get; private set; }
        public static string DebugCompilerFullFilename { get; private set; }
        public static string ReleaseCompilerFullFilename { get; private set; }
        public static string CompilerFullFilename { get; private set; }
        public static string NintendoWareSdkPath { get; private set; }

        public static bool TryInitialize(string nintendoWareSdkPath)
        {
            try
            {
                Initialize(nintendoWareSdkPath);
                return true;
            }
            catch
            {
                return false;
            }
        }

        public static void Initialize(string nintendoWareSdkPath)
        {
            IsAvailable = false;

            if (string.IsNullOrWhiteSpace(nintendoWareSdkPath))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "nintendoWareSdkPath"), "nintendoWareSdkPath");

            NintendoWareSdkPath = nintendoWareSdkPath;

            var debugFilename = string.Format("{0}D{1}", CompilerFilename, CompilerExtension);
            var releaseFilename = string.Format("{0}{1}", CompilerFilename, CompilerExtension);

            var archiPath = Win32Path;
            if (IntPtr.Size == 8)
                archiPath = Win64Path;

            var path = string.Format("{0}\\{1}\\{2}",
                NintendoWareSdkPath,
                CompilerLibraryRelativePath,
                archiPath);

            DebugCompilerFullFilename = string.Format("{0}\\{1}", path, debugFilename);
            ReleaseCompilerFullFilename = string.Format("{0}\\{1}", path, releaseFilename);

            if (File.Exists(DebugCompilerFullFilename) == false)
            {
                if (File.Exists(ReleaseCompilerFullFilename) == false)
                    throw new FileNotFoundException(string.Format(Messages.EXCEPTION_FILE_NOT_FOUND, releaseFilename), ReleaseCompilerFullFilename);
                CompilerFullFilename = ReleaseCompilerFullFilename;
            }
            else
                CompilerFullFilename = DebugCompilerFullFilename;

            IsAvailable = true;
        }

        #endregion // Static members
    }
}
