﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace WinFormsTestApp
{
    public class ReportManager
    {
        private readonly ListView reportListView;

        public ReportManager(ListView reportListView)
        {
            if (reportListView == null)
                throw new ArgumentNullException("reportListView");

            this.reportListView = reportListView;
            reportListView.ColumnClick += ReportListViewColumnClick;
        }

        private void ReportListViewColumnClick(object sender, ColumnClickEventArgs e)
        {
            reportListView.ListViewItemSorter = new ReportItemsComparer(e.Column, reportListView.Sorting == SortOrder.Ascending);
            reportListView.Sort();

            if (reportListView.Sorting == SortOrder.Ascending)
                reportListView.Sorting = SortOrder.Descending;
            else
                reportListView.Sorting = SortOrder.Ascending;
        }

        public event EventHandler ItemsCleared;
        public event EventHandler ItemAdded;

        public void Clear()
        {
            reportListView.Items.Clear();

            var handler = ItemsCleared;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        public void AddInfoReport(InfoReport infoReport)
        {
            if (infoReport == null)
                throw new ArgumentNullException("infoReport");

            reportListView.Items.Add(infoReport);
            NotifyItemAdded();
        }

        public void AddWarningReport(WarningReport warningReport)
        {
            if (warningReport == null)
                throw new ArgumentNullException("warningReport");

            reportListView.Items.Add(warningReport);
            NotifyItemAdded();
        }

        public void AddErrorReport(ErrorReport errorReport)
        {
            if (errorReport == null)
                throw new ArgumentNullException("errorReport");

            reportListView.Items.Add(errorReport);
            NotifyItemAdded();
        }

        private void NotifyItemAdded()
        {
            var handler = ItemAdded;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        private class ReportItemsComparer : System.Collections.IComparer
        {
            private readonly int column;
            private readonly bool order;

            public ReportItemsComparer(int column, bool order)
            {
                this.column = column;
                this.order = order;
            }

            public int Compare(object x, object y)
            {
                var a = (ReportListViewItem)x;
                var b = (ReportListViewItem)y;

                switch (column)
                {
                    case 0: // description
                        {
                            if (order)
                                return string.Compare(a.Description, b.Description, StringComparison.InvariantCultureIgnoreCase);
                            return string.Compare(b.Description, a.Description, StringComparison.InvariantCultureIgnoreCase);
                        }
                    case 1: // category
                        {
                            if (order)
                                return a.Category.CompareTo(b.Category);
                            return b.Category.CompareTo(a.Category);
                        }
                    case 2: // datetime
                        {
                            if (order)
                                return a.GeneratedOn.CompareTo(b.GeneratedOn);
                            return b.GeneratedOn.CompareTo(a.GeneratedOn);
                        }
                }

                return 0; // bad index, no reordering
            }
        }
    }

    public enum ReportCategory
    {
        Application,
        RendererComposition,
        Renderer,
    }

    public enum ReportSeverity
    {
        Info,
        Warning,
        Error,
    }

    public class InfoReport : ReportListViewItem
    {
        public InfoReport(string description, ReportCategory category)
            : base(ReportSeverity.Info, category, description)
        {
        }
    }

    public class WarningReport : ReportListViewItem
    {
        public WarningReport(string description, ReportCategory category)
            : base(ReportSeverity.Warning, category, description)
        {
        }
    }

    public class ErrorReport : ReportListViewItem
    {
        public ErrorReport(string description, ReportCategory category)
            : base(ReportSeverity.Error, category, description)
        {
        }
    }

    public class ReportListViewItem : ListViewItem
    {
        public ReportSeverity Severity { get; private set; }
        public ReportCategory Category { get; private set; }
        public DateTime GeneratedOn { get; private set; }
        public string Description { get; private set; }

        public ReportListViewItem(ReportSeverity severity, ReportCategory category, string description)
        {
            if (string.IsNullOrWhiteSpace(description))
                throw new ArgumentException("Invalid description", "description");

            GeneratedOn = DateTime.Now;
            Severity = severity;
            Category = category;
            Description = description;

            ImageIndex = (int)severity;
            SubItems[0].Text = description;
            SubItems.Add(category.ToString());
            SubItems.Add(GeneratedOn.ToString("yyyy/MM/dd HH:mm:ss.fff"));
        }
    }
}
