﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using Renderer2D.Core;
using Renderer2D.Core.WinForms;
using System.ComponentModel.Composition;
using System.Threading;
using System.Diagnostics;
using System.ComponentModel;

namespace Renderer2D.Direct2D1.WinForms
{
    [Export(typeof(Renderer))]
    [DisplayName("Direct2D Renderer")]
    public class WinFormsDirect2D1Renderer : Direct2D1Renderer, IDisposable
    {
        private readonly object renderSyncRoot = new object();

        private SharpDX.Direct2D1.Factory d2dFactory;
        private SharpDX.DirectWrite.Factory dwriteFactory;

        public override void Initialize()
        {
            base.Initialize();

            lock (renderSyncRoot)
            {
                CreateFactories();
                CreateDeviceResources();
            }
        }

        private void CreateFactories()
        {
            d2dFactory = new SharpDX.Direct2D1.Factory(SharpDX.Direct2D1.FactoryType.MultiThreaded);
            Debug.Assert(d2dFactory.NativePointer != IntPtr.Zero);

            dwriteFactory = new SharpDX.DirectWrite.Factory();
            Debug.Assert(dwriteFactory.NativePointer != IntPtr.Zero);
        }

        private void CreateDeviceResources()
        {
            lock (renderSyncRoot)
            {
                if (renderTarget != null)
                {
                    renderTarget.Dispose();
                    renderTarget = null;
                }

                var props = new SharpDX.Direct2D1.RenderTargetProperties
                {
                    PixelFormat = new SharpDX.Direct2D1.PixelFormat(
                        SharpDX.DXGI.Format.B8G8R8A8_UNorm,
                        SharpDX.Direct2D1.AlphaMode.Premultiplied),
                    Usage = SharpDX.Direct2D1.RenderTargetUsage.GdiCompatible,
                };

                renderTarget = new SharpDX.Direct2D1.DeviceContextRenderTarget(d2dFactory, props);
            }
        }

        private System.Drawing.Graphics currentGraphics;

        public override void BeginDraw(RenderContext renderContext)
        {
            var context = CheckRenderContext<WinFormsRenderContext>(renderContext);

            currentGraphics = context.RenderGraphics;
            var vpSize = context.ViewportSize;

            //render scene directly to DC
            ((SharpDX.Direct2D1.DeviceContextRenderTarget)renderTarget).BindDeviceContext(
                currentGraphics.GetHdc(),
                new SharpDX.Rectangle(0, 0, (int)vpSize.Width, (int)vpSize.Height));

            base.BeginDraw(renderContext);
        }

        public override void EndDraw()
        {
            base.EndDraw();

            if (currentGraphics != null)
                currentGraphics.ReleaseHdc();
        }

        protected override ISize GetSystemDpi()
        {
            return new Size(d2dFactory.DesktopDpi.Width, d2dFactory.DesktopDpi.Height);
        }

        protected override SharpDX.DirectWrite.TextFormat CreateTextFormat(ITextFormat textFormat)
        {
            return new SharpDX.DirectWrite.TextFormat(dwriteFactory,
                textFormat.Font.FamilyName,
                textFormat.Font.Weight.ToD2D1Weight(),
                textFormat.Font.Style.ToD2D1Style(),
                textFormat.Font.Stretch.ToD2D1FontStretch(),
                (float)((textFormat.Font.Size * 96.0) / 72.0));
        }

        protected override SharpDX.DirectWrite.TextLayout CreateTextLayout(string text, ITextFormat textFormat, double maxWidth, double maxHeight)
        {
            using (var nativeTextFormat = CreateTextFormat(textFormat))
            {
                return new SharpDX.DirectWrite.TextLayout(dwriteFactory, text, nativeTextFormat, (float)maxWidth, (float)maxHeight);
            }
        }

        protected override SharpDX.Direct2D1.PathGeometry CreatePathGeometry()
        {
            return new SharpDX.Direct2D1.PathGeometry(d2dFactory);
        }

        public void Dispose()
        {
            lock (renderSyncRoot)
            {
                if (d2dFactory != null)
                {
                    d2dFactory.Dispose();
                    d2dFactory = null;
                }

                if (dwriteFactory != null)
                {
                    dwriteFactory.Dispose();
                    dwriteFactory = null;
                }

                if (renderTarget != null)
                {
                    renderTarget.Dispose();
                    renderTarget = null;
                }
            }
        }
    }
}
