﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;

namespace Renderer2D.Core
{
    public enum DrawTextOptions
    {
        /// <summary>
        /// Text is not vertically snapped to pixel boundaries.
        /// This setting is recommended for text that is being animated.
        /// </summary>
        NoSnap,
        /// <summary>
        /// Text is clipped to the layout rectangle.
        /// </summary>
        Clip,
        /// <summary>
        /// Text is vertically snapped to pixel boundaries and is not clipped to the layout rectangle.
        /// </summary>
        None,
    }

    public enum LineSpacingMethod
    {
        Default,
        Uniform,
    }

    public enum VerticalAlignment
    {
        Top,
        Center,
        Bottom,
    }

    public enum TextHorizontalAlignment
    {
        /// <summary>
        /// The leading edge of the paragraph text is aligned to the leading edge of the layout box.
        /// </summary>
        Leading,
        /// <summary>
        /// The trailing edge of the paragraph text is aligned to the trailing edge of the layout box.
        /// </summary>
        Trailing,
        /// <summary>
        /// The center of the paragraph text is aligned to the center of the layout box.
        /// </summary>
        Center,
        /// <summary>
        /// Align text to the leading side, and also justify text to fill the lines.
        /// </summary>
        Justified,
    }

    public enum ReadingDirection
    {
        LeftToRight,
        RightToLeft,
    }

    public enum WordWrapping
    {
        Wrap,
        NoWrap,
    }

    public interface ILineSpacing
    {
        /// <summary>
        /// A value that indicates how line height is determined.
        /// </summary>
        LineSpacingMethod LineSpacingMethod { get; set; }
        /// <summary>
        /// When this method returns, contains the line height, or distance between one baseline to another.
        /// </summary>
        double Spacing { get; set; }
        /// <summary>
        /// When this method returns, contains the distance from top of line to baseline.
        /// A reasonable ratio to lineSpacing is 80 percent.
        /// </summary>
        double BaseLine { get; set; }
    }

    public class LineSpacing : ILineSpacing
    {
        /// <summary>
        /// Warning: This value is irrevelant for the moment.
        /// </summary>
        public static readonly ILineSpacing Default = new LineSpacing(LineSpacingMethod.Default, 5.0, 80.0);

        public LineSpacing()
        {
        }

        public LineSpacing(LineSpacingMethod lineSpacingMethod, double spacing, double baseLine)
        {
            LineSpacingMethod = lineSpacingMethod;
            Spacing = spacing;
            BaseLine = baseLine;
        }

        /// <summary>
        /// A value that indicates how line height is determined.
        /// </summary>
        public LineSpacingMethod LineSpacingMethod { get; set; }
        /// <summary>
        /// When this method returns, contains the line height, or distance between one baseline to another.
        /// </summary>
        public double Spacing { get; set; }
        /// <summary>
        /// When this method returns, contains the distance from top of line to baseline.
        /// A reasonable ratio to lineSpacing is 80 percent.
        /// </summary>
        public double BaseLine { get; set; }
    }

    public interface ITextFormat
    {
        IFont Font { get; set; }
        ILineSpacing LineSpacing { get; set; }
        VerticalAlignment VerticalAlignment { get; set; }
        TextHorizontalAlignment HorizontalAlignment { get; set; }
        ReadingDirection ReadingDirection { get; set; }
        WordWrapping Wrapping { get; set; }
    }

    public class TextFormat : ITextFormat
    {
        public TextFormat(IFont font)
            : this(font, Core.LineSpacing.Default, VerticalAlignment.Top, TextHorizontalAlignment.Leading)
        {
        }

        public TextFormat(IFont font, ILineSpacing lineSpacing)
            : this(font, lineSpacing, VerticalAlignment.Top, TextHorizontalAlignment.Leading)
        {
        }

        public TextFormat(IFont font, ILineSpacing lineSpacing,
            VerticalAlignment verticalAlignment, TextHorizontalAlignment horizontalAlignment)
            : this(font, lineSpacing, verticalAlignment, horizontalAlignment, ReadingDirection.LeftToRight, WordWrapping.NoWrap)
        {
        }

        public TextFormat(IFont font, ILineSpacing lineSpacing,
            VerticalAlignment verticalAlignment, TextHorizontalAlignment horizontalAlignment,
            ReadingDirection readingDirection)
            : this(font, lineSpacing, verticalAlignment, horizontalAlignment, readingDirection, WordWrapping.NoWrap)
        {
        }

        public TextFormat(IFont font, ILineSpacing lineSpacing,
            VerticalAlignment verticalAlignment, TextHorizontalAlignment horizontalAlignment,
            ReadingDirection readingDirection, WordWrapping wrapping)
        {
            if (font == null)
                throw new ArgumentNullException("font");
            if (lineSpacing == null)
                throw new ArgumentNullException("lineSpacing");

            Font = font;
            LineSpacing = lineSpacing;
            VerticalAlignment = verticalAlignment;
            HorizontalAlignment = horizontalAlignment;
            ReadingDirection = readingDirection;
            Wrapping = wrapping;
        }

        public IFont Font { get; set; }
        public ILineSpacing LineSpacing { get; set; }
        public VerticalAlignment VerticalAlignment { get; set; }
        public TextHorizontalAlignment HorizontalAlignment { get; set; }
        public ReadingDirection ReadingDirection { get; set; }
        public WordWrapping Wrapping { get; set; }
    }
}
