﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;

namespace Renderer2D.Core
{
    public enum CornerStyle
    {
        RoundedUpper,
        RoundedBottom,
        RoundedAll,
    }

    public interface IRectangle
    {
        double X { get; set; }
        double Y { get; set; }
        double Width { get; set; }
        double Height { get; set; }
    }

    public interface IRoundedRectangle : IRectangle
    {
        double RadiusX { get; set; }
        double RadiusY { get; set; }
        CornerStyle Style { get; set; }
    }

    public static class RectangleExtensions
    {
        public static void Deform(this IRectangle rect, double deltaSize)
        {
            if (rect == null)
                throw new ArgumentNullException("rect");

            rect.X += deltaSize;
            rect.Y += deltaSize;
            rect.Width += deltaSize;
            rect.Height += deltaSize;
        }

        public static void Deform(this IRectangle rect, double deltaLeft, double deltaTop, double deltaWidth, double deltaHeight)
        {
            if (rect == null)
                throw new ArgumentNullException("rect");

            rect.X += deltaLeft;
            rect.Y += deltaTop;
            rect.Width += deltaWidth;
            rect.Height += deltaHeight;
        }

        public static IRectangle MakeDeformed(this IRectangle rect, double deltaSize)
        {
            if (rect == null)
                throw new ArgumentNullException("rect");

            return new Rectangle(rect.X + deltaSize, rect.Y + deltaSize, rect.Width + deltaSize, rect.Height + deltaSize);
        }

        public static IRectangle MakeDeformed(this IRectangle rect, double deltaLeft, double deltaTop, double deltaWidth, double deltaHeight)
        {
            if (rect == null)
                throw new ArgumentNullException("rect");

            return new Rectangle(rect.X + deltaLeft, rect.Y + deltaTop, rect.Width + deltaWidth, rect.Height + deltaHeight);
        }

        public static double Right(this IRectangle rect)
        {
            if (rect == null)
                throw new ArgumentNullException("rect");

            return rect.X + rect.Width;
        }

        public static double Bottom(this IRectangle rect)
        {
            if (rect == null)
                throw new ArgumentNullException("rect");

            return rect.Y + rect.Height;
        }

        public static IRectangle Union(this IEnumerable<IRectangle> rectangles)
        {
            if (rectangles == null)
                throw new ArgumentNullException("rectangles");

            var left = double.MaxValue;
            var top = double.MaxValue;
            var right = double.MinValue;
            var bottom = double.MinValue;

            foreach (var rect in rectangles)
            {
                left = Math.Min(left, rect.X);
                top = Math.Min(top, rect.Y);
                right = Math.Max(right, rect.Right());
                bottom = Math.Max(bottom, rect.Bottom());
            }

            return new Rectangle(left, top, right - left, bottom - top);
        }
    }

    public class Rectangle : IRectangle
    {
        public Rectangle()
        {
        }

        public Rectangle(double x, double y, double width, double height)
        {
            X = x;
            Y = y;
            Width = width;
            Height = height;
        }

        public Rectangle(IPoint position)
        {
            if (position == null)
                throw new ArgumentNullException("position");

            X = position.X;
            Y = position.Y;
        }

        public Rectangle(IPoint point1, IPoint point2)
        {
            if (point1 == null)
                throw new ArgumentNullException("point1");
            if (point2 == null)
                throw new ArgumentNullException("point2");

            var left = Math.Min(point1.X, point2.X);
            var top = Math.Min(point1.Y, point2.Y);
            var right = Math.Max(point1.X, point2.X);
            var bottom = Math.Max(point1.Y, point2.Y);

            X = left;
            Y = top;
            Width = right - left;
            Height = bottom - top;
        }

        public Rectangle(ISize size)
        {
            if (size == null)
                throw new ArgumentNullException("size");

            Width = size.Width;
            Height = size.Height;
        }

        public Rectangle(IPoint position, ISize size)
        {
            if (position == null)
                throw new ArgumentNullException("position");
            if (size == null)
                throw new ArgumentNullException("size");

            X = position.X;
            Y = position.Y;
            Width = size.Width;
            Height = size.Height;
        }

        public Rectangle(IRectangle rectangle)
        {
            if (rectangle == null)
                throw new ArgumentNullException("rectangle");

            X = rectangle.X;
            Y = rectangle.Y;
            Width = rectangle.Width;
            Height = rectangle.Height;
        }

        public double X { get; set; }
        public double Y { get; set; }

        public double Left
        {
            get { return X; }
            set { X = value; }
        }

        public double Top
        {
            get { return Y; }
            set { Y = value; }
        }

        private double width;
        public double Width
        {
            get { return width; }
            set
            {
                if (value < 0.0)
                    throw new ArgumentOutOfRangeException("value", string.Format(Messages.EXCEPTION_GREATER_THAN_OR_EQUAL_TO_ZERO, "Width"));
                width = value;
            }
        }

        private double height;
        public double Height
        {
            get { return height; }
            set
            {
                if (value < 0.0)
                    throw new ArgumentOutOfRangeException("value", string.Format(Messages.EXCEPTION_GREATER_THAN_OR_EQUAL_TO_ZERO, "Height"));
                height = value;
            }
        }

        public double Right
        {
            get { return X + Width; }
            set { Width = value - X; }
        }

        public double Bottom
        {
            get { return Y + Height; }
            set { Height = value - Y; }
        }
    }

    public class RoundedRectangle : Rectangle, IRoundedRectangle
    {
        public RoundedRectangle(double x, double y, double width, double height)
            : base(x, y, width, height)
        {
            this.Style = CornerStyle.RoundedAll;
        }

        public RoundedRectangle(IPoint position, ISize size)
            : base(position, size)
        {
            this.Style = CornerStyle.RoundedAll;
        }

        public RoundedRectangle(IRectangle rectangle)
            : base(rectangle)
        {
            this.Style = CornerStyle.RoundedAll;
        }

        public RoundedRectangle(double x, double y, double width, double height, double radiusX, double radiusY)
            : base(x, y, width, height)
        {
            RadiusX = radiusX;
            RadiusY = radiusY;
            this.Style = CornerStyle.RoundedAll;
        }

        public RoundedRectangle(double x, double y, double width, double height, double radiusX, double radiusY, CornerStyle style)
            : base(x, y, width, height)
        {
            RadiusX = radiusX;
            RadiusY = radiusY;
            this.Style = style;
        }

        public RoundedRectangle(IPoint position, ISize size, double radiusX, double radiusY)
            : base(position, size)
        {
            RadiusX = radiusX;
            RadiusY = radiusY;
            this.Style = CornerStyle.RoundedAll;
        }

        public RoundedRectangle(IRectangle rectangle, double radiusX, double radiusY)
            : base(rectangle)
        {
            RadiusX = radiusX;
            RadiusY = radiusY;
            this.Style = CornerStyle.RoundedAll;
        }

        public RoundedRectangle(IRectangle rectangle, double radiusX, double radiusY, CornerStyle style)
            : base(rectangle)
        {
            RadiusX = radiusX;
            RadiusY = radiusY;
            this.Style = style;
        }

        public double RadiusX { get; set; }
        public double RadiusY { get; set; }
        public CornerStyle Style { get; set; }
    }
}
