﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;

namespace Renderer2D.Core
{
    public enum BitmapInterpolationMode
    {
        /// <summary>
        /// Use the exact color of the nearest bitmap pixel to the current rendering pixel.
        /// </summary>
        NearestNeighbor,
        /// <summary>
        /// Interpolate a color from the four bitmap pixels that are the nearest to the rendering pixel.
        /// </summary>
        Linear,
    }

    public interface IBitmap
    {
        ISize Size { get; }
    }

    /// <summary>
    /// Represent a 32 bits RGBA bitmap definition.
    /// </summary>
    public interface IBitmapDefinition
    {
        byte[] Data { get; }
        ISize Size { get; }
    }

    public class BitmapDefinition : IBitmapDefinition
    {
        public BitmapDefinition(byte[] data, ISize size)
        {
            if (data == null)
                throw new ArgumentNullException("data");
            if (size == null)
                throw new ArgumentNullException("size");

            Initialize(data, (int)size.Width, (int)size.Height);
        }

        public BitmapDefinition(byte[] data, int width, int height)
        {
            if (data == null)
                throw new ArgumentNullException("data");

            Initialize(data, width, height);
        }

        private void Initialize(byte[] data, int width, int height)
        {
            if (width < 0)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_GREATER_THAN_OR_EQUAL_TO_ZERO, "width"), "width");
            if (height < 0)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_GREATER_THAN_OR_EQUAL_TO_ZERO, "height"), "height");

            if (width * height * 4 != data.Length)
            {
                var message = string.Format(Messages.EXCEPTION_DATA_SIZE_MISMATCH,
                    width, height, width * height * 4, data.Length);
                throw new ArgumentOutOfRangeException(message);
            }

            Data = data;
            Size = new Size(width, height);
        }

        public byte[] Data { get; private set; }
        public ISize Size { get; private set; }
    }
}
