﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace Renderer2D.Core.WinForms
{
    public abstract partial class RenderControl : UserControl
    {
        public Renderer Renderer { get; private set; }

        public WinFormsRenderContext RenderContext { get; protected set; }

        private readonly BufferedGraphicsContext doubleBufferedContext = new BufferedGraphicsContext();
        private BufferedGraphics bufferedGraphics;

        protected RenderControl()
        {
            InitializeComponent();

            SetStyle(ControlStyles.AllPaintingInWmPaint | ControlStyles.UserPaint, true);

            bufferedGraphics = doubleBufferedContext.Allocate(CreateGraphics(), ClientRectangle);

            RenderContext = new WinFormsRenderContext(this);
            RenderContext.UpdateGraphics(bufferedGraphics.Graphics);
        }

        public void SetRenderer(Renderer renderer)
        {
            if (renderer == null)
                throw new ArgumentNullException("renderer");

            Renderer = renderer;

            OnRendererAcquired();
        }

        public event EventHandler RendererAcquired;

        protected virtual void OnRendererAcquired()
        {
            var handler = RendererAcquired;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        protected sealed override void OnPaintBackground(PaintEventArgs e)
        {
            // commented on purpose
            //base.OnPaintBackground(e);
        }

        protected sealed override void OnPaint(PaintEventArgs e)
        {
            // commented on purpose
            //base.OnPaint(e);

            if (DesignMode || bufferedGraphics == null || Renderer == null)
                return;

            RenderContext.UpdateGraphics(bufferedGraphics.Graphics);
            RenderContext.UpdateViewportSize(new Size(Width, Height));

            Renderer.BeginDraw(RenderContext);

            try
            {
                OnRender();
            }
            finally
            {
                Renderer.EndDraw();
            }

            bufferedGraphics.Render(e.Graphics);
        }

        protected abstract void OnRender();

        protected sealed override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            if (DesignMode)
                return;

            if (bufferedGraphics != null)
            {
                bufferedGraphics.Dispose();
                bufferedGraphics = null;
            }

            if (Size.Width <= 0 || Size.Height <= 0)
                return;

            bufferedGraphics = doubleBufferedContext.Allocate(CreateGraphics(), ClientRectangle);

            if (Renderer != null)
                Renderer.OnViewportResized(new Size(Width, Height));

            OnResized();

            Invalidate();
        }

        protected virtual void OnResized()
        {
        }

        private readonly object syncRoot = new object();
        private bool isTerminated;

        public void Terminate()
        {
            if (isTerminated)
                return;

            lock (syncRoot)
            {
                if (isTerminated)
                    return;

                isTerminated = true;
            }

            if (bufferedGraphics != null)
            {
                bufferedGraphics.Dispose();
                bufferedGraphics = null;
            }

            OnTerminate();
        }

        protected virtual void OnTerminate()
        {
        }
    }
}
