﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

namespace OperationManager.Core
{
    /// <summary>
    /// 操作単位を表すインタフェースです。
    /// </summary>
    public interface IOperation
    {
        /// <summary>
        /// 操作を実行します。
        /// </summary>
        void Execute();

        /// <summary>
        /// 操作を取り消します。
        /// </summary>
        void Rollback();
    }

    /// <summary>
    /// 具体的な操作単位を実装するための基底クラスです。
    /// </summary>
    public abstract class Operation : IOperation
    {
        /// <summary>
        /// 操作を識別するGUIDです。
        /// </summary>
        private readonly Guid identifier = Guid.NewGuid();

        /// <summary>
        /// UIに表示する履歴名です。
        /// </summary>
        protected readonly string displayName;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="displayName">UIに表示する履歴名</param>
        protected Operation(string displayName)
        {
            if (string.IsNullOrWhiteSpace(displayName))
            {
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "displayName"), "displayName");
            }

            this.displayName = displayName;
        }

        /// <summary>
        /// UIに表示する履歴名を取得します。
        /// </summary>
        public string DisplayName { get { return displayName; } }

        /// <summary>
        /// 実行可能な操作であるかを取得します。
        /// <remarks>基底クラスの実装では常にtrueを返します。</remarks>
        /// </summary>
        public virtual bool CanExecute { get { return true; } }

        /// <summary>
        /// 取り消し可能な操作であるかを取得します。
        /// <remarks>基底クラスの実装では常にtrueを返します。</remarks>
        /// </summary>
        public virtual bool CanRollback { get { return true; } }

        /// <summary>
        /// 操作を識別するGUIDを取得します。
        /// </summary>
        public Guid Identifier { get { return identifier; } }

        /// <summary>
        /// 操作を実行します。
        /// </summary>
        public abstract void Execute();

        /// <summary>
        /// 操作を取り消します。
        /// </summary>
        public abstract void Rollback();
    }

    /// <summary>
    /// 任意の処理を登録可能な操作単位の実装です。
    /// </summary>
    public class AnonymousOperation : Operation
    {
        /// <summary>
        /// アンドゥ処理
        /// </summary>
        private readonly Action rollbackAction;

        /// <summary>
        /// リドゥ処理
        /// </summary>
        private readonly Action executeAction;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="displayName">UIに表示する履歴名</param>
        /// <param name="onRollback">アンドゥ時に実行するアクション</param>
        /// <param name="onExecute">リドゥ時に実行するアクション</param>
        public AnonymousOperation(string displayName, Action onRollback, Action onExecute)
            : base(displayName)
        {
            if (onRollback == null)
            {
                throw new ArgumentNullException("onRollback");
            }

            if (onExecute == null)
            {
                throw new ArgumentNullException("onExecute");
            }

            this.rollbackAction = onRollback;
            this.executeAction = onExecute;
        }

        /// <summary>
        /// 操作を実行します。
        /// </summary>
        public override void Execute()
        {
            this.executeAction();
        }

        /// <summary>
        /// 取り消し操作を実行します。
        /// </summary>
        public override void Rollback()
        {
            this.rollbackAction();
        }
    }
}
