﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace OperationManager.Core
{
    /// <summary>
    /// 操作履歴のマネージャを実装するためのインタフェースです。
    /// </summary>
    /// <typeparam name="T">操作単位に属する型</typeparam>
    public interface IOperationManager<T> where T : Operation
    {
        /// <summary>
        /// 操作単位のコレクションを取得します。
        /// </summary>
        IEnumerable<T> Operations { get; }

        /// <summary>
        /// アンドゥが実行可能な状態か否かを取得します。
        /// </summary>
        bool CanUndo { get; }

        /// <summary>
        /// リドゥが実行可能な状態か否かを取得します。
        /// </summary>
        bool CanRedo { get; }

        /// <summary>
        /// アンドゥ実行可能状態が変化した際に発生するイベントです。
        /// </summary>
        event EventHandler CanUndoChanged;

        /// <summary>
        /// リドゥ実行可能状態が変化した際に発生するイベントです。
        /// </summary>
        event EventHandler CanRedoChanged;

        /// <summary>
        /// 操作履歴を追加します。
        /// </summary>
        /// <param name="operation">追加する操作単位</param>
        /// <param name="runRedo">操作履歴の追加時にリドゥ操作を実行するか否か</param>
        void Add(T operation, bool runRedo);

        /// <summary>
        /// 複数の操作履歴を一括で追加します。
        /// </summary>
        /// <param name="operations">追加する履歴単位のコレクション</param>
        void AddRange(IEnumerable<T> operations);

        /// <summary>
        /// 操作履歴をクリアします。
        /// </summary>
        void Clear();

        /// <summary>
        /// アンドゥを実行します。
        /// </summary>
        /// <returns>アンドゥの実行が完了したらtrue,そうでなければfalse</returns>
        bool Undo();

        /// <summary>
        /// リドゥを実行します。
        /// </summary>
        /// <returns>リドゥの実行が完了したらtrue,そうでなければfalse</returns>
        bool Redo();

        /// <summary>
        /// 操作履歴の現在位置を示すGUIDを発行します。
        /// </summary>
        /// <returns>操作履歴の現在位置を示すGUID</returns>
        Guid AcquireCurrentGuid();

        /// <summary>
        /// 操作履歴の現在位置が引数に与えられたGUIDが指す操作履歴と一致するか否かを返します。
        /// </summary>
        /// <param name="identifier">現在位置に対して一致しているか否か判定したいGUID</param>
        /// <returns>引数が現在位置を指していたらtrue,そうでなければfalse</returns>
        bool IsCurrentGuid(Guid identifier);
    }
}
