﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using EffectDefinitions;

namespace EffectDefinitionsUnitTests
{
    /// <summary>
    /// Unit tests for the ShaderTypeDefinition class
    /// </summary>
    [TestClass]
    public class ShaderRuntimeTypeUnitTest
    {
        public ShaderRuntimeTypeUnitTest()
        {
            //
            // TODO: Add constructor logic here
            //
        }

        #region Void

        [TestMethod]
        public void VoidConstructor()
        {
            var voidType = ShaderTypeDefinition.Void;
            if (!voidType.IsVoid)
                throw new Exception("IsVoid should be true: void");

            var defaultType = new ShaderTypeDefinition();
            if (!defaultType.IsVoid)
                throw new Exception("IsVoid should be true: default");

            var explicitVoidType = new ShaderTypeDefinition("void");
            if (!explicitVoidType.IsVoid)
                throw new Exception("IsVoid should be true: explicit void");
        }

        #endregion

        /*
        #region Readable / Writable

        [TestMethod]
        public void ReadableWritableVoid()
        {
            var voidType = ShaderTypeDefinition.Void;
            if (voidType.IsAnOuput)
                throw new Exception("IsAnOuput should be false: void");
            if (voidType.IsAnInput)
                throw new Exception("IsAnInput should be false: void");
        }

        [TestMethod]
        public void ReadableWritableInt()
        {
            var intType = new ShaderTypeDefinition("int");
            if (!intType.IsAnOuput)
                throw new Exception("IsAnOuput should be true: int");
            if (!intType.IsAnInput)
                throw new Exception("IsAnInput should be true: int");
        }

        [TestMethod]
        public void ReadableWritableFloat()
        {
            var floatType = new ShaderTypeDefinition("float");
            if (!floatType.IsAnOuput)
                throw new Exception("IsAnOuput should be true: float");
            if (!floatType.IsAnInput)
                throw new Exception("IsAnInput should be true: float");
        }

        [TestMethod]
        public void ReadableWritableVec2()
        {
            var vec2Type = new ShaderTypeDefinition("vec2");
            if (!vec2Type.IsAnOuput)
                throw new Exception("IsAnOuput should be true: vec2");
            if (!vec2Type.IsAnInput)
                throw new Exception("IsAnInput should be true: vec2");
        }

        #endregion
        */

        #region GetHashCode

        [TestMethod]
        public void GetHashCodeTwiceVoid()
        {
            var voidType = ShaderTypeDefinition.Void;
            if (voidType.GetHashCode() != voidType.GetHashCode())
                throw new Exception("GetHashCode should be the same: same void object");

            var voidTypeCopy = voidType;
            if (voidType.GetHashCode() != voidTypeCopy.GetHashCode())
                throw new Exception("GetHashCode should be the same: void / void copy");

            var defaultType = new ShaderTypeDefinition();
            if (voidType.GetHashCode() != defaultType.GetHashCode())
                throw new Exception("GetHashCode should be the same: void / default");

            var explicitVoidType = new ShaderTypeDefinition("void");
            if (voidType.GetHashCode() != explicitVoidType.GetHashCode())
                throw new Exception("GetHashCode should be the same: void / explicit");
        }

        [TestMethod]
        public void GetHashCodeTwiceInt()
        {
            var intType = new ShaderTypeDefinition("int");
            if (intType.GetHashCode() != intType.GetHashCode())
                throw new Exception("GetHashCode should be the same: same int object");

            var otherIntType = new ShaderTypeDefinition("int");
            if (intType.GetHashCode() != otherIntType.GetHashCode())
                throw new Exception("GetHashCode should be the same: int / int");
        }

        [TestMethod]
        public void GetHashCodeTwiceFloat()
        {
            var floatType = new ShaderTypeDefinition("float");
            if (floatType.GetHashCode() != floatType.GetHashCode())
                throw new Exception("GetHashCode should be the same: same float object");

            var otherFloatType = new ShaderTypeDefinition("float");
            if (floatType.GetHashCode() != otherFloatType.GetHashCode())
                throw new Exception("GetHashCode should be the same: float / float");
        }

        [TestMethod]
        public void CompareHashCodeVoidInt()
        {
            var voidType = ShaderTypeDefinition.Void;
            var intType = new ShaderTypeDefinition("int");
            if (voidType.GetHashCode() == intType.GetHashCode())
                throw new Exception("GetHashCode should be different: void / int");
        }

        [TestMethod]
        public void CompareHashCodeVoidFloat()
        {
            var voidType = ShaderTypeDefinition.Void;
            var floatType = new ShaderTypeDefinition("float");
            if (voidType.GetHashCode() == floatType.GetHashCode())
                throw new Exception("GetHashCode should be different: void / float");
        }

        [TestMethod]
        public void CompareHashCodeIntFloat()
        {
            var intType = new ShaderTypeDefinition("int");
            var floatType = new ShaderTypeDefinition("float");
            if (intType.GetHashCode() == floatType.GetHashCode())
                throw new Exception("GetHashCode should be different: int / float");
        }

        [TestMethod]
        public void CompareHashCodeFloatFloat2()
        {
            var floatType = new ShaderTypeDefinition("float");
            var float2Type = new ShaderTypeDefinition("float2");
            if (floatType.GetHashCode() == float2Type.GetHashCode())
                throw new Exception("GetHashCode should be different: float / float2");
        }

        [TestMethod]
        public void CompareHashCodeVec2Vec3()
        {
            var vec2Type = new ShaderTypeDefinition("vec2");
            var vec3Type = new ShaderTypeDefinition("vec3");
            if (vec2Type.GetHashCode() == vec3Type.GetHashCode())
                throw new Exception("GetHashCode should be different: vec2 / vec3");
        }

        #endregion

        #region Compare

        [TestMethod]
        public void CompareVoidSelf()
        {
            var voidType = ShaderTypeDefinition.Void;
            var voidTypeCopy = voidType;
            if (voidType != voidTypeCopy)
                throw new Exception("Self-comparing should return equal: default");
        }

        [TestMethod]
        public void CompareDefaultSelf()
        {
            var defaultType = new ShaderTypeDefinition();
            var defaultTypeCopy = defaultType;
            if (defaultType != defaultTypeCopy)
                throw new Exception("Self-comparing should return equal: void");
        }

        [TestMethod]
        public void CompareIntSelf()
        {
            var intType = new ShaderTypeDefinition("int");
            var intTypeCopy = intType;
            if (intType != intTypeCopy)
                throw new Exception("Self-comparing should return equal: int");
        }

        [TestMethod]
        public void CompareIntInt()
        {
            var intType = new ShaderTypeDefinition("int");
            var otherIntType = new ShaderTypeDefinition("int");
            if (intType != otherIntType)
                throw new Exception("Comparing should return equal: int / int");
        }

        [TestMethod]
        public void CompareFloatFloat()
        {
            var floatType = new ShaderTypeDefinition("float");
            var otherFloatType = new ShaderTypeDefinition("float");
            if (floatType != otherFloatType)
                throw new Exception("Comparing should return equal: float / float");
        }

        [TestMethod]
        public void CompareVoidDefault()
        {
            var voidType = ShaderTypeDefinition.Void;
            var defaultType = new ShaderTypeDefinition();
            if (voidType != defaultType)
                throw new Exception("Comparing should return equal: void / default");
        }

        [TestMethod]
        public void CompareVoidExplicitVoid()
        {
            var voidType = ShaderTypeDefinition.Void;
            var explicitVoidType = new ShaderTypeDefinition("void");
            if (voidType != explicitVoidType)
                throw new Exception("Comparing should return equal: void / explicit void");
        }

        [TestMethod]
        public void CompareVoidInt()
        {
            var voidType = ShaderTypeDefinition.Void;
            var intType = new ShaderTypeDefinition("int");
            if (voidType == intType)
                throw new Exception("Comparing should return not equal: void / int");
        }

        [TestMethod]
        public void CompareIntUpcaseInt()
        {
            var intType = new ShaderTypeDefinition("int");
            var intTypeUpcase = new ShaderTypeDefinition("Int");
            if (intType == intTypeUpcase)
                throw new Exception("Comparing should return not equal: int / Int");
        }

        [TestMethod]
        public void CompareIntFloat()
        {
            var intType = new ShaderTypeDefinition("int");
            var floatType = new ShaderTypeDefinition("float");
            if (intType == floatType)
                throw new Exception("Comparing should return not equal: int / float");
        }

        [TestMethod]
        public void CompareFloatFloat2()
        {
            var floatType = new ShaderTypeDefinition("float");
            var float2Type = new ShaderTypeDefinition("float2");
            if (floatType == float2Type)
                throw new Exception("Comparing should return not equal: float / float2");
        }

        [TestMethod]
        public void CompareVec2Vec3()
        {
            var vec2Type = new ShaderTypeDefinition("vec2");
            var vec3Type = new ShaderTypeDefinition("vec3");
            if (vec2Type == vec3Type)
                throw new Exception("Comparing should return not equal: vec2 / vec3");
        }

        #endregion

        #region ToString

        [TestMethod]
        public void StringVoid()
        {
            var voidType = ShaderTypeDefinition.Void;
            var voidTypeToString = voidType.ToString();
            if (voidTypeToString != "void")
                throw new Exception(string.Format("ToString should return \"void\" but returned \"{0}\": void", voidTypeToString));

            var defaultType = new ShaderTypeDefinition();
            var defaultTypeToString = defaultType.ToString();
            if (defaultTypeToString != "void")
                throw new Exception(string.Format("ToString should return \"void\" but returned \"{0}\": default", defaultTypeToString));

            var explicitVoidType = new ShaderTypeDefinition("void");
            var explicitVoidTypeToString = explicitVoidType.ToString();
            if (explicitVoidTypeToString != "void")
                throw new Exception(string.Format("ToString should return \"void\" but returned \"{0}\": explicit void", explicitVoidTypeToString));
        }

        [TestMethod]
        public void StringInt()
        {
            var intType = new ShaderTypeDefinition("int");
            var intTypeToString = intType.ToString();
            if (intTypeToString != "int")
                throw new Exception(string.Format("ToString should return \"int\" but returned \"{0}\": int", intTypeToString));
        }

        [TestMethod]
        public void StringFloat()
        {
            var floatType = new ShaderTypeDefinition("float");
            var floatTypeToString = floatType.ToString();
            if (floatTypeToString != "float")
                throw new Exception(string.Format("ToString should return \"float\" but returned \"{0}\": float", floatTypeToString));
        }

        [TestMethod]
        public void StringVec2()
        {
            var vec2Type = new ShaderTypeDefinition("vec2");
            var vec2TypeToString = vec2Type.ToString();
            if (vec2TypeToString != "vec2")
                throw new Exception(string.Format("ToString should return \"vec2\" but returned \"{0}\": vec2", vec2TypeToString));
        }

        [TestMethod]
        public void StringVec3()
        {
            var vec3Type = new ShaderTypeDefinition("vec3");
            var vec3TypeToString = vec3Type.ToString();
            if (vec3TypeToString != "vec3")
                throw new Exception(string.Format("ToString should return \"vec3\" but returned \"{0}\": vec3", vec3TypeToString));
        }

        #endregion
    }
}
