﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml;
using System.Xml.Linq;

namespace EffectDefinitions
{
    public class PureXmlDefinitionReader : IDefinitionProvider
    {
        private IErrorReporter errorReporter;
        private Uri uri;
        private XDocument doc;
        private int lineOffset;

        public PureXmlDefinitionReader(IErrorReporter errorReporter, string data, Uri uri, int lineOffset = 0)
        {
            this.errorReporter = errorReporter;
            this.uri = uri;

            this.doc = XDocument.Parse(data, LoadOptions.SetLineInfo | LoadOptions.PreserveWhitespace);
            this.lineOffset = lineOffset;
        }

        public void Dispose() { }

        public DefinitionsGroup ProvideDefinitions()
        {
            return LoadDefinitionsFromXml();
        }

        #region Small utility functions

        // This is not the XPath
        private string AbsolutePath(XElement element)
        {
            if (element == null)
                return string.Empty;

            return AbsolutePath(element.Parent) + "/" + element.Name.LocalName;
        }

        private DefinitionLocation Location(IXmlLineInfo lineInfo, int length)
        {
            return new DefinitionLocation(uri,
                lineInfo.HasLineInfo(),
                lineInfo.LineNumber - lineOffset,
                lineInfo.LinePosition,
                length);
        }

        private DefinitionLocation NodeLocation(XNode node)
        {
            return Location((IXmlLineInfo)node, 0);
        }

        private DefinitionLocation ElementLocation(XElement element)
        {
            return Location((IXmlLineInfo)element, element.Name.LocalName.Length);
        }

        private DefinitionLocation AttributeLocation(XAttribute attribute)
        {
            return Location((IXmlLineInfo)attribute, attribute.Name.LocalName.Length);
        }

        private Tuple<DefinitionLocation, string> GetInnerTextOrFailLoudly(ref Error.Status status, XElement element)
        {
            var innerText = element.Value;
            if (string.IsNullOrWhiteSpace(innerText))
            {
                errorReporter.Report(Error.XMLMissingInnerTextError(ElementLocation(element), element.Name.LocalName));
                status = Error.Worse(status, Error.Status.Unrecoverable);
                return Tuple.Create((DefinitionLocation)null, string.Empty);
            }
            var childNode = element.FirstNode;
            var location = (childNode != null && ((IXmlLineInfo)childNode).HasLineInfo() ? NodeLocation(childNode) : ElementLocation(element));
            return Tuple.Create(location, innerText);
        }

        private Tuple<DefinitionLocation, string> GetAttributeOrFailLoudly(ref Error.Status status, XElement element, string attributeName)
        {
            var attribute = element.Attribute(attributeName);
            if (attribute == null || string.IsNullOrWhiteSpace(attribute.Value))
            {
                var location = (attribute == null || ((IXmlLineInfo)attribute).HasLineInfo() == false ? ElementLocation(element) : AttributeLocation(attribute));
                errorReporter.Report(Error.XMLMissingAttributeError(location, element.Name.LocalName, attributeName));
                status = Error.Worse(status, Error.Status.Unrecoverable);
                return Tuple.Create((DefinitionLocation)null, string.Empty);
            }
            return Tuple.Create(AttributeLocation(attribute), attribute.Value.Trim());
        }

        private Tuple<DefinitionLocation, string> GetAttributeIfExists(XElement element, string attributeName)
        {
            var attribute = element.Attribute(attributeName);
            if (attribute == null || string.IsNullOrWhiteSpace(attribute.Value))
            {
                return Tuple.Create((DefinitionLocation)null, string.Empty);
            }
            return Tuple.Create(AttributeLocation(attribute), attribute.Value.Trim());
        }

        #endregion

        private DefinitionsGroup LoadDefinitionsFromXml()
        {
            var elements = doc.Element("root").Elements();

            // The guess for functionDefinitions and blockDefinitions is safe,
            // but for taggings it's a pure bet
            var functionDefinitions = new List<FunctionDefinitionWithSource>();
            var blockDefinitions = new List<BlockDefinitionWithSource>();
            var taggings = new List<TaggingWithSource>();

            foreach (XElement element in elements)
            {
                switch (element.Name.LocalName)
                {
                    case "func":
                        IEnumerable<Tuple<DefinitionLocation, string>> tags;
                        var functionDefinition = LoadFunctionDefinitionFromXmlNode(element, out tags);
                        if (functionDefinition != null)
                        {
                            functionDefinitions.Add(functionDefinition);
                            var functionTaggings = tags.Select(x =>
                                new TaggingWithSource(ElementLocation(element), x.Item1, x.Item2, functionDefinition.IdSource, functionDefinition.Id));
                            taggings.AddRange(functionTaggings);
                        }
                        break;
                    case "block":
                        var blockDefinition = LoadBlockDefinitionFromXmlNode(element);
                        if (blockDefinition != null)
                        {
                            blockDefinitions.Add(blockDefinition);
                        }
                        break;
                    case "tag":
                        taggings.AddRange(LoadTaggingsFromXmlNode(element));
                        break;
                    case "code":
                        // TODO: ブロックからcodeを参照できるようにする。(関連課題SIGLO-64734)
                        break;
                    default:
                        errorReporter.Report(Error.XMLUnknownTagError(ElementLocation(element), element.Name.LocalName));
                        continue;
                }
            }
            return new DefinitionsGroup(functionDefinitions, blockDefinitions, taggings);
        }

        private FunctionDefinitionWithSource LoadFunctionDefinitionFromXmlNode(XElement element,
            out IEnumerable<Tuple<DefinitionLocation, string>> tags)
        {
            Error.Status status = Error.Status.OK;

            foreach (var attribute in element.Attributes())
            {
                switch (attribute.Name.LocalName)
                {
                    case "id": continue;
                    case "name": continue;
                    default:
                        errorReporter.Report(Error.XMLUnknownAttributeError(AttributeLocation(attribute), element.Name.LocalName, attribute.Name.LocalName));
                        status = Error.Worse(status, Error.Status.Recoverable);
                        continue;
                }
            }

            var id = GetAttributeOrFailLoudly(ref status, element, "id");
            var name = GetAttributeOrFailLoudly(ref status, element, "name");
            Tuple<DefinitionLocation, string> description;
            Tuple<DefinitionLocation, ShaderTypeDefinition> returnType;
            Tuple<DefinitionLocation, string> returnDescription;
            Tuple<DefinitionLocation, string> usage;
            List<ParameterDefinition> parameters;
            List<Tuple<DefinitionLocation, string>> foundTags;

            ScanDescription(ref status, element, out description);
            ScanReturnValue(ref status, element, out returnType, out usage, out returnDescription);
            ScanArguments(ref status, element, out parameters);
            ScanTags(element, out foundTags);

            if (status <= Error.Status.Recoverable)
            {
                tags = foundTags;
                return new FunctionDefinitionWithSource(ElementLocation(element),
                    id.Item1, id.Item2,
                    name.Item1, name.Item2,
                    returnType.Item1, returnType.Item2,
                    returnDescription.Item1, returnDescription.Item2,
                    usage.Item1, usage.Item2,
                    description.Item1, description.Item2,
                    parameters.ToArray());
            }

            tags = Enumerable.Empty<Tuple<DefinitionLocation, string>>();
            return null;
        }

        public BlockDefinitionWithSource LoadBlockDefinitionFromXmlNode(XElement element)
        {
            Error.Status status = Error.Status.OK;

            foreach (var attribute in element.Attributes())
            {
                switch (attribute.Name.LocalName)
                {
                    case "displayname": continue;
                    case "guid": continue;
                    case "group": continue;
                    default:
                        errorReporter.Report(Error.XMLUnknownAttributeError(AttributeLocation(attribute), element.Name.LocalName, attribute.Name.LocalName));
                        status = Error.Worse(status, Error.Status.Recoverable);
                        continue;
                }
            }

            var name = GetAttributeOrFailLoudly(ref status, element, "displayname");
            var guid = GetAttributeOrFailLoudly(ref status, element, "guid");
            var group = GetAttributeIfExists(element, "group");

            List<Plug> inputPlugs;
            List<Plug> outputPlugs;
            List<FunctionBinding> functionBindings;
            UniformDefinition uniform;
            PreviewDefinition preview;

            ScanPlugs(ref status, element, out inputPlugs, out outputPlugs);
            ScanFuncIds(ref status, element, out functionBindings);
            ScanUniform(ref status, element, out uniform);
            ScanPreview(ref status, element, out preview);

            if (functionBindings.Any() == false)
            {
                errorReporter.Report(Error.XMLMissingTagError(ElementLocation(element), element.Name.LocalName, "func"));
                status = Error.Worse(status, Error.Status.Unrecoverable);
            }

            if (status <= Error.Status.Recoverable)
            {
                FunctionBinding[] uniqueFunctionBindings = functionBindings
                    .GroupBy(x => x.Id)
                    .Select(g => g.First())
                    .ToArray();
                var blockDefinition = new BlockDefinitionWithSource(ElementLocation(element),
                    name.Item1, name.Item2,
                    guid.Item1, guid.Item2,
                    group.Item1, group.Item2,
                    inputPlugs, outputPlugs, uniform, preview, uniqueFunctionBindings);

                var duplicates = functionBindings.Except(uniqueFunctionBindings);
                foreach (var duplicate in duplicates)
                {
                    errorReporter.Report(Error.BlockHasDuplicateFunctionIdError(blockDefinition, duplicate));
                    status = Error.Status.Recoverable;
                }

                return blockDefinition;
            }

            return null;
        }

        public IEnumerable<TaggingWithSource> LoadTaggingsFromXmlNode(XElement element)
        {
            Error.Status tagStatus = Error.Status.OK;

            foreach (var attribute in element.Attributes())
            {
                switch (attribute.Name.LocalName)
                {
                    case "name": continue;
                    default:
                        errorReporter.Report(Error.XMLUnknownAttributeError(AttributeLocation(attribute), element.Name.LocalName, attribute.Name.LocalName));
                        tagStatus = Error.Worse(tagStatus, Error.Status.Recoverable);
                        continue;
                }
            }

            var name = GetAttributeOrFailLoudly(ref tagStatus, element, "name");

            foreach (XElement subNode in element.Elements())
            {
                Error.Status funcStatus = Error.Status.OK;

                switch (subNode.Name.LocalName)
                {
                    case "func": break;
                    default:
                        errorReporter.Report(Error.XMLUnknownTagError(ElementLocation(subNode), subNode.Name.LocalName));
                        tagStatus = Error.Worse(tagStatus, Error.Status.Recoverable);
                        continue;
                }

                foreach (var attribute in subNode.Attributes())
                {
                    switch (attribute.Name.LocalName)
                    {
                        case "id": continue;
                        default:
                            errorReporter.Report(Error.XMLUnknownAttributeError(AttributeLocation(attribute), subNode.Name.LocalName, attribute.Name.LocalName));
                            funcStatus = Error.Worse(tagStatus, Error.Status.Recoverable);
                            continue;
                    }
                }

                var funcId = GetAttributeOrFailLoudly(ref funcStatus, subNode, "id");

                if (Error.Worse(tagStatus, funcStatus) <= Error.Status.Recoverable)
                {
                    yield return new TaggingWithSource(ElementLocation(element),
                        name.Item1, name.Item2,
                        funcId.Item1, funcId.Item2);
                }
            }
        }


        #region Function description XML reading

        private void ScanDescription(ref Error.Status status,
                                     XElement funcNode,
                                     out Tuple<DefinitionLocation, string> description)
        {
            description = null;

            IEnumerable<XElement> descNodes = funcNode.Elements("description");

            bool firstNode = true;
            foreach (XElement descNode in descNodes)
                if (firstNode)
                {
                    description = Tuple.Create(ElementLocation(descNode), descNode.Value);
                    firstNode = false;
                }
                else
                {
                    errorReporter.Report(Error.XMLDuplicateTagError(ElementLocation(descNode), descNode.Name.LocalName));
                    status = Error.Worse(status, Error.Status.Recoverable);
                    break;
                }

            if (description == null)
                description = Tuple.Create((DefinitionLocation)null, string.Empty);
        }

        private void ScanReturnValue(ref Error.Status status,
                                     XElement funcNode,
                                     out Tuple<DefinitionLocation, ShaderTypeDefinition> returnType,
                                     out Tuple<DefinitionLocation, string> usage,
                                     out Tuple<DefinitionLocation, string> returnDescription)
        {
            returnType = null;
            usage = null;
            returnDescription = null;

            IEnumerable<XElement> returnNodes = funcNode.Elements("return");

            bool firstNode = true;
            foreach (XElement returnNode in returnNodes)
                if (firstNode)
                {
                    foreach (var attribute in returnNode.Attributes())
                    {
                        switch (attribute.Name.LocalName)
                        {
                            case "type":
                            case "usage": continue;
                            default:
                                errorReporter.Report(Error.XMLUnknownAttributeError(AttributeLocation(attribute), returnNode.Name.LocalName, attribute.Name.LocalName));
                                status = Error.Worse(status, Error.Status.Recoverable);
                                continue;
                        }
                    }

                    var type = GetAttributeOrFailLoudly(ref status, returnNode, "type");
                    usage = GetAttributeIfExists(returnNode, "usage");
                    returnDescription = Tuple.Create(ElementLocation(returnNode), returnNode.Value);

                    if (status <= Error.Status.Recoverable)
                    {
                        returnType = Tuple.Create(type.Item1, new ShaderTypeDefinition(type.Item2));
                    }
                    firstNode = false;
                }
                else
                {
                    errorReporter.Report(Error.XMLDuplicateTagError(ElementLocation(returnNode), returnNode.Name.LocalName));
                    status = Error.Worse(status, Error.Status.Recoverable);
                    break;
                }

            if (returnDescription == null)
                returnDescription = Tuple.Create((DefinitionLocation)null, string.Empty);
            if (returnType == null)
                returnType = Tuple.Create((DefinitionLocation)null, new ShaderTypeDefinition());
            if (usage == null)
                usage = Tuple.Create((DefinitionLocation)null, string.Empty);
        }

        private void ScanArguments(ref Error.Status status,
                                   XElement funcNode,
                                   out List<ParameterDefinition> arguments)
        {
            arguments = new List<ParameterDefinition>();
            foreach (XElement subNode in funcNode.Elements())
            {
                ShaderTyping.ParameterDirection direction;

                switch (subNode.Name.LocalName)
                {
                    case "in": direction = ShaderTyping.ParameterDirection.Input; break;
                    case "out": direction = ShaderTyping.ParameterDirection.Output; break;
                    case "ref": direction = ShaderTyping.ParameterDirection.Reference; break;
                    case "description":
                    case "tag":
                    case "return":
                        continue;
                    default:
                        errorReporter.Report(Error.XMLUnknownTagError(ElementLocation(subNode), subNode.Name.LocalName));
                        status = Error.Worse(status, Error.Status.Recoverable);
                        continue;
                }

                foreach (var attribute in subNode.Attributes())
                {
                    switch (attribute.Name.LocalName)
                    {
                        case "name":
                        case "type":
                        case "usage":
                            continue;
                        default:
                            errorReporter.Report(Error.XMLUnknownAttributeError(AttributeLocation(attribute), subNode.Name.LocalName, attribute.Name.LocalName));
                            status = Error.Worse(status, Error.Status.Recoverable);
                            continue;
                    }
                }

                var argName = GetAttributeOrFailLoudly(ref status, subNode, "name");
                var type = GetAttributeOrFailLoudly(ref status, subNode, "type");
                var s = GetAttributeIfExists(subNode, "usage");

                if (status <= Error.Status.Recoverable)
                {
                    var argDescription = Tuple.Create(ElementLocation(subNode), subNode.Value);
                    arguments.Add(new ParameterDefinitionWithSource(ElementLocation(subNode),
                        argName.Item1, argName.Item2,
                        type.Item1, new ShaderTypeDefinition(type.Item2),
                        direction,
                        s.Item1, s.Item2,
                        argDescription.Item1, argDescription.Item2));
                }
            }
        }

        private void ScanTags(XElement funcNode,
                              out List<Tuple<DefinitionLocation, string>> tags)
        {
            tags = new List<Tuple<DefinitionLocation, string>>();
            foreach (XElement tagNode in funcNode.Elements("tag"))
            {
                Error.Status status = Error.Status.OK;

                foreach (var attribute in tagNode.Attributes())
                {
                    errorReporter.Report(Error.XMLUnknownAttributeError(AttributeLocation(attribute), tagNode.Name.LocalName, attribute.Name.LocalName));
                    status = Error.Worse(status, Error.Status.Recoverable);
                }

                var tag = GetInnerTextOrFailLoudly(ref status, tagNode);

                if (status <= Error.Status.Recoverable)
                {
                    tags.Add(tag);
                }
            }
        }

        #endregion

        #region BlockDefinition description XML reading

        private void ScanPlugs(ref Error.Status status,
                               XElement blockNode,
                               out List<Plug> inputPlugs,
                               out List<Plug> outputPlugs)
        {
            inputPlugs = new List<Plug>();
            outputPlugs = new List<Plug>();
            foreach (XElement subNode in blockNode.Elements())
            {
                List<Plug> plugs;
                bool isInput;
                switch (subNode.Name.LocalName)
                {
                    case "in": plugs = inputPlugs; isInput = true; break;
                    case "out": plugs = outputPlugs; isInput = false; break;
                    case "func": continue;
                    case "uniform": continue;
                    case "preview": continue;
                    default:
                        errorReporter.Report(Error.XMLUnknownTagError(ElementLocation(subNode), subNode.Name.LocalName));
                        status = Error.Worse(status, Error.Status.Recoverable);
                        continue;
                }

                foreach (var attribute in subNode.Attributes())
                {
                    switch (attribute.Name.LocalName)
                    {
                        case "displayname":
                        case "type":
                        case "target":
                        case "semantic":
                            continue;
                        case "defaultvalue":
                            {
                                if (isInput == false)
                                {
                                    errorReporter.Report(Error.XMLUnknownAttributeError(AttributeLocation(attribute), subNode.Name.LocalName, attribute.Name.LocalName));
                                    status = Error.Worse(status, Error.Status.Recoverable);
                                }
                                continue;
                            }
                        default:
                            errorReporter.Report(Error.XMLUnknownAttributeError(AttributeLocation(attribute), subNode.Name.LocalName, attribute.Name.LocalName));
                            status = Error.Worse(status, Error.Status.Recoverable);
                            continue;
                    }
                }

                var displayName = GetAttributeOrFailLoudly(ref status, subNode, "displayname");
                var type = GetAttributeOrFailLoudly(ref status, subNode, "type");
                var plugTarget = GetAttributeOrFailLoudly(ref status, subNode, "target");
                var semantic = GetAttributeIfExists(subNode, "semantic");

                Tuple<DefinitionLocation, string> defaultValue = null;
                if (isInput)
                    defaultValue = GetAttributeIfExists(subNode, "defaultvalue");

                if (status <= Error.Status.Recoverable)
                {
                    plugs.Add(new PlugWithSource(ElementLocation(subNode),
                        displayName.Item1, displayName.Item2,
                        type.Item1, new ShaderTypeDefinition(type.Item2),
                        plugTarget.Item1, plugTarget.Item2,
                        semantic.Item1, semantic.Item2,
                        defaultValue != null ? defaultValue.Item1 : null,
                        defaultValue != null ? SplitDefaultValues(defaultValue.Item2) : null,
                        isInput));
                }
            }
        }

        /// <summary>
        /// XML の uniform 要素をパースします。
        /// </summary>
        /// <param name="status">パースステータス</param>
        /// <param name="blockNode">uniform 要素</param>
        /// <param name="preview">パース結果</param>
        private void ScanUniform(ref Error.Status status, XElement blockNode, out UniformDefinition uniform)
        {
            uniform = null;

            foreach (XElement subNode in blockNode.Elements())
            {
                if (subNode.Name.LocalName == "uniform")
                {
                    var name = GetAttributeIfExists(subNode, "name");
                    var type = GetAttributeIfExists(subNode, "type");
                    var displayname = GetAttributeIfExists(subNode, "displayname");
                    var extension = GetAttributeIfExists(subNode, "extension");

                    uniform = new UniformDefinitionWithSource(ElementLocation(subNode),
                        name.Item1, name.Item2,
                        displayname.Item1, displayname.Item2,
                        type.Item1, type.Item2,
                        extension.Item1, extension.Item2);

                    break;
                }
            }
        }

        /// <summary>
        /// XML の preview 要素をパースします。
        /// </summary>
        /// <param name="status">パースステータス</param>
        /// <param name="blockNode">preview 要素</param>
        /// <param name="preview">パース結果</param>
        private void ScanPreview(ref Error.Status status, XElement blockNode, out PreviewDefinition preview)
        {
            preview = null;

            foreach (XElement subNode in blockNode.Elements())
            {
                if (subNode.Name.LocalName == "preview")
                {
                    var enabled = GetAttributeIfExists(subNode, "enabled");
                    var location = GetAttributeIfExists(subNode, "location");
                    var width = GetAttributeIfExists(subNode, "width");
                    var height = GetAttributeIfExists(subNode, "height");

                    preview = new PreviewDefinitionWithSource(ElementLocation(subNode),
                        enabled.Item1, enabled.Item2,
                        location.Item1, location.Item2,
                        width.Item1, width.Item2,
                        height.Item1, height.Item2);

                    break;
                }
            }

            if (preview == null)
            {
                preview = new PreviewDefinition();
            }
        }

        private string[] SplitDefaultValues(string defaultValues)
        {
            if (string.IsNullOrWhiteSpace(defaultValues))
                return null;

            return defaultValues.Split(';', ',').Select(v => v.Trim()).ToArray();
        }

        private void ScanFuncIds(ref Error.Status status,
                                 XElement blockNode,
                                 out List<FunctionBinding> functionBindings)
        {
            functionBindings = new List<FunctionBinding>();
            foreach (XElement funcNode in blockNode.Elements("func"))
            {
                Error.Status funcIdStatus = Error.Status.OK;

                foreach (var attribute in funcNode.Attributes())
                {
                    switch (attribute.Name.LocalName)
                    {
                        case "id": continue;
                        case "displayname": continue;
                        default:
                            errorReporter.Report(Error.XMLUnknownAttributeError(AttributeLocation(attribute), funcNode.Name.LocalName, attribute.Name.LocalName));
                            status = Error.Worse(status, Error.Status.Recoverable);
                            continue;
                    }
                }

                var funcId = GetAttributeOrFailLoudly(ref funcIdStatus, funcNode, "id");
                var displayName = GetAttributeIfExists(funcNode, "displayname");
                if (funcIdStatus == Error.Status.OK)
                {
                    functionBindings.Add(new FunctionBinding(ElementLocation(funcNode),
                        funcId.Item1, funcId.Item2,
                        displayName.Item1, displayName.Item2));
                }
                status = Error.Worse(status, funcIdStatus);
            }
        }

        #endregion
    }
}
