﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace EffectDefinitions
{
    #region Error description enums

    public enum ErrorSeverity
    {
        /// <summary>
        /// This could be an error, or this could be intentional.
        /// </summary>
        Warning,

        /// <summary>
        /// This is an error without a doubt.
        /// </summary>
        Error
    }

    public enum ErrorType
    {
        /// <summary>
        /// We can't even access the data.
        /// </summary>
        FileSystem,

        /// <summary>
        /// We can't read the data.
        /// </summary>
        Syntax,

        /// <summary>
        /// Some information is missing.
        /// </summary>
        MissingInformation,

        /// <summary>
        /// Information is not consistent.
        /// </summary>
        Structure
    }

    public enum DetailedErrorType
    {
        FileSystemPathNotFound,

        XMLException,
        XMLMissingTag,
        XMLUnknownTag,
        XMLDuplicateTag,
        XMLMissingInnerText,
        XMLMissingAttribute,
        XMLUnknownAttribute,

        DuplicateFunctionId,
        DuplicateBlockGuid,
        DuplicatePlugSemantic,
        FunctionNotFound,
        BlockHasDuplicateFunction,
        BlockIncompatibleWithFunction,
    }

    public static class Extensions
    {
        public static string ToString(this ErrorSeverity severity)
        {
            switch (severity)
            {
                case ErrorSeverity.Warning: return Messages.TEXT_WARNING;
                case ErrorSeverity.Error: return Messages.TEXT_ERROR;
                default:
                    throw new NotImplementedException();
            }
        }

        public static string ToString(this ErrorType type)
        {
            switch (type)
            {
                case ErrorType.Syntax: return Messages.TEXT_SYNTAX;
                case ErrorType.MissingInformation: return Messages.TEXT_MISSING_INFO;
                case ErrorType.Structure: return Messages.TEXT_STRUCTURE;
                default:
                    throw new NotImplementedException();
            }
        }

        public static string ToString(this DetailedErrorType detailedType)
        {
            switch (detailedType)
            {
                case DetailedErrorType.FileSystemPathNotFound: return Messages.TEXT_PATH_NOT_FOUND;
                case DetailedErrorType.XMLException: return Messages.TEXT_XML_EXCEPTION;
                case DetailedErrorType.XMLMissingTag: return Messages.TEXT_MISSING_XML_TAG;
                case DetailedErrorType.XMLUnknownTag: return Messages.TEXT_UNKNOWN_XML_TAG;
                case DetailedErrorType.XMLDuplicateTag: return Messages.TEXT_DUPLICATE_XML_TAG;
                case DetailedErrorType.XMLMissingInnerText: return Messages.TEXT_MISSING_INNER_TEXT;
                case DetailedErrorType.XMLMissingAttribute: return Messages.TEXT_MISSING_ATTRIBUTE;
                case DetailedErrorType.DuplicateFunctionId: return Messages.TEXT_DUPLICATE_FUNC_ID;
                case DetailedErrorType.DuplicateBlockGuid: return Messages.TEXT_DUPLICATE_BLOCK_ID;
                case DetailedErrorType.FunctionNotFound: return Messages.TEXT_FUNC_NOT_FOUND;
                case DetailedErrorType.BlockIncompatibleWithFunction: return Messages.TEXT_BLOCK_INCOMPATIBLE_WITH_FUNC;
                default:
                    throw new NotImplementedException();
            }
        }
    }

    #endregion

    public struct Error
    {
        internal enum Status
        {
            OK,
            Recoverable,
            Unrecoverable
        }

        internal static Status Worse(Status a, Status b)
        {
            return b > a ? b : a;
        }

        public Error(ErrorSeverity severity,
            ErrorType type,
            DetailedErrorType detailedType,
            string fileSystemPath,
            DefinitionLocation location,
            Exception exception,
            string xmlElementName,
            string xmlAttributeName,
            string functionId,
            FunctionDefinition functionDefinition,
            BlockDefinition blockDefinition,
            Func<string> messageFunc)
            : this()
        {
            Severity = severity;
            Type = type;
            DetailedType = detailedType;
            FileSystemPath = fileSystemPath;
            Location = location;
            Exception = exception;
            XMLElementName = xmlElementName;
            XMLAttributeName = xmlAttributeName;
            FunctionId = functionId;
            FunctionDefinition = functionDefinition;
            BlockDefinition = blockDefinition;
            this.messageFunc = messageFunc;
        }

        #region Helpers

        public static Error FileSystemPathNotFoundError(string fileSystemPath)
        {
            return new Error(ErrorSeverity.Error, ErrorType.FileSystem, DetailedErrorType.FileSystemPathNotFound,
                fileSystemPath, null, null, null, null, null, null, null,
                () => string.Format(Messages.FILE_DIRECTORY_NOT_FOUND, fileSystemPath));
        }

        public static Error XMLExceptionError(Uri uri, System.Xml.XmlException exception)
        {
            var location = new DefinitionLocation(uri, true, exception.LineNumber, exception.LinePosition, 1);

            return new Error(ErrorSeverity.Error, ErrorType.Syntax, DetailedErrorType.XMLException,
                null, location, exception, null, null, null, null, null, () => exception.Message);
        }

        public static Error XMLMissingTagError(DefinitionLocation parsingErrorLocation, string xmlTagName, string xmlMissingTagName)
        {
            return new Error(ErrorSeverity.Error, ErrorType.MissingInformation, DetailedErrorType.XMLMissingTag,
                null, parsingErrorLocation, null, xmlTagName, xmlMissingTagName, null, null, null,
                () => string.Format(Messages.XML_ELEMENT_IS_MISSING_CHILD_NODE, xmlTagName, xmlMissingTagName)); // TODO: verify message
        }

        public static Error XMLUnknownTagError(DefinitionLocation parsingErrorLocation, string xmlTagName)
        {
            return new Error(ErrorSeverity.Warning, ErrorType.Syntax, DetailedErrorType.XMLUnknownTag,
                null, parsingErrorLocation, null, xmlTagName, null, null, null, null,
                () => string.Format(Messages.XML_UNKNOWN_TAG, xmlTagName));
        }

        public static Error XMLDuplicateTagError(DefinitionLocation parsingErrorLocation, string xmlTagName)
        {
            return new Error(ErrorSeverity.Error, ErrorType.Syntax, DetailedErrorType.XMLDuplicateTag,
                null, parsingErrorLocation, null, xmlTagName, null, null, null, null,
                () => string.Format(Messages.XML_UNEXPECTED_TAG, xmlTagName));
        }

        public static Error XMLMissingInnerTextError(DefinitionLocation parsingErrorLocation, string xmlTagName)
        {
            return new Error(ErrorSeverity.Error, ErrorType.MissingInformation, DetailedErrorType.XMLMissingInnerText,
                null, parsingErrorLocation, null, xmlTagName, null, null, null, null,
                () => string.Format(Messages.XML_MISSING_INNER_TEXT, xmlTagName));
        }

        public static Error XMLUnknownAttributeError(DefinitionLocation parsingErrorLocation, string xmlTagName, string xmlAttributeName)
        {
            return new Error(ErrorSeverity.Warning, ErrorType.Syntax, DetailedErrorType.XMLUnknownAttribute,
                null, parsingErrorLocation, null, xmlTagName, xmlAttributeName, null, null, null,
                () => string.Format(Messages.XML_UNKNOWN_ATTRIBUTE, xmlAttributeName, xmlTagName));
        }

        public static Error XMLMissingAttributeError(DefinitionLocation parsingErrorLocation, string xmlTagName, string xmlAttributeName)
        {
            return new Error(ErrorSeverity.Error, ErrorType.MissingInformation, DetailedErrorType.XMLMissingAttribute,
                null, parsingErrorLocation, null, xmlTagName, xmlAttributeName, null, null, null,
                () => string.Format(Messages.XML_MISSING_ATTRIBUTE, xmlTagName, xmlAttributeName));
        }

        public static Error DuplicateFunctionIdError(FunctionDefinition functionDefinition)
        {
            var location = functionDefinition is FunctionDefinitionWithSource ?
                ((FunctionDefinitionWithSource)functionDefinition).IdSource : functionDefinition.Source;

            return new Error(ErrorSeverity.Error, ErrorType.Structure, DetailedErrorType.DuplicateFunctionId,
                null, location, null, null, null, null, functionDefinition, null,
                () => string.Format(Messages.DUPLICATE_FUNC_ID, functionDefinition.Id));
        }

        public static Error DuplicateBlockGuidError(BlockDefinition blockDefinition)
        {
            var location = blockDefinition is BlockDefinitionWithSource ?
                ((BlockDefinitionWithSource)blockDefinition).GuidSource : blockDefinition.Source;

            return new Error(ErrorSeverity.Error, ErrorType.Structure, DetailedErrorType.DuplicateBlockGuid,
                null, location, null, null, null, null, null, blockDefinition,
                () => string.Format(Messages.DUPLICATE_BLOCK_ID, blockDefinition.Name, blockDefinition.Guid));
        }

        public static Error DuplicatePlugSemanticError(BlockDefinition blockDefinition, Plug plug)
        {
            var location = plug is PlugWithSource ?
                ((PlugWithSource)plug).SemanticSource : blockDefinition.Source;

            return new Error(ErrorSeverity.Error, ErrorType.Structure, DetailedErrorType.DuplicatePlugSemantic,
                null, location, null, null, null, null, null, blockDefinition,
                () => string.Format(Messages.DUPLICATE_PLUG_SEMANTIC, plug.Semantic, blockDefinition.Name, blockDefinition.Guid));
        }

        public static Error FunctionNotFoundError(BlockDefinition blockDefinition, FunctionBinding binding)
        {
            return new Error(ErrorSeverity.Error, ErrorType.Structure, DetailedErrorType.FunctionNotFound,
                null,  binding.IdSource, null, null, null, binding.Id, null, blockDefinition,
                () => string.Format(Messages.MISSING_FUNCTION, blockDefinition.Name, binding.Id));
        }

        public static Error BlockHasDuplicateFunctionIdError(BlockDefinitionWithSource blockDefinition, FunctionBinding binding)
        {
            return new Error(ErrorSeverity.Warning, ErrorType.Structure, DetailedErrorType.BlockHasDuplicateFunction,
                null, binding.IdSource, null, null, null, binding.Id, null, blockDefinition,
                () => string.Format(Messages.BLOCK_ALREADY_REFERENCING_FUNC, blockDefinition.Name, binding.Id));
        }

        public static Error BlockIncompatibleWithFunctionError(BlockDefinition blockDefinition, FunctionBinding binding, FunctionDefinition functionDefinition)
        {
            return new Error(ErrorSeverity.Error, ErrorType.Structure, DetailedErrorType.BlockIncompatibleWithFunction,
                null, binding.IdSource, null, null, null, null, functionDefinition, blockDefinition,
                () => string.Format(Messages.INCOMPATIBLE_BLOCK_AND_FUNC, blockDefinition.Name, functionDefinition.Id));
        }

        #endregion

        public ErrorSeverity Severity { get; private set; }
        public ErrorType Type { get; private set; }
        public DetailedErrorType DetailedType { get; private set; }

        private Func<string> messageFunc;
        public string Message { get { return messageFunc != null ? messageFunc() : string.Empty; } }

        public string FileSystemPath { get; private set; }
        public DefinitionLocation Location { get; private set; }

        public Exception Exception { get; private set; }
        public string XMLElementName { get; private set; }
        public string XMLAttributeName { get; private set; }

        public string FunctionId { get; private set; }
        public FunctionDefinition FunctionDefinition { get; private set; }
        public BlockDefinition BlockDefinition { get; private set; }

        public override string ToString()
        {
            return string.Format("{0}:{1}:{2}: {3}", Severity, Type, DetailedType, Message);
        }
    }

    public interface IErrorReporter
    {
        IEnumerable<Error> Errors { get; }

        void Report(Error error);
        void ClearErrors();
    }
}
