﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectCombiner.Core;

namespace EffectDefinitions
{
    #region Event args

    public class ErrorEventArgs : EventArgs
    {
        public ErrorEventArgs(IEnumerable<Error> errors)
        {
            Errors = errors;
        }

        public IEnumerable<Error> Errors { get; set; }
    }

    public class FileIOEventArgs : EventArgs
    {
        public FileIOEventArgs(string path)
        {
            Path = path;
        }

        public string Path { get; set; }
    }

    public class DefinitionsPathEventArgs : EventArgs
    {
        public DefinitionsPathEventArgs(IEnumerable<PathLookupInfo> pathsLoadInfo)
        {
            PathsLoadInfo = pathsLoadInfo;
        }

        public IEnumerable<PathLookupInfo> PathsLoadInfo { get; set; }
    }

    public class FunctionDefinitionEventArgs : EventArgs
    {
        public FunctionDefinitionEventArgs(IEnumerable<FunctionDefinition> functions)
        {
            Functions = functions;
        }

        public IEnumerable<FunctionDefinition> Functions { get; set; }
    }

    public class BlockDefinitionEventArgs : EventArgs
    {
        public BlockDefinitionEventArgs(IEnumerable<BlockDefinition> blocks)
        {
            Blocks = blocks;
        }

        public IEnumerable<BlockDefinition> Blocks { get; set; }
    }

    public class TaggingEventArgs : EventArgs
    {
        public TaggingEventArgs(IEnumerable<Tagging> taggings)
        {
            Taggings = taggings;
        }

        public IEnumerable<Tagging> Taggings { get; set; }
    }

    public class SourcesResolvedEventArgs : EventArgs
    {
        public string[] Sources { get; private set; }

        public SourcesResolvedEventArgs(params string[] sources)
        {
            Sources = sources;
        }
    }

    public class BlockBindingEventArgs : EventArgs
    {
        public BlockBindingEventArgs(IEnumerable<BlockDefinition> blocks)
        {
            Blocks = blocks;
        }

        public IEnumerable<BlockDefinition> Blocks { get; set; }
    }

    #endregion

    public partial class EffectDefinitionsContainer
    {
        public event EventHandler NewErrorsReported;
        protected virtual void OnNewErrors(IEnumerable<Error> errors)
        {
            var handler = NewErrorsReported;
            if (handler != null)
            {
                handler(this, new ErrorEventArgs(errors));
            }
        }

        #region File IO related events

        public event EventHandler<SourcesResolvedEventArgs> SourcesResolved;
        protected virtual void OnSourcesResolved(SourcesResolvedEventArgs e)
        {
            var handler = SourcesResolved;
            if (handler != null)
                handler(this, e);
        }

        public event EventHandler<FileIOEventArgs> DirectoryExploringStarted;
        protected virtual void OnDirectoryExploringStarted(string dirPath)
        {
            var handler = DirectoryExploringStarted;
            if (handler != null)
            {
                handler(this, new FileIOEventArgs(dirPath));
            }
        }

        public event EventHandler<FileIOEventArgs> DirectoryExploringFinished;
        protected virtual void OnDirectoryExploringFinished(string dirPath)
        {
            var handler = DirectoryExploringFinished;
            if (handler != null)
            {
                handler(this, new FileIOEventArgs(dirPath));
            }
        }

        public event EventHandler<FileIOEventArgs> FileLoadingStarted;
        protected virtual void OnFileLoadingStarted(string filePath)
        {
            var handler = FileLoadingStarted;
            if (handler != null)
            {
                handler(this, new FileIOEventArgs(filePath));
            }
        }

        public event EventHandler<FileIOEventArgs> FileLoadingFinished;
        protected virtual void OnFileLoadingFinished(string filePath)
        {
            var handler = FileLoadingFinished;
            if (handler != null)
            {
                handler(this, new FileIOEventArgs(filePath));
            }
        }

        #endregion

        #region Definitions paths related events

        public event EventHandler<DefinitionsPathEventArgs> DefinitionsPathsAdded;
        protected virtual void OnDefinitionsPathsAdded(IEnumerable<PathLookupInfo> pathsLoadInfo)
        {
            var handler = DefinitionsPathsAdded;
            if (handler != null && pathsLoadInfo.Any())
            {
                handler(this, new DefinitionsPathEventArgs(pathsLoadInfo));
            }
        }

        public event EventHandler<DefinitionsPathEventArgs> DefinitionsPathsRemoved;
        protected virtual void OnDefinitionsPathsRemoved(IEnumerable<PathLookupInfo> pathsLoadInfo)
        {
            var handler = DefinitionsPathsRemoved;
            if (handler != null && pathsLoadInfo.Any())
            {
                handler(this, new DefinitionsPathEventArgs(pathsLoadInfo));
            }
        }

        #endregion

        #region Function pathsLoadInfo related events

        public event EventHandler<FunctionDefinitionEventArgs> FunctionDefinitionsAdded;
        protected virtual void OnFunctionDefinitionsAdded(IEnumerable<FunctionDefinition> definitions)
        {
            var handler = FunctionDefinitionsAdded;
            if (handler != null && definitions.Any())
            {
                handler(this, new FunctionDefinitionEventArgs(definitions));
            }
        }

        public event EventHandler<FunctionDefinitionEventArgs> FunctionDefinitionsUpdated;
        protected virtual void OnFunctionDefinitionsUpdated(IEnumerable<FunctionDefinition> definitions)
        {
            var handler = FunctionDefinitionsUpdated;
            if (handler != null && definitions.Any())
            {
                handler(this, new FunctionDefinitionEventArgs(definitions));
            }
        }

        public event EventHandler<FunctionDefinitionEventArgs> FunctionDefinitionsRemoved;
        protected virtual void OnFunctionDefinitionsRemoved(IEnumerable<FunctionDefinition> definitions)
        {
            var handler = FunctionDefinitionsRemoved;
            if (handler != null && definitions.Any())
            {
                handler(this, new FunctionDefinitionEventArgs(definitions));
            }
        }

        #endregion

        #region Block pathsLoadInfo related events

        public event EventHandler<BlockDefinitionEventArgs> BlockDefinitionsAdded;
        protected virtual void OnBlockDefinitionsAdded(IEnumerable<BlockDefinition> definitions)
        {
            var handler = BlockDefinitionsAdded;
            if (handler != null && definitions.Any())
            {
                handler(this, new BlockDefinitionEventArgs(definitions));
            }
        }

        public event EventHandler<BlockDefinitionEventArgs> BlockDefinitionsUpdated;
        protected virtual void OnBlockDefinitionsUpdated(IEnumerable<BlockDefinition> definitions)
        {
            var handler = BlockDefinitionsUpdated;
            if (handler != null && definitions.Any())
            {
                handler(this, new BlockDefinitionEventArgs(definitions));
            }
        }

        public event EventHandler<BlockDefinitionEventArgs> BlockDefinitionsRemoved;
        protected virtual void OnBlockDefinitionsRemoved(IEnumerable<BlockDefinition> definitions)
        {
            var handler = BlockDefinitionsRemoved;
            if (handler != null && definitions.Any())
            {
                handler(this, new BlockDefinitionEventArgs(definitions));
            }
        }

        #endregion

        #region Taggings related events

        public event EventHandler<TaggingEventArgs> TaggingsAdded;
        protected virtual void OnTaggingsAdded(IEnumerable<Tagging> taggings)
        {
            var handler = TaggingsAdded;
            if (handler != null && taggings.Any())
            {
                handler(this, new TaggingEventArgs(taggings));
            }
        }

        public event EventHandler<TaggingEventArgs> TaggingsRemoved;
        protected virtual void OnTaggingsRemoved(IEnumerable<Tagging> taggings)
        {
            var handler = TaggingsRemoved;
            if (handler != null && taggings.Any())
            {
                handler(this, new TaggingEventArgs(taggings));
            }
        }

        #endregion

        public event EventHandler<BlockBindingEventArgs> BindingsUpdated;
        protected virtual void OnBindingsUpdated(IEnumerable<BlockDefinition> definitions)
        {
            var handler = BindingsUpdated;
            if (handler != null && definitions.Any())
            {
                handler(this, new BlockBindingEventArgs(definitions));
            }
        }
    }
}
