﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Collections.ObjectModel;
using System.Text.RegularExpressions;
using System.Threading;
using EffectCombiner.Core;

namespace EffectDefinitions
{
    /// <summary>
    /// Holds all the loaded definitions, and provides loading mechanisms.
    /// </summary>
    public partial class EffectDefinitionsContainer
    {
        public EffectDefinitionsContainer()
        {
            Errors = new ReadOnlyCollection<Error>(errors);
            Taggings = new ReadOnlyCollection<TaggingWithSource>(taggings);
        }

        public void SetDefinitionsPaths(CancellationToken cancellationToken, params string[] paths)
        {
            SetDefinitionsPaths(cancellationToken, paths.Select(x => new PathLookupInfo(x)).ToArray());
        }

        public void SetDefinitionsPaths(CancellationToken cancellationToken, params PathLookupInfo[] pathsLoadInfo)
        {
            var localErrorContainer = new ErrorContainer(errors);
            LoadDefinitionsFromPaths(cancellationToken, localErrorContainer, pathsLoadInfo, true);
            ReportErrorsIfAny(localErrorContainer);
        }

        public void ReloadDefinitions(CancellationToken cancellationToken)
        {
            var localErrorContainer = new ErrorContainer(errors);
            LoadDefinitionsFromPaths(cancellationToken, localErrorContainer, definitionsPaths, false);
            ReportErrorsIfAny(localErrorContainer);
        }

        public void ClearDefinitions()
        {
            definitionsPaths.Clear();
            functions.Clear();
            blocks.Clear();
            taggings.Clear();
        }

        #region Adding definitions from runtime

        public void AddDefinitionsFromXmlChunk(string xmlString, Uri uri)
        {
            var localErrorContainer = new ErrorContainer(errors);
            var loadedDefinitions = new DefinitionsGroup();

            loadedDefinitions.AddDefinitions(GetDefinitionsFromXmlChunk(localErrorContainer, xmlString, uri));

            if (loadedDefinitions.CheckUniqueness(localErrorContainer))
            {
                InternalAddOrUpdateFunctionDefinitions(localErrorContainer, loadedDefinitions.FunctionDefinitions, true, true);
                InternalAddOrUpdateTaggings(loadedDefinitions.Taggings, true, true);
                InternalAddOrUpdateBlockDefinitions(localErrorContainer, loadedDefinitions.BlockDefinitions, true, true);
            }

            ReportErrorsIfAny(localErrorContainer);
        }

        public void AddFunctionDefinitions(params FunctionDefinition[] definitions)
        {
            var localErrorContainer = new ErrorContainer(errors);

            if (DefinitionsGroup.CheckFunctionsIdUniqueness(definitions, localErrorContainer))
            {
                var definitionsWithSource = definitions.Select(x =>
                    x is FunctionDefinitionWithSource ?
                    x as FunctionDefinitionWithSource :
                    x);
                InternalAddOrUpdateFunctionDefinitions(localErrorContainer, definitionsWithSource, false, true);
            }
            ReportErrorsIfAny(localErrorContainer);
        }

        public void AddBlockDefinitions(params BlockDefinition[] definitions)
        {
            var localErrorContainer = new ErrorContainer(errors);

            if (DefinitionsGroup.CheckBlocksGuidUniqueness(definitions, localErrorContainer))
            {
                var definitionsWithSource = definitions.Select(x =>
                    x is BlockDefinitionWithSource ?
                    x as BlockDefinitionWithSource :
                    x);
                InternalAddOrUpdateBlockDefinitions(localErrorContainer, definitionsWithSource, false, true);
            }
            ReportErrorsIfAny(localErrorContainer);
        }

        public void AddTaggings(params Tagging[] newTaggings)
        {
            var taggingsWithSource = newTaggings.Select(x =>
                x is TaggingWithSource ?
                x as TaggingWithSource :
                new TaggingWithSource(x));
            InternalAddOrUpdateTaggings(taggingsWithSource, false, true);
        }

        #endregion

        #region Loading from file resources

        private void LoadDefinitionsFromPaths(CancellationToken cancellationToken, ErrorContainer errorContainer, IEnumerable<PathLookupInfo> pathsLoadInfo, bool reportMissingPath)
        {
            var filePaths = pathsLoadInfo
                .SelectMany(x => GetFilesFromPath(errorContainer, x, reportMissingPath))
                .Distinct()
                .ToArray();

            OnSourcesResolved(new SourcesResolvedEventArgs(filePaths));

            var loadedDefinitions = GetDefinitionsFromFiles(cancellationToken, errorContainer, filePaths);
            if (loadedDefinitions.CheckUniqueness(errorContainer))
            {
                InternalUpdateDefinitionsPaths(errorContainer, pathsLoadInfo);
                InternalAddOrUpdateFunctionDefinitions(errorContainer, loadedDefinitions.FunctionDefinitions, true, false);
                InternalAddOrUpdateTaggings(loadedDefinitions.Taggings, true, false);
                InternalAddOrUpdateBlockDefinitions(errorContainer, loadedDefinitions.BlockDefinitions, true, false);
            }
        }

        private IEnumerable<string> GetFilesFromPath(ErrorContainer errorContainer, PathLookupInfo sourceLoadInfo, bool reportMissingPath)
        {
            string path = sourceLoadInfo.Path;

            if (Directory.Exists(path))
            {
                OnDirectoryExploringStarted(path);
                var option = sourceLoadInfo.IncludeSubDirectories ? SearchOption.AllDirectories : SearchOption.TopDirectoryOnly;

                foreach (var pattern in sourceLoadInfo.SearchPatterns)
                    foreach (string filePath in Directory.GetFiles(path, pattern, option))
                    {
                        yield return Path.GetFullPath(filePath);
                    }

                OnDirectoryExploringFinished(path);
            }
            else if (File.Exists(path))
            {
                yield return path;
            }
            else if (reportMissingPath)
            {
                errorContainer.Report(Error.FileSystemPathNotFoundError(path));
            }
        }

        #endregion

        #region Extracting definitions from resources

        protected virtual string GetRawContent(string filePath, Encoding encoding)
        {
            using (var reader = new StreamReader(filePath, encoding))
                return reader.ReadToEnd();
        }

        private DefinitionsGroup GetDefinitionsFromFiles(CancellationToken cancellationToken, ErrorContainer errorContainer, IEnumerable<string> filePaths)
        {
            DefinitionsGroup definitions = new DefinitionsGroup();

            foreach (string filePath in filePaths)
            {
                if (cancellationToken.IsCancellationRequested)
                {
                    return new DefinitionsGroup();
                }

                OnFileLoadingStarted(filePath);
                var uri = new Uri(filePath);
                var content = GetRawContent(filePath, Encoding.UTF8);
                var fileDefinitions = (Utility.IsEDMLFileName(filePath) ?
                    DefinitionsGroup.ExtractFromXML(errorContainer, content, uri) :
                    DefinitionsGroup.ExtractFromCommentedShader(errorContainer, content, uri));
                OnFileLoadingFinished(filePath);

                definitions.AddDefinitions(fileDefinitions);
            }

            return definitions;
        }

        private DefinitionsGroup GetDefinitionsFromXmlChunk(ErrorContainer errorContainer, string xmlString, Uri uri)
        {
            var content = "<root>\r\n" + xmlString + "\r\n</root>";
            var definitions = DefinitionsGroup.ExtractFromXML(errorContainer, content, uri);

            return definitions;
        }

        #endregion

        #region Add/remove uniqueDefinitions implementation details

        private void InternalUpdateDefinitionsPaths(ErrorContainer errorContainer, IEnumerable<PathLookupInfo> pathsLoadInfo)
        {
            PathLookupInfo[] pathsToAdd = pathsLoadInfo.Except(definitionsPaths).ToArray();
            PathLookupInfo[] pathsToRemove = definitionsPaths.Except(pathsLoadInfo).ToArray();

            definitionsPaths = pathsLoadInfo.ToList();

            OnDefinitionsPathsAdded(pathsToAdd);
            OnDefinitionsPathsRemoved(pathsToRemove);
        }

        private void InternalAddOrUpdateFunctionDefinitions(ErrorContainer errorContainer, IEnumerable<FunctionDefinition> uniqueDefinitions, bool updateMode, bool adHoc)
        {
            foreach (var definition in uniqueDefinitions)
            {
                definition.SetTagDelegate(TagsByFunction);
            }

            FunctionDefinition[] definitionsToAdd = uniqueDefinitions
                .Where(x => functions.ContainsKey(x.Id) == false).ToArray();

            FunctionDefinition[] alreadyExistingDefinitions = uniqueDefinitions
                .Where(x => functions.ContainsKey(x.Id) == true).ToArray();

            FunctionDefinition[] definitionsToUpdate = alreadyExistingDefinitions
                .Where(x => functions[x.Id] != x).ToArray();

            FunctionDefinition[] definitionsToRemove = functions.Keys
                .Except(uniqueDefinitions.Select(x => x.Id))
                .Except(adHocFunctions.Select(x => x.Id))
                .Select(k => functions[k])
                .ToArray();

            if (updateMode)
            {
                foreach (var definition in definitionsToUpdate)
                {
                    functions[definition.Id] = definition;
                }

                foreach (var definition in definitionsToRemove)
                {
                    functions.Remove(definition.Id);
                }
            }
            else
            {
                foreach (var duplicate in alreadyExistingDefinitions)
                {
                    var original = functions[duplicate.Id];
                    errorContainer.Report(Error.DuplicateFunctionIdError(original));
                    errorContainer.Report(Error.DuplicateFunctionIdError(duplicate));
                }
            }

            foreach (var definition in definitionsToAdd)
            {
                functions[definition.Id] = definition;
            }

            if (adHoc)
            {
                adHocFunctions.AddRange(definitionsToAdd);
            }

            OnFunctionDefinitionsAdded(definitionsToAdd);
            if (updateMode)
            {
                OnFunctionDefinitionsUpdated(definitionsToUpdate);
                OnFunctionDefinitionsRemoved(definitionsToRemove);
            }
        }

        private void InternalAddOrUpdateBlockDefinitions(ErrorContainer errorContainer, IEnumerable<BlockDefinition> uniqueDefinitions, bool updateMode, bool adHoc)
        {
            BlockDefinition[] definitionsToAdd = uniqueDefinitions
                .Where(x => blocks.ContainsKey(x.Guid) == false).ToArray();

            BlockDefinition[] alreadyExistingDefinitions = uniqueDefinitions
                .Where(x => blocks.ContainsKey(x.Guid) == true).ToArray();

            BlockDefinition[] definitionsToUpdate = alreadyExistingDefinitions
                .Where(x => blocks[x.Guid] != x).ToArray();

            BlockDefinition[] definitionsToRemove = blocks.Keys
                .Except(uniqueDefinitions.Select(x => x.Guid))
                .Except(adHocBlocks.Select(x => x.Guid))
                .Select(k => blocks[k])
                .ToArray();

            if (updateMode)
            {
                foreach (var definition in definitionsToUpdate)
                {
                    blocks[definition.Guid] = definition;
                }

                foreach (var definition in definitionsToRemove)
                {
                    blocks.Remove(definition.Guid);
                }
            }
            else
            {
                foreach (var duplicate in alreadyExistingDefinitions)
                {
                    var original = blocks[duplicate.Guid];
                    errorContainer.Report(Error.DuplicateBlockGuidError(original));
                    errorContainer.Report(Error.DuplicateBlockGuidError(duplicate));
                }
            }

            foreach (var definition in definitionsToAdd)
            {
                blocks[definition.Guid] = definition;
            }

            if (adHoc)
            {
                adHocBlocks.AddRange(definitionsToAdd);
            }

            OnBlockDefinitionsAdded(definitionsToAdd);
            if (updateMode)
            {
                OnBlockDefinitionsUpdated(definitionsToUpdate);
                OnBlockDefinitionsRemoved(definitionsToRemove);
            }
        }

        private void InternalAddOrUpdateTaggings(IEnumerable<TaggingWithSource> newTaggings, bool updateMode, bool adHoc)
        {
            TaggingWithSource[] taggingsToAdd = newTaggings
                .Except(taggings).ToArray();

            TaggingWithSource[] taggingsToRemove = taggings
                .Except(newTaggings)
                .Except(adHocTaggings)
                .ToArray();

            taggings.AddRange(taggingsToAdd);
            if (updateMode)
                foreach (var tagging in taggingsToRemove)
                {
                    taggings.Remove(tagging);
                }

            if (adHoc)
            {
                adHocTaggings.AddRange(taggingsToAdd);
            }

            OnTaggingsAdded(taggingsToAdd);
            if (updateMode)
            {
                OnTaggingsRemoved(taggingsToRemove);
            }
        }

        #endregion

        #region Bindings accross data

        public void UpdateBindings()
        {
            var localErrorContainer = new ErrorContainer(errors);
            BindFunctionsToBlocks(localErrorContainer);
            ReportErrorsIfAny(localErrorContainer);
        }

        private void BindFunctionsToBlocks(ErrorContainer errorContainer)
        {
            var updatedBlocks = new List<BlockDefinition>();

            foreach (var block in blocks.Values)
            {
                var oldBindings = block.FunctionBindings
                    .Where(x => x.Definition != null)
                    .Select(x => x.Id)
                    .ToArray();

                foreach (var binding in block.FunctionBindings)
                {
                    binding.Definition = null;

                    FunctionDefinition functionDefinition;
                    if (functions.TryGetValue(binding.Id, out functionDefinition))
                    {
                        if (block.BindToFunctionDefinition(functionDefinition) == false)
                        {
                            errorContainer.Report(Error.BlockIncompatibleWithFunctionError(block, binding, functionDefinition));
                        }
                    }
                    else
                    {
                        errorContainer.Report(Error.FunctionNotFoundError(block, binding));
                    }
                }

                var newBindings = block.FunctionBindings
                    .Where(x => x.Definition != null)
                    .Select(x => x.Id)
                    .ToArray();

                var difference = newBindings.Union(oldBindings).Except(newBindings.Intersect(oldBindings));
                if (difference.Any())
                    updatedBlocks.Add(block);
            }
            OnBindingsUpdated(updatedBlocks);
        }

        #endregion

        #region Errors

        public IEnumerable<Error> Errors { get; private set; }
        private readonly List<Error> errors = new List<Error>();

        public void ClearErrors()
        {
            errors.Clear();
        }

        private void ReportErrorsIfAny(ErrorContainer errorContainer)
        {
            if (errorContainer.Errors.Any())
            {
                OnNewErrors(errorContainer.Errors);
            }
        }

        #endregion

        #region Actual data

        #region Helpers

        public IEnumerable<FunctionDefinition> Functions { get { return functions.Values; } }
        public IEnumerable<BlockDefinition> Blocks { get { return blocks.Values; } }
        public IEnumerable<Tagging> Taggings { get; private set; }

        public IEnumerable<string> Tags
        {
            get { return taggings.Select(x => x.Name).Distinct().ToArray(); }
        }

        public IEnumerable<FunctionDefinition> FunctionsByTag(string tag)
        {
            return Functions.Where(x => x.Tags.Contains(tag));
        }

        public IEnumerable<string> TagsByFunction(FunctionDefinition function)
        {
            return taggings
                .Where(x => x.FunctionId == function.Id)
                .Select(x => x.Name)
                .Distinct();
        }

        public IEnumerable<BlockDefinition> BlocksByTag(string tag)
        {
            return Blocks.Where(x => x.Tags.Contains(tag));
        }

        public IEnumerable<BlockDefinition> BlocksByPlugs(BlockDefinition blockDef)
        {
            return Blocks.Where(x => x.HasSamePlugsTypesAs(blockDef));
        }

        public IEnumerable<BlockDefinition> BlocksByPlugsAndTag(BlockDefinition blockDef, string tag)
        {
            return Blocks.Where(x => x.Tags.Contains(tag) && x.HasSamePlugsTypesAs(blockDef));
        }

        public IEnumerable<BlockDefinition> BlocksByFunction(FunctionDefinition definition)
        {
            return Blocks.Where(x => x.FunctionBindings.Any(a => a.Definition == definition));
        }

        public IEnumerable<BlockDefinition> BlocksByFunctionId(string functionId)
        {
            return Blocks.Where(x => x.FunctionBindings.Any(a => a.Id == functionId));
        }

        #endregion

        private List<PathLookupInfo> definitionsPaths = new List<PathLookupInfo>();

        private Dictionary<string, FunctionDefinition> functions = new Dictionary<string, FunctionDefinition>();
        private Dictionary<string, BlockDefinition> blocks = new Dictionary<string, BlockDefinition>();
        private List<TaggingWithSource> taggings = new List<TaggingWithSource>();

        private List<FunctionDefinition> adHocFunctions = new List<FunctionDefinition>();
        private List<BlockDefinition> adHocBlocks = new List<BlockDefinition>();
        private List<TaggingWithSource> adHocTaggings = new List<TaggingWithSource>();

        #endregion
    }
}
