﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using EffectCombiner.Core.Extensions;

namespace EffectDefinitions
{
    /// <summary>
    /// Set of function definitions, block definitions and taggings.
    /// </summary>
    public class DefinitionsGroup
    {
        internal static DefinitionsGroup ExtractFromXML(ErrorContainer errorContainer, string content, Uri uri)
        {
            try
            {
                using (var reader = new PureXmlDefinitionReader(errorContainer, content, uri))
                    return reader.ProvideDefinitions();
            }
            catch (System.Xml.XmlException e)
            {
                errorContainer.Report(Error.XMLExceptionError(uri, e));
            }
            return null;
        }

        internal static DefinitionsGroup ExtractFromCommentedShader(ErrorContainer errorContainer, string content, Uri uri)
        {
            try
            {
                using (var reader = new CommentedShaderDefinitionReader(errorContainer, content, uri))
                    return reader.ProvideDefinitions();
            }
            catch (System.Xml.XmlException e)
            {
                errorContainer.Report(Error.XMLExceptionError(uri, e));
            }
            return null;
        }

        public DefinitionsGroup()
        {
            FunctionDefinitions = new List<FunctionDefinitionWithSource>();
            BlockDefinitions = new List<BlockDefinitionWithSource>();
            Taggings = new List<TaggingWithSource>();
        }

        public DefinitionsGroup(IEnumerable<FunctionDefinitionWithSource> functionDefinitions,
                                IEnumerable<BlockDefinitionWithSource> blockDefinitions,
                                IEnumerable<TaggingWithSource> taggings)
        {
            FunctionDefinitions = functionDefinitions.ToList();
            BlockDefinitions = blockDefinitions.ToList();
            Taggings = taggings.ToList();
        }

        public List<FunctionDefinitionWithSource> FunctionDefinitions { get; private set; }
        public List<BlockDefinitionWithSource> BlockDefinitions { get; private set; }
        public List<TaggingWithSource> Taggings { get; private set; }

        public void AddDefinitions(DefinitionsGroup additionalDefinitions)
        {
            if (additionalDefinitions == null)
                return;

            FunctionDefinitions.AddRange(additionalDefinitions.FunctionDefinitions);
            BlockDefinitions.AddRange(additionalDefinitions.BlockDefinitions);
            Taggings.AddRange(additionalDefinitions.Taggings);
        }

        #region Uniqueness

        private static IEnumerable<IGrouping<string, FunctionDefinition>> Duplicates(IEnumerable<FunctionDefinition> sequence)
        {
            return sequence.GroupBy(x => x.Id).Where(x => x.MoreThanOne());
        }

        private static IEnumerable<IGrouping<string, BlockDefinition>> Duplicates(IEnumerable<BlockDefinition> sequence)
        {
            return sequence.GroupBy(x => x.Guid).Where(x => x.MoreThanOne());
        }

        private static IEnumerable<IGrouping<string, Tuple<BlockDefinition, Plug>>> SemanticDuplicates(IEnumerable<Tuple<BlockDefinition, Plug>> sequence)
        {
            return sequence.GroupBy(x => x.Item2.Semantic).Where(x => x.MoreThanOne());
        }

        public static bool CheckFunctionsIdUniqueness(IEnumerable<FunctionDefinition> definitions, IErrorReporter errorReporter)
        {
            bool isOk = true;
            var duplicatesById = Duplicates(definitions);
            foreach (var duplicates in duplicatesById)
                foreach (var duplicate in duplicates)
                {
                    errorReporter.Report(Error.DuplicateFunctionIdError(duplicate));
                    isOk = false;
                }
            return isOk;
        }

        public static bool CheckBlocksGuidUniqueness(IEnumerable<BlockDefinition> definitions, IErrorReporter errorReporter)
        {
            bool isOk = true;
            var duplicatesByGuid = Duplicates(definitions);
            foreach (var duplicates in duplicatesByGuid)
                foreach (var duplicate in duplicates)
                {
                    errorReporter.Report(Error.DuplicateBlockGuidError(duplicate));
                    isOk = false;
                }
            return isOk;
        }

        public static bool CheckPlugsSemanticsUniqueness(IEnumerable<BlockDefinition> definitions, IErrorReporter errorReporter)
        {
            bool isOk = true;
            var outputSemanticPlugs = definitions
                .SelectMany(x => x.OutputPlugs.Select(y => Tuple.Create(x, y)))
                .Where(x => string.IsNullOrEmpty(x.Item2.Semantic) == false);
            var duplicatesBySemantic = SemanticDuplicates(outputSemanticPlugs);
            foreach (var duplicates in duplicatesBySemantic)
                foreach (var duplicate in duplicates)
                {
                    errorReporter.Report(Error.DuplicatePlugSemanticError(duplicate.Item1, duplicate.Item2));
                    isOk = false;
                }
            return isOk;
        }

        public bool CheckUniqueness(IErrorReporter errorReporter)
        {
            bool functionsOk = CheckFunctionsIdUniqueness(FunctionDefinitions, errorReporter);
            bool blocksOk = CheckBlocksGuidUniqueness(BlockDefinitions, errorReporter);
            bool plugsOk = CheckPlugsSemanticsUniqueness(BlockDefinitions, errorReporter);

            return functionsOk && blocksOk;
        }

        #endregion
    }
}
