﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml;
using System.Text.RegularExpressions;

namespace EffectDefinitions
{
    public class CommentedShaderDefinitionReader : IDefinitionProvider
    {
        private static readonly Regex commentRegex =
            new Regex(@"((?<linespaces>[ \t]*)/\*\*(?<msg>([^*]|[\r\n]|(\*+([^*/]|[\r\n])))*\**)\*/)|((?<linespaces>[ \r\t]*)///(?<msg>[^\r\n]*)|(([^\r\n]*)(?<newline>\r\n)))",
                RegexOptions.Compiled | RegexOptions.Multiline);

        private IErrorReporter errorReporter;
        private readonly PureXmlDefinitionReader xmlReader;

        public CommentedShaderDefinitionReader(IErrorReporter errorReporter, string data, Uri uri)
        {
            this.errorReporter = errorReporter;

            data = ExtractXmlFromComments(data);
            xmlReader = new PureXmlDefinitionReader(errorReporter, "<root>\r\n" + data + "\r\n</root>", uri, 1);
        }

        public void Dispose() { }

        public DefinitionsGroup ProvideDefinitions()
        {
            return xmlReader.ProvideDefinitions();
        }

        private static string ExtractXmlFromComments(string data)
        {
            if (data == null)
                throw new ArgumentNullException("data");

            var matches = commentRegex.Matches(data);

            string strippedData = string.Empty;
            foreach (Match match in matches)
            {
                var spaces = match.Groups["linespaces"];
                if (spaces != null && string.IsNullOrEmpty(spaces.Value) == false)
                {
                    strippedData += spaces.Value;
                }

                var content = match.Groups["msg"];
                if (content != null && string.IsNullOrEmpty(content.Value) == false)
                {
                    strippedData += "   " + content.Value;
                }

                var newline = match.Groups["newline"];
                if (newline != null && string.IsNullOrEmpty(newline.Value) == false)
                {
                    strippedData += newline.Value;
                }
            }

            return strippedData;
        }
    }
}
