﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Security.AccessControl;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
//using EffectMaker.Foundation.Log;

namespace EffectCombiner.Core
{
    /// <summary>
    /// PathUtility.
    /// </summary>
    public static class PathUtility
    {
        /// <summary>
        /// Get the directory name of the given path without the ending "\" and "/".
        /// The given path must be an absolute path and must not be empty.
        /// - For a file path, System.IO.Path.GetDirectoryName() returns correct path.
        /// - For a dir path with ending "/" or "\", System.IO.Path.GetDirectoryName()
        ///   returns correct path.
        /// - For a dir path WITHOUT ending "/" or "\", System.IO.Path.GetDirectoryName()
        ///   returns the parent directory path.
        /// So we need a method which always returns the current directory without "/" and "\".
        /// </summary>
        /// <param name="path">The file or directory path.</param>
        /// <returns>The directory path without ending "\" and "/".</returns>
        public static string GetDirectoryName(string path)
        {
            // Validate the given path first ( must not be empty, and must be absolute path ).
            if (string.IsNullOrEmpty(path) == true)
            {
                return string.Empty;
            }

            if (Path.IsPathRooted(path) == false)
            {
                return string.Empty;
            }

            string dirPath;

            // Is the given path a directory?
            // We want the directory path without the ending "\" and "/"
            if (string.IsNullOrEmpty(Path.GetExtension(path)) == true)
            {
                // It is a directory, remove the ending "\" and "/"
                dirPath = path.TrimEnd('\\', '/');
            }
            else
            {
                // It is a file path, just get the directory name.
                dirPath = Path.GetDirectoryName(path);
            }

            return Path.GetFullPath(dirPath);
        }

        /// <summary>
        /// True if the path is both set and absolute
        /// </summary>
        /// <param name="path">Path</param>
        /// <returns>True if both set and absolute</returns>
        public static bool IsPathSetAndRooted(string path)
        {
            try
            {
                if (string.IsNullOrEmpty(path) == false &&
                    Path.IsPathRooted(path) == true)
                {
                    return true;
                }
            }
            catch
            {
            }

            return false;
        }

        /// <summary>
        /// List all the valid asset file paths for the given emitter set file path.
        /// The first returned path would be the directory that the emitter set is
        /// at, no matter if it exists or not.
        /// </summary>
        /// <param name="assetFolderName">The folder name of the assets.</param>
        /// <param name="esetFilePath">The emitter set file path ( or the directory ).</param>
        /// <param name="checkDirExistence">True to filter out the paths that do not exist.</param>
        /// <returns>A list of valid and existing asset directory paths.</returns>
        public static List<string> ListValidAssetPaths(
            string assetFolderName,
            string esetFilePath,
            bool checkDirExistence)
        {
            List<string> pathList = new List<string>();

            string dirPath = GetDirectoryName(esetFilePath);

            // Validate the eset file path first.
            if (IsPathSetAndRooted(dirPath) == false)
            {
                return pathList;
            }

            // Add the directory of the eset file to our list.
            pathList.Add(dirPath);

            // If the folder exists, add it to our list.
            string texPath = Path.Combine(dirPath, assetFolderName);
            if (checkDirExistence == false ||
                 Directory.Exists(texPath) == true)
            {
                pathList.Add(texPath);
            }

            // Traverse to the root directory
            while (Path.IsPathRooted(dirPath) == true)
            {
                // Get to the parent directory.
                dirPath = Path.GetDirectoryName(dirPath);
                if (string.IsNullOrEmpty(dirPath) == true)
                {
                    break;
                }

                // If the folder exists, add it to our list.
                texPath = Path.Combine(dirPath, assetFolderName);
                if (checkDirExistence == false ||
                     Directory.Exists(texPath) == true)
                {
                    pathList.Add(texPath);
                }
            }

            return pathList;
        }

        /// <summary>
        /// エクスプローラーでフォルダを開きます。
        /// </summary>
        /// <param name="path">フォルダパス</param>
        /// <param name="selectFile">ファイル選択のON/OFF</param>
        public static void OpenPathByExplorer(
            string path,
            bool selectFile = true)
        {
            if (path.Length > 0)
            {
                string commandline;

                if (selectFile == true)
                {
                    commandline = "/e, /select,\"" + path + "\"";
                }
                else
                {
                    commandline = "/e,\"" + path + "\"";
                }

                System.Diagnostics.Process.Start("EXPLORER.EXE", commandline);
            }
        }

        /// <summary>
        /// 絶対パスを取得します。
        /// filePathが絶対パスでないときはdirPath + filePath、
        /// filePathが絶対パスのときはfilePathをそのままかえします。
        /// </summary>
        /// <param name="dirPath">ディレクトリパス</param>
        /// <param name="filePath">ファイルパス</param>
        /// <returns>絶対パスを返します。</returns>
        public static string GetRootedPath(string dirPath, string filePath)
        {
            if (string.IsNullOrEmpty(filePath) == true)
            {
                return string.Empty;
            }

            if (string.IsNullOrEmpty(dirPath) == true)
            {
                return filePath;
            }

            string path = filePath;
            if (Path.IsPathRooted(path) == false)
            {
                path = Path.Combine(dirPath, filePath);
            }

            return Path.GetFullPath(path);
        }

        /// <summary>
        /// 相対パスを取得します。
        /// </summary>
        /// <param name="rootPath">元にするルートパス</param>
        /// <param name="fullPath">相対パスを求めるパス</param>
        /// <returns>相対パスを返します。</returns>
        public static string GetRelativePath(string rootPath, string fullPath)
        {
            try
            {
                if (string.IsNullOrEmpty(rootPath))
                {
                    return fullPath;
                }
                else
                {
                    Uri root = new Uri(rootPath, UriKind.RelativeOrAbsolute);
                    Uri path = new Uri(fullPath, UriKind.RelativeOrAbsolute);
                    path = root.MakeRelativeUri(path);
                    return path.ToString();
                }
            }
            catch
            {
                return fullPath;
            }
        }

        /// <summary>
        /// パスに埋め込まれた環境変数をロードします。
        /// </summary>
        /// <param name="path">ファイルパス</param>
        /// <returns>環境変数をロードしたパスを返します。</returns>
        public static string LoadEnvironments(string path)
        {
            if (string.IsNullOrEmpty(path))
            {
                return path;
            }

            MatchCollection matches;

            // $(Env)に該当する文字列を検索, 置換
            matches = Regex.Matches(path, @"\$\(\w+\)", RegexOptions.IgnoreCase);

            foreach (Match item in matches)
            {
                string envName = item.Value.Substring(2, item.Value.Length - 3);
                string envValue = Environment.GetEnvironmentVariable(envName);

                // 未設定の環境変数は置換しない
                if (envValue == null)
                {
                    continue;
                }

                path = path.Replace(item.Value, envValue);
            }

            // %Env%に該当する文字列を検索, 置換
            matches = Regex.Matches(path, @"\%\w+\%", RegexOptions.IgnoreCase);

            foreach (Match item in matches)
            {
                string envName = item.Value.Substring(1, item.Value.Length - 2);
                string envValue = Environment.GetEnvironmentVariable(envName);

                // 未設定の環境変数は置換しない
                if (envValue == null)
                {
                    continue;
                }

                path = path.Replace(item.Value, envValue);
            }

            return path;
        }

        /// <summary>
        /// Helper method for make the given path consistent as an absolute file path.
        /// </summary>
        /// <param name="path">The file path.</param>
        /// <param name="workingFolder">The working folder.</param>
        /// <returns>The absolute path.</returns>
        public static string ToAbsolutePath(
            string path,
            string workingFolder = null)
        {
            if (string.IsNullOrEmpty(path) == true)
            {
                return string.Empty;
            }

            // 環境変数をロード
            path = LoadEnvironments(path);

            if (Path.IsPathRooted(path) == false)
            {
                if (string.IsNullOrEmpty(workingFolder) == true)
                {
                    return string.Empty;
                }
                else
                {
                    return Path.GetFullPath(Path.Combine(workingFolder, path));
                }
            }

            return Path.GetFullPath(path);
        }

        /// <summary>
        /// Enumerate through all the sub folders.
        /// </summary>
        /// <param name="currDirInfo">The directory info of the folder to start seaching.</param>
        /// <param name="includeSelf">Inlcude self.</param>
        /// <returns>The directory info of all the sub folders.</returns>
        public static IEnumerable<DirectoryInfo> EnumerateSubFolders(
            DirectoryInfo currDirInfo,
            bool includeSelf = true)
        {
            // Validate the given folder path.
            if (currDirInfo == null || currDirInfo.Exists == false)
            {
                yield break;
            }

            if ((currDirInfo.Attributes & FileAttributes.System) != 0)
            {
                yield break;
            }

            try
            {
                DirectorySecurity dirSecurity = currDirInfo.GetAccessControl();
            }
            catch (Exception)
            {
                // The current user is not authorized to access this folder.
                yield break;
            }

            if (includeSelf == true)
            {
                yield return currDirInfo;
            }

            // Enumerate through sub folders.
            foreach (DirectoryInfo subDirInfo in currDirInfo.EnumerateDirectories())
            {
                IEnumerable<DirectoryInfo> enumerator = null;
                try
                {
                    // Get the enumerator for the children of the sub folder.
                    enumerator = EnumerateSubFolders(subDirInfo, true);
                }
                catch
                {
                    enumerator = null;
                }

                if (enumerator != null)
                {
                    // Enumerate through the children of the sub folder.
                    foreach (DirectoryInfo dirInfo in enumerator)
                    {
                        if ((dirInfo.Attributes & FileAttributes.System) == 0)
                        {
                            yield return dirInfo;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 指定したパス内のフォルダが一定個数以下になるように削除します。
        /// </summary>
        /// <param name="path">サブフォルダの個数を管理したいパス</param>
        /// <param name="num">キープしたいフォルダの個数</param>
        public static void DeleteSubDirectoriesOverNum(string path, int num)
        {
            var errorDirInfo = new DirectoryInfo(path);
            if (!errorDirInfo.Exists)
            {
                return;
            }

            var dirs = errorDirInfo.EnumerateDirectories().OrderBy(i => i.LastWriteTime);
            int dirCount = dirs.Count();
            foreach (var dir in dirs)
            {
                if (dirCount <= num)
                {
                    break;
                }

                try
                {
                    dir.Delete(true);
                    --dirCount;
                }
                catch
                {
                    // 消せなかったらスキップ
                }
            }
        }

        /// <summary>
        /// 指定したパスに空のディレクトリを作成します。
        /// 既にディレクトリがある場合は中身を空にします。
        /// </summary>
        /// <param name="path">空のディレクトリを作成するパス</param>
        /// <returns>フォルダが作成できたらtrue,できなかったらfalse.</returns>
        public static bool MakeEmptyDirectory(string path)
        {
            // Check if the output folder exists.
            var dirInfo = new DirectoryInfo(path);
            if (dirInfo.Exists == true)
            {
                // The folder already exists, delete the files in the folder.
                foreach (var fileInfo in dirInfo.GetFileSystemInfos())
                {
                    try
                    {
                        // The files are readonly, so we need to set it back then delete them.
                        fileInfo.Attributes = FileAttributes.Normal;
                        fileInfo.Delete();
                    }
                    catch
                    {
                        // Just skip the file.
                    }
                }
            }
            else
            {
                try
                {
                    // Create the folder.
                    dirInfo.Create();
                }
                catch
                {
                    // 並列起動時に作成に失敗することがありうるので、その場合は存在さえしていればよしとする
                    if (!Directory.Exists(path))
                    {
//                        Logger.Log("Console,LogView", LogLevels.Error, "Output folder {0} is not created!", path);
                        return false;
                    }
                }
            }

            return true;
        }
    }
}
