﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;

namespace EffectCombiner.Core
{
    public struct PathLookupInfo : IEquatable<PathLookupInfo>
    {
        public string Path { get; private set; }
        public string[] SearchPatterns { get; private set; }
        public bool IncludeSubDirectories { get; private set; }

        public PathLookupInfo(string path)
            : this(path, false)
        {
        }

        public PathLookupInfo(string path, params string[] searchPatterns)
            : this(path, false, searchPatterns)
        {
        }

        public PathLookupInfo(string path, bool includeSubDirectories)
            : this(path, includeSubDirectories, new string[0])
        {
        }

        public PathLookupInfo(string path, bool includeSubDirectories, params string[] searchPatterns)
            : this()
        {
            if (string.IsNullOrWhiteSpace(path))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "path"), "path");

            Path = path;
            SearchPatterns = searchPatterns.DefaultIfEmpty("*.*").ToArray();
            IncludeSubDirectories = includeSubDirectories;
        }

        public override string ToString()
        {
            return string.Format("{0} : {1}{2}",
                Path, string.Join(", ", SearchPatterns), IncludeSubDirectories ? " ; recursive" : string.Empty);
        }

        public string ToClipboardString()
        {
            return string.Format("{0}|{1}|{2}",
                Path,
                IncludeSubDirectories,
                string.Join(";", SearchPatterns));
        }

        public static bool FromClipboardString(string clipboardString, out PathLookupInfo pathLookupInfo)
        {
            pathLookupInfo = new PathLookupInfo();

            if (string.IsNullOrWhiteSpace(clipboardString))
                return false;

            var elements = clipboardString.Split(new[] { '|' }, StringSplitOptions.None);
            if (elements.Length != 3)
                return false;

            bool subdirs;
            if (bool.TryParse(elements[1].Trim(), out subdirs) == false)
                return false;

            pathLookupInfo = new PathLookupInfo(
                elements[0].Trim(),
                subdirs,
                CoreUtility.Split(elements[2].Trim(), ',', ';', ' '));

            return true;
        }

        #region Equals

        public bool Equals(PathLookupInfo other)
        {
            try
            {
                return
                    string.Compare(System.IO.Path.GetFullPath(Path).TrimEnd('\\'),
                                   System.IO.Path.GetFullPath(other.Path).TrimEnd('\\'),
                                   StringComparison.InvariantCultureIgnoreCase) == 0 &&
                    SearchPatterns.SequenceEqual(other.SearchPatterns) &&
                    IncludeSubDirectories == other.IncludeSubDirectories;
            }
            catch (ArgumentException) { return false; }
            catch (System.Security.SecurityException) { return false; }
        }

        public override bool Equals(object obj)
        {
            if ((obj is PathLookupInfo) == false)
                return false;

            return Equals((PathLookupInfo)obj);
        }

        public override int GetHashCode()
        {
            return (Path +
                string.Concat(SearchPatterns) +
                IncludeSubDirectories).GetHashCode();
        }

        public static bool operator ==(PathLookupInfo t1, PathLookupInfo t2)
        {
            return t1.Equals(t2);
        }

        public static bool operator !=(PathLookupInfo t1, PathLookupInfo t2)
        {
            return !(t1 == t2);
        }

        #endregion
    }
}
