﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;
using System.IO;
using System.Diagnostics;
using System.Text.RegularExpressions;

namespace EffectCombiner.Core
{
    public static class CoreUtility
    {
        public static string ShortenPath(string path)
        {
            const string pattern = @"^(\w+:[\\]|\\).*(\\[^\\]+\\[^\\]+)$";
            const string replacement = "$1...$2";

            if (Regex.IsMatch(path, pattern))
                return Regex.Replace(path, pattern, replacement);

            return path;
        }

        public static bool ArrayEquals(string[,] array1, string[,] array2)
        {
            if (array1 == null || array2 == null)
                return false;

            var dimX = array1.GetLength(0);
            var dimY = array1.GetLength(1);

            if (dimX != array2.GetLength(0) || dimY != array2.GetLength(1))
                return false;

            for (var y = 0; y < dimY; y++)
            {
                for (var x = 0; x < dimX; x++)
                {
                    if (array1[x, y] != array2[x, y])
                        return false;
                }
            }

            return true;
        }

        public static string[,] DuplicateArray(string[,] source)
        {
            if (source == null)
                throw new ArgumentNullException("source");

            var dimX = source.GetLength(0);
            var dimY = source.GetLength(1);

            var array = new string[dimX, dimY];

            for (var y = 0; y < dimY; y++)
            {
                for (var x = 0; x < dimX; x++)
                    array[x, y] = source[x, y];
            }

            return array;
        }

        public static void ArrayCopy(string[,] destination, string[,] source)
        {
            if (source == null || destination == null)
                return;

            var minDimX = Math.Min(source.GetLength(0), destination.GetLength(0));
            var minDimY = Math.Min(source.GetLength(1), destination.GetLength(1));

            for (var y = 0; y < minDimY; y++)
            {
                for (var x = 0; x < minDimX; x++)
                    destination[x, y] = source[x, y];
            }
        }

        public static string MultiDimArrayToString(string[,] array)
        {
            if (array == null)
                throw new ArgumentNullException("array");

            var dimensionX = array.GetLength(0);
            var dimensionY = array.GetLength(1);

            var k = 0;
            var elements = new string[dimensionX * dimensionY];

            for (var y = 0; y < dimensionY; y++)
            {
                for (var x = 0; x < dimensionX; x++)
                    elements[k++] = array[x, y];
            }

            return string.Join(";", elements);
        }

        public static string[,] StringToMultiDimArray(string values, int dimensionX, int dimensionY)
        {
            if (string.IsNullOrWhiteSpace(values))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "values"), "values");

            var elements = values.Split(';');

            var array = new string[dimensionX, dimensionY];

            var k = 0;
            for (var y = 0; y < dimensionY; y++)
            {
                for (var x = 0; x < dimensionX; x++)
                    array[x, y] = elements[k++];
            }

            return array;
        }

        public static IEnumerable<string> RetrieveFiles(IEnumerable<PathLookupInfo> pathLookupInfos)
        {
            return from pathInfo in pathLookupInfos
                   from ext in pathInfo.SearchPatterns
                   let opt = pathInfo.IncludeSubDirectories ?
                   SearchOption.AllDirectories :
                   SearchOption.TopDirectoryOnly
                   from file in Directory.GetFiles(pathInfo.Path, ext, opt)
                   select file;
        }

        public static string[] Split(string inline, params char[] sep)
        {
            if (inline == null)
                return new string[0];

            return inline
                .Trim()
                .Split(sep, StringSplitOptions.RemoveEmptyEntries)
                .Select(p => p.Trim())
                .ToArray();
        }

        public static string UsePath(string relativePath)
        {
            var assemblyPath = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
            var absolutePath = string.Format("{0}\\{1}", assemblyPath, relativePath);

            if (Directory.Exists(absolutePath) == false)
                Directory.CreateDirectory(absolutePath);

            return absolutePath;
        }

        public static T[] GetAttribute<T>(object instance) where T : Attribute
        {
            if (instance == null)
                throw new ArgumentNullException("instance");

            return instance
                .GetType()
                .GetCustomAttributes(typeof(T), true)
                .Cast<T>()
                .ToArray();
        }

        public static void ShowFileInExplorer(string file)
        {
            Process.Start("explorer.exe", string.Format("/select, \"{0}\"", file));
        }

        public static string DumpException(Exception ex)
        {
            if (ex == null)
                throw new ArgumentNullException("ex");
            return DumpExceptionRecursive(ex, 0);
        }

        private static string DumpExceptionRecursive(Exception ex, int level)
        {
            var sb = new StringBuilder();

            var indent = new string(' ', level * 4);

            sb.AppendLine(string.Format("{0}{1}: {2}", indent, Messages.EXCEPTION_TEXT_TYPE, ex.GetType().FullName));
            sb.AppendLine(string.Format("{0}{1}: {2}", indent, Messages.EXCEPTION_TEXT_MESSAGE, ex.Message));

            if (ex.Data.Keys.Count > 0)
            {
                sb.AppendLine(string.Format("{0}{1}:", indent, Messages.EXCEPTION_TEXT_DATA));
                foreach (var key in ex.Data.Keys)
                {
                    sb.AppendLine(string.Format("{0}  {1}: {2}, {3}: {4}", indent,
                        Messages.EXCEPTION_TEXT_KEY, key,
                        Messages.EXCEPTION_TEXT_VALUE, ex.Data[key]));
                }
            }

            if (ex is ReflectionTypeLoadException)
            {
                foreach (var loaderException in ((ReflectionTypeLoadException)ex).LoaderExceptions)
                    sb.AppendLine(DumpExceptionRecursive(loaderException, level + 1));
            }
            else if (ex is AggregateException)
            {
                foreach (var innerException in ((AggregateException)ex).InnerExceptions)
                    sb.AppendLine(DumpExceptionRecursive(innerException, level + 1));
            }

            if (ex.StackTrace != null)
            {
                sb.AppendLine(string.Format("{0}{1}:", indent, Messages.EXCEPTION_TEXT_STACK));
                DumpStackTrace(level, ex.StackTrace, sb);
            }

            if (ex.InnerException != null)
            {
                sb.AppendLine();
                sb.AppendLine(DumpExceptionRecursive(ex.InnerException, level + 1));
            }

            return sb.ToString();
        }

        private static void DumpStackTrace(int level, string stackTrace, StringBuilder sb)
        {
            var indent = new string(' ', 2 + level * 4);

            foreach (var line in stackTrace.Split('\n').Select(l => l.Trim()))
                sb.AppendLine(string.Format("{0}{1}", indent, line));
        }
    }
}
