﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.IO.Compression;
using System.Collections.Concurrent;

namespace EffectCombiner.Core
{
    public enum CacheMode
    {
        None,
        Regular,
        Compressed,
    }

    public class ContentRepository : IContentRepository
    {
        public CacheMode CacheMode { get; private set; }
        private readonly ConcurrentDictionary<Uri, CacheItemInfo> cache = new ConcurrentDictionary<Uri, CacheItemInfo>();

        public ContentRepository(CacheMode cacheMode)
        {
            CacheMode = cacheMode;
        }

        protected virtual string ResolveResourceContent(Uri resourceLocation)
        {
            return null;
        }

        public void ClearContent(Uri resource)
        {
            CacheItemInfo dummy;
            cache.TryRemove(resource, out dummy);
        }

        public void ClearContent()
        {
            cache.Clear();
        }

        private DateTime GetTimeStamp(Uri resourceLocation)
        {
            if (resourceLocation.IsFile)
            {
                return File.GetLastWriteTimeUtc(resourceLocation.LocalPath);
            }
            else
            {
                return DateTime.MinValue;
            }
        }

        public string GetContent(Uri resourceLocation, Encoding encoding)
        {
            if (resourceLocation == null)
                throw new ArgumentNullException("resourceLocation");

            if (CacheMode == CacheMode.None)
                return ExtractContent(resourceLocation, encoding);

            byte[] data;
            CacheItemInfo itemInfo;
            DateTime lastModification = GetTimeStamp(resourceLocation);

            if (cache.ContainsKey(resourceLocation) &&
                cache[resourceLocation].TimeStamp == lastModification)
            {
                itemInfo = cache[resourceLocation];

                data = itemInfo.RawData;

                if (itemInfo.IsCompressed)
                    data = Decompress(data, itemInfo.OriginalSize);

                return itemInfo.Encoding.GetString(data);
            }

            var content = ExtractContent(resourceLocation, encoding);
            if (content == null)
                return null;

            data = encoding.GetBytes(content);

            if (data.Length == 0)
                return content;

            var originalSize = data.Length;
            var isCompressed = false;

            if (CacheMode == CacheMode.Compressed)
            {
                isCompressed = true;
                data = Compress(data);
            }

            itemInfo = new CacheItemInfo
            {
                RawData = data,
                Encoding = encoding,
                IsCompressed = isCompressed,
                OriginalSize = originalSize,
                TimeStamp = lastModification,
            };

            cache[resourceLocation] = itemInfo;

            return content;
        }

        private string ExtractContent(Uri resourceLocation, Encoding encoding)
        {
            string content = null;

            if (resourceLocation.IsFile)
            {
                using (var reader = new StreamReader(resourceLocation.LocalPath, encoding))
                {
                    content = reader.ReadToEnd();
                }
            }
            else
            {
                content = ResolveResourceContent(resourceLocation);
            }
            return content;
        }

        private static byte[] Compress(byte[] data)
        {
            var ms = new MemoryStream();
            //using (var stream = new GZipStream(ms, CompressionMode.Compress))
            using (var stream = new DeflateStream(ms, CompressionMode.Compress))
            {
                stream.Write(data, 0, data.Length);
            }
            var compressedData = ms.ToArray();
            return compressedData;
        }

        private static byte[] Decompress(byte[] data, int originalSize)
        {
            var ms = new MemoryStream();
            //using (var stream = new GZipStream(new MemoryStream(data), CompressionMode.Decompress))
            using (var stream = new DeflateStream(new MemoryStream(data), CompressionMode.Decompress))
            {
                stream.CopyTo(ms, originalSize);
            }
            var uncompressedData = ms.ToArray();
            return uncompressedData;
        }

        private struct CacheItemInfo
        {
            public byte[] RawData;
            public Encoding Encoding;
            public bool IsCompressed;
            public int OriginalSize;
            public DateTime TimeStamp;
        }
    }
}
