﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.ServiceModel;
using System.Threading.Tasks;
using System.ServiceModel.Channels;

namespace Communication.Core
{
    internal class CommunicationGlobals
    {
        internal static readonly string ServiceBaseAddress = "net.pipe://localhost";

        internal static Binding GetChannelBinding()
        {
            return new NetNamedPipeBinding
            {
                TransferMode = TransferMode.Streamed,
                MaxBufferSize = 65536 * 2,
                MaxReceivedMessageSize = 65536 * 2,
                ReaderQuotas =
                {
                    MaxStringContentLength = 65536 * 2
                },
            };
        }
    }

    public class CommunicationServer<TInterface> where TInterface : class
    {
        private ServiceHost host;
        private Task runTask;

        private readonly string endPointName;
        private readonly object syncRoot = new object();

        public CommunicationServer()
        {
            var interfaceType = typeof(TInterface);
            endPointName = interfaceType.FullName;

            if (endPointName == null)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_TYPE_PARAM, "TInterface"));
        }

        public Task Start(TInterface serviceInstance)
        {
            if (serviceInstance == null)
                throw new ArgumentNullException("serviceInstance");

            if (runTask != null)
                return runTask;

            lock (syncRoot)
            {
                if (runTask != null)
                    return runTask;

                runTask = Task.Factory.StartNew(() =>
                    {
                        host = new ServiceHost(serviceInstance);

                        host.AddServiceEndpoint(typeof(TInterface),
                            CommunicationGlobals.GetChannelBinding(),
                            CommunicationGlobals.ServiceBaseAddress + "/" + endPointName);

                        host.Open();
                    });
            }

            return runTask;
        }

        public Task Stop()
        {
            lock (syncRoot)
            {
                if (runTask == null)
                    throw new InvalidOperationException(Messages.SERVICE_NOT_YET_STARTED);

                if (runTask.IsFaulted && runTask.Exception != null)
                    throw runTask.Exception;

                if (runTask.IsCompleted)
                    return Task.Factory.StartNew(host.Close);

                return runTask.ContinueWith(t => Stop());
            }
        }
    }

    public static class CommunicationClient
    {
        public static TInterface CreateProxy<TInterface>()
        {
            var interfaceType = typeof(TInterface);
            var endPointName = interfaceType.FullName;

            if (endPointName == null)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_TYPE_PARAM, "TInterface"));

            var channelFactory = new ChannelFactory<TInterface>(CommunicationGlobals.GetChannelBinding());
            return channelFactory.CreateChannel(new EndpointAddress(CommunicationGlobals.ServiceBaseAddress + "/" + endPointName));
        }
    }
}
