﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;

namespace Blocks.Core
{
    /// <summary>
    /// ポインタ入力情報のベースクラスです。
    /// </summary>
    public abstract class PointerInputBase
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="x">X 座標</param>
        /// <param name="y">Y 座標</param>
        protected PointerInputBase(double x, double y)
        {
            this.X = x;
            this.Y = y;
        }

        /// <summary>
        /// X 座標を取得します。
        /// </summary>
        public double X { get; private set; }

        /// <summary>
        /// Y 座標を取得します。
        /// </summary>
        public double Y { get; private set; }
    }

    /// <summary>
    /// マウスボタンのステータスです。
    /// </summary>
    public enum MouseButtonState
    {
        /// <summary>
        /// 押下状態
        /// </summary>
        Pressed,

        /// <summary>
        /// 押上状態
        /// </summary>
        Released,
    }

    /// <summary>
    /// マウスの入力情報です。
    /// </summary>
    public class MouseInput : PointerInputBase
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="x">X 座標</param>
        /// <param name="y">Y 座標</param>
        /// <param name="wheelDelta">ホイールの回転量</param>
        /// <param name="left">左ボタンのステータス</param>
        /// <param name="middle">中ボタンのステータス</param>
        /// <param name="right">右ボタンのステータス</param>
        public MouseInput(double x, double y, int wheelDelta, MouseButtonState left, MouseButtonState middle, MouseButtonState right)
            : base(x, y)
        {
            this.WheelDelta = wheelDelta;

            this.LeftButton = left;
            this.MiddleButton = middle;
            this.RightButton = right;
        }

        /// <summary>
        /// ホイールの回転量を取得します。
        /// </summary>
        public int WheelDelta { get; private set; }

        /// <summary>
        /// 左ボタンのステータスを取得します。
        /// </summary>
        public MouseButtonState LeftButton { get; private set; }

        /// <summary>
        /// 中ボタンのステータスを取得します。
        /// </summary>
        public MouseButtonState MiddleButton { get; private set; }

        /// <summary>
        /// 右ボタンのステータスを取得します。
        /// </summary>
        public MouseButtonState RightButton { get; private set; }
    }

    /// <summary>
    /// ドラッグ & ドロップ中のステータスです。
    /// </summary>
    public enum DragDropState
    {
        /// <summary>
        /// デフォルト
        /// </summary>
        Default = 0,

        /// <summary>
        /// 無効
        /// </summary>
        None = 0x01 << 0,

        /// <summary>
        /// コピー
        /// </summary>
        Copy = 0x01 << 1,

        /// <summary>
        /// リンク
        /// </summary>
        Link = 0x01 << 2,

        /// <summary>
        /// 移動
        /// </summary>
        Move = 0x01 << 3,

        /// <summary>
        /// スクロール
        /// </summary>
        Scroll = 0x01 << 4,

        /// <summary>
        /// Copy, Move, Scroll の組み合わせ
        /// </summary>
        All = Copy | Move | Scroll,
    }

    /// <summary>
    /// ドラッグ & ドロップ中のデータのタイプです。
    /// </summary>
    public enum DragDropDataType
    {
        /// <summary>
        /// ファイル
        /// </summary>
        File,

        /// <summary>
        /// object データ
        /// </summary>
        ObjectData,
    }

    /// <summary>
    /// キーボードの入力情報です。
    /// </summary>
    public class KeyboardInput
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="keyCode">キーコード</param>
        /// <param name="alt">Alt キーが押されているかどうか</param>
        /// <param name="control">Control キーが押されているかどうか</param>
        /// <param name="shift">Shift キーが押されているかどうか</param>
        public KeyboardInput(int keyCode, bool alt, bool control, bool shift)
        {
            this.KeyCode = keyCode;
            this.Alt = alt;
            this.Control = control;
            this.Shift = shift;
        }

        /// <summary>
        /// キーコードを取得します。
        /// </summary>
        public int KeyCode { get; private set; }

        /// <summary>
        /// Alt キーが押されているかどうか取得します。
        /// </summary>
        public bool Alt { get; private set; }

        /// <summary>
        /// Control キーが押されているかどうか取得します。
        /// </summary>
        public bool Control { get; private set; }

        /// <summary>
        /// Shift キーが押されているかどうか取得します。
        /// </summary>
        public bool Shift { get; private set; }
    }
}
