﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using EffectDefinitions;

namespace Blocks.Core
{
    /// <summary>
    /// ブロックエレメントを管理するクラスです。
    /// </summary>
    public abstract class BlockManagerBase : IBlockElementGenerator
    {
        /// <summary>
        /// ブロックエレメントリストの処理について同期をとるためのロックオブジェクトです。
        /// </summary>
        public readonly object BlocksSyncRoot = new object();

        /// <summary>
        /// ブロックエレメントリストです。
        /// </summary>
        private readonly List<BlockElementBase> blockElements = new List<BlockElementBase>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        protected BlockManagerBase()
        {
            this.BlockElements = new ReadOnlyCollection<BlockElementBase>(this.blockElements);
        }

        /// <summary>
        /// ブロックエレメントリストを取得します。
        /// </summary>
        public IList<BlockElementBase> BlockElements { get; private set; }

        /// <summary>
        /// ブロックエレメントを作成します。
        /// </summary>
        /// <param name="blockDefinition">ブロック定義データ</param>
        /// <returns>作成したブロックエレメントを返します。</returns>
        public abstract BlockElementBase CreateBlockElement(BlockDefinition blockDefinition);

        #region Visibility

        /// <summary>
        /// ブロックエレメントの表示状態が変わったときに発生するイベントです。
        /// </summary>
        public event EventHandler<BlockElementEventArgs> VisibilityChanged;

        /// <summary>
        /// ブロックエレメントの表示状態が変わったときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected virtual void OnVisibilityChanged(BlockElementEventArgs e)
        {
            this.VisibilityChanged?.Invoke(this, e);
        }

        /// <summary>
        /// VisibilityChanged イベントを発生させます。
        /// </summary>
        /// <param name="blockElementBase">ブロックエレメント</param>
        internal void NotifyVisibilityChanged(BlockElementBase blockElementBase)
        {
            this.OnVisibilityChanged(new BlockElementEventArgs(blockElementBase));
        }

        #endregion

        #region Selection

        /// <summary>
        /// 全てのブロックエレメントの選択をまとめて解除します。
        /// </summary>
        public void ClearSelections()
        {
            lock (this.BlocksSyncRoot)
            {
                foreach (var blockElement in this.SelectedItems)
                {
                    blockElement.IsSelected = false;
                }
            }
        }

        /// <summary>
        /// 選択中のブロックエレメントを取得します。
        /// </summary>
        public BlockElementBase[] SelectedItems
        {
            get
            {
                lock (this.BlocksSyncRoot)
                {
                    return this.blockElements.Where(b => b.IsSelected).ToArray();
                }
            }
        }

        /// <summary>
        /// ブロックエレメントの選択状態が変わったときに発生するイベントです。
        /// </summary>
        public event EventHandler<BlockElementEventArgs> SelectionChanged;

        /// <summary>
        /// ブロックエレメントの選択状態が変わったときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected virtual void OnSelectionChanged(BlockElementEventArgs e)
        {
            this.SelectionChanged?.Invoke(this, e);
        }

        /// <summary>
        /// SelectionChanged イベントを発生させます。
        /// </summary>
        /// <param name="blockElement">ブロックエレメント</param>
        internal void NotifySelectionChanged(BlockElementBase blockElement)
        {
            this.OnSelectionChanged(new BlockElementEventArgs(blockElement));
        }

        /// <summary>
        /// ブロックエレメントのドラッグ状態が変わったときに発生するイベントです。
        /// </summary>
        public event EventHandler<BlockElementEventArgs> DraggingChanged;

        /// <summary>
        /// ブロックエレメントのドラッグ状態が変わったときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected virtual void OnDraggingChanged(BlockElementEventArgs e)
        {
            this.DraggingChanged?.Invoke(this, e);
        }

        /// <summary>
        /// DraggingChanged イベントを発生させます。
        /// </summary>
        /// <param name="blockElement">ブロックエレメント</param>
        internal void NotifyDraggingChanged(BlockElementBase blockElement)
        {
            this.OnDraggingChanged(new BlockElementEventArgs(blockElement));
        }

        #endregion

        #region AddBlock

        /// <summary>
        /// ブロックエレメントを追加します。
        /// </summary>
        /// <param name="blockElement">ブロックエレメント</param>
        public bool AddBlock(BlockElementBase blockElement)
        {
            if (blockElement == null)
                throw new ArgumentNullException("blockElement");

            lock (this.BlocksSyncRoot)
            {
                if (this.blockElements.Contains(blockElement))
                {
                    return false;
                }

                this.blockElements.Add(blockElement);
                blockElement.BlockManager = this;
                blockElement.OnAddedToBlockManager();
                blockElement.PositionChanged += this.BlockElementPositionChangedHook;
                blockElement.SizeChanged += this.BlockElementSizeChangedHook;
                blockElement.ZOrderChanged += this.BlockElementZOrderChangedHook;

                this.UpdateZOrders();
            }

            this.OnBlockAdded(new BlockElementEventArgs(blockElement));

            return true;
        }

        /// <summary>
        /// ブロックエレメントが追加されたときに発生するイベントです。
        /// </summary>
        public event EventHandler<BlockElementEventArgs> BlockAdded;

        /// <summary>
        /// ブロックエレメントが追加されたときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected virtual void OnBlockAdded(BlockElementEventArgs e)
        {
            this.BlockAdded?.Invoke(this, e);
        }

        #endregion

        #region RemoveBlock

        /// <summary>
        /// 全てのブロックエレメントをまとめて削除します。
        /// </summary>
        public void ClearBlocks()
        {
            lock (this.BlocksSyncRoot)
            {
                this.ClearSelections();

                // RemoveBlock メソッドで blockElements の内容が変わるので配列にしてから処理する
                foreach (var blockElement in this.blockElements.ToArray())
                {
                    this.RemoveBlock(blockElement);
                }
            }
        }

        /// <summary>
        /// ブロックエレメントを削除します。
        /// </summary>
        /// <param name="blockElement">ブロックエレメント</param>
        /// <returns>ブロックエレメントを削除したときtrue、それ以外はfalseを返します。</returns>
        public bool RemoveBlock(BlockElementBase blockElement)
        {
            if (blockElement == null)
                throw new ArgumentNullException("blockElement");

            lock (this.BlocksSyncRoot)
            {
                if (this.blockElements.Contains(blockElement) == false)
                {
                    return false;
                }

                this.blockElements.Remove(blockElement);
                blockElement.PositionChanged -= this.BlockElementPositionChangedHook;
                blockElement.SizeChanged -= this.BlockElementSizeChangedHook;
                blockElement.ZOrderChanged -= this.BlockElementZOrderChangedHook;
                blockElement.OnRemovedFromBlockManager();
                blockElement.BlockManager = null;

                this.UpdateZOrders();
            }

            this.OnBlockRemoved(new BlockElementEventArgs(blockElement));

            return true;
        }

        /// <summary>
        /// ブロックエレメントが削除されたときに発生するイベントです。
        /// </summary>
        public event EventHandler<BlockElementEventArgs> BlockRemoved;

        /// <summary>
        /// ブロックエレメントが削除されたときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected virtual void OnBlockRemoved(BlockElementEventArgs e)
        {
            this.BlockRemoved?.Invoke(this, e);
        }

        #endregion

        #region ReplaceBlock

        /// <summary>
        /// ブロックエレメントを置換します。
        /// </summary>
        /// <param name="oldBlockElement">元のブロック</param>
        /// <param name="newBlockElement">新しいブロック</param>
        public bool ReplaceBlock(BlockElementBase oldBlockElement, BlockElementBase newBlockElement)
        {
            if (oldBlockElement == null)
                throw new ArgumentNullException("oldBlockElement");
            if (newBlockElement == null)
                throw new ArgumentNullException("newBlockElement");

            lock (this.BlocksSyncRoot)
            {
                if (this.blockElements.Contains(oldBlockElement) == false ||
                    this.blockElements.Contains(newBlockElement))
                {
                    return false;
                }

                int zOrder = oldBlockElement.ZOrder;

                this.blockElements.Remove(oldBlockElement);
                oldBlockElement.PositionChanged -= this.BlockElementPositionChangedHook;
                oldBlockElement.SizeChanged -= this.BlockElementSizeChangedHook;
                oldBlockElement.ZOrderChanged -= this.BlockElementZOrderChangedHook;
                oldBlockElement.OnRemovedFromBlockManager();
                oldBlockElement.BlockManager = null;

                // ゴーストブロックを持つブロックの場合 ZOrder が範囲外になることがある
                zOrder = Math.Min(zOrder, this.blockElements.Count);

                this.blockElements.Insert(zOrder, newBlockElement);
                newBlockElement.BlockManager = this;
                newBlockElement.OnAddedToBlockManager();
                newBlockElement.PositionChanged += this.BlockElementPositionChangedHook;
                newBlockElement.SizeChanged += this.BlockElementSizeChangedHook;
                newBlockElement.ZOrderChanged += this.BlockElementZOrderChangedHook;

                this.UpdateZOrders();
            }

            this.OnBlockReplaced(new BlockElementReplacedEventArgs(oldBlockElement, newBlockElement));

            return true;
        }

        /// <summary>
        /// ブロックエレメントが置換されたときに発生するイベントです。
        /// </summary>
        public event EventHandler<BlockElementReplacedEventArgs> BlockReplaced;

        /// <summary>
        /// ブロックエレメントが置換されたときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected virtual void OnBlockReplaced(BlockElementReplacedEventArgs e)
        {
            this.BlockReplaced?.Invoke(this, e);
        }

        #endregion

        #region Z order

        /// <summary>
        /// ブロックエレメントを最前面に移動させます。
        /// </summary>
        /// <param name="blockElement">ブロックエレメント</param>
        public void BringToFront(BlockElementBase blockElement)
        {
            bool removed = this.blockElements.Remove(blockElement);

            if (removed)
            {
                this.blockElements.Insert(0, blockElement);
            }

            this.UpdateZOrders();
        }

        /// <summary>
        /// ブロックエレメントを最背面に移動させます。
        /// </summary>
        /// <param name="blockElement">ブロックエレメント</param>
        public void SendToBack(BlockElementBase blockElement)
        {
            bool removed = this.blockElements.Remove(blockElement);

            if (removed)
            {
                this.blockElements.Add(blockElement);
            }

            this.UpdateZOrders();
        }

        /// <summary>
        /// ブロックエレメントの Z オーダーを更新します。
        /// </summary>
        private void UpdateZOrders()
        {
            for (var i = 0; i < this.blockElements.Count; i++)
            {
                this.blockElements[i].ZOrder = i;
            }
        }

        #endregion

        #region BlockElementPositionChanged

        /// <summary>
        /// ブロックエレメントの位置が変わったときに発生するイベントです。
        /// </summary>
        public event EventHandler<BlockElementPositionChangedEventArgs> BlockElementPositionChanged;

        /// <summary>
        /// ブロックエレメントの位置が変わったときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected virtual void OnBlockElementPositionChanged(BlockElementPositionChangedEventArgs e)
        {
            this.BlockElementPositionChanged?.Invoke(this, e);
        }

        /// <summary>
        /// 管理しているブロックエレメントから発生した PositionChanged イベントを処理します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BlockElementPositionChangedHook(object sender, PositionChangedEventArgs e)
        {
            this.OnBlockElementPositionChanged(new BlockElementPositionChangedEventArgs((BlockElementBase)sender, e));
        }

        #endregion

        #region BlockElementSizeChanged

        /// <summary>
        /// ブロックエレメントのサイズが変わったときに発生するイベントです。
        /// </summary>
        public event EventHandler<BlockElementSizeChangedEventArgs> BlockElementSizeChanged;

        /// <summary>
        /// ブロックエレメントのサイズが変わったときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected virtual void OnBlockElementSizeChanged(BlockElementSizeChangedEventArgs e)
        {
            this.BlockElementSizeChanged?.Invoke(this, e);
        }

        /// <summary>
        /// 管理しているブロックエレメントから発生した SizeChanged イベントを処理します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BlockElementSizeChangedHook(object sender, SizeChangedEventArgs e)
        {
            this.OnBlockElementSizeChanged(new BlockElementSizeChangedEventArgs((BlockElementBase)sender, e));
        }

        #endregion

        #region BlockElementZOrderChanged

        /// <summary>
        /// ブロックエレメントの Z オーダーが変わったときに発生するイベントです。
        /// </summary>
        public event EventHandler<BlockElementZOrderChangedEventArgs> BlockElementZOrderChanged;

        /// <summary>
        /// ブロックエレメントの Z オーダーが変わったときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected virtual void OnBlockElementZOrderChanged(BlockElementZOrderChangedEventArgs e)
        {
            this.BlockElementZOrderChanged?.Invoke(this, e);
        }

        /// <summary>
        /// 管理しているブロックエレメントから発生した ZOrderChanged イベントを処理します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void BlockElementZOrderChangedHook(object sender, ZOrderChangedEventArgs e)
        {
            this.OnBlockElementZOrderChanged(new BlockElementZOrderChangedEventArgs((BlockElementBase)sender, e));
        }

        #endregion
    }
}
