﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using EffectCombiner.Core.Extensions;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.Usage.Descriptors;
using EffectDefinitions;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Workflow.Core;

namespace EffectCombinerUnitTests
{
    [TestClass]
    public class TexCoordUsageDescriptorTest
    {
        [TestMethod]
        public void InputSubPlugBaseTargetTest()
        {
            var textureUsage = new TexCoordUsageDescriptor();
            var plug = Utility.CreateInputPlug("texCoords", "vec2");
            var subPlugs = textureUsage.ExpandInput(plug);

            foreach (var subPlug in subPlugs)
                if (subPlug.BaseTarget != plug.BaseTarget)
                {
                    throw new Exception(string.Format("Sub-plug should have \"{0}\" as a base target but has \"{1}\"", plug.BaseTarget, subPlug.BaseTarget));
                }
        }

        [TestMethod]
        public void OutputSubPlugBaseTargetTest()
        {
            var textureUsage = new TexCoordUsageDescriptor();
            var plug = Utility.CreateOutputPlug("texCoords", "vec2");
            var subPlugs = textureUsage.ExpandOutput(plug);

            foreach (var subPlug in subPlugs)
                if (subPlug.BaseTarget != plug.BaseTarget)
                {
                    throw new Exception(string.Format("Sub-plug should have \"{0}\" as a base target but has \"{1}\"", plug.BaseTarget, subPlug.BaseTarget));
                }
        }

        [TestMethod]
        public void InputExpandToOutputTest()
        {
            var textureUsage = new TexCoordUsageDescriptor();
            var plug = Utility.CreateInputPlug("texCoords", "vec2");

            SubPlug[] subPlugs = null;
            try
            {
                subPlugs = textureUsage.ExpandOutput(plug);
            }
            catch
            {
                return;
            }
            if (subPlugs.IsNullOrEmpty() == false)
                throw new Exception("Calling ExpandOutput on an input plug was expected to fail");
        }

        [TestMethod]
        public void OutputExpandToInputTest()
        {
            var textureUsage = new TexCoordUsageDescriptor();
            var outputPlug = Utility.CreateOutputPlug("texCoords", "vec2");

            SubPlug[] subPlugs = null;
            try
            {
                subPlugs = textureUsage.ExpandInput(outputPlug);
            }
            catch
            {
                return;
            }
            if (subPlugs.IsNullOrEmpty() == false)
                throw new Exception("Calling ExpandInput on an output plug was expected to fail");
        }

        #region Plug/block tools

        private static void CreateTextureOutputBlock(out TexCoordUsageDescriptor textureUsage, out EffectOutputPlug outputPlug, out EffectOutputPlug[] outputSubPlugs)
        {
            textureUsage = new TexCoordUsageDescriptor();

            var plugUV = Utility.CreateOutputPlug("uv", "vec2");
            var subPlugs = textureUsage.ExpandOutput(plugUV);
            var blockElement = Utility.CreateDummyOutputBlock(new Plug[] { plugUV }.Concat(subPlugs).ToArray());
            outputPlug = new EffectOutputPlug(plugUV, blockElement);
            outputSubPlugs = subPlugs.Select(x => new EffectOutputPlug(x, blockElement)).ToArray();
        }

        private static void CreateTextureInputBlock(out TexCoordUsageDescriptor textureUsage, out EffectInputPlug inputPlug, out EffectInputPlug[] inputSubPlugs)
        {
            textureUsage = new TexCoordUsageDescriptor();

            var plugUV = Utility.CreateInputPlug("uv", "vec2");
            var subPlugs = textureUsage.ExpandInput(plugUV);
            var blockElement = Utility.CreateDummyInputBlock(new Plug[] { plugUV }.Concat(subPlugs).ToArray());
            inputPlug = new EffectInputPlug(plugUV, blockElement);
            inputSubPlugs = subPlugs.Select(x => new EffectInputPlug(x, blockElement)).ToArray();
        }

        private static void CreateUvUVInputBlock(out EffectInputPlug inputPlugUV, out EffectInputPlug inputPlugU, out EffectInputPlug inputPlugV)
        {
            var plugUV = Utility.CreateInputPlug("uv", "vec2");
            var plugU = Utility.CreateInputPlug("u", "float");
            var plugV = Utility.CreateInputPlug("v", "float");
            var blockElement = Utility.CreateDummyInputBlock(plugUV, plugU, plugV);
            inputPlugUV = new EffectInputPlug(plugUV, blockElement);
            inputPlugU = new EffectInputPlug(plugU, blockElement);
            inputPlugV = new EffectInputPlug(plugV, blockElement);
        }

        private static void CreateUvUVOutputBlock(out EffectOutputPlug outputPlugUV, out EffectOutputPlug outputPlugU, out EffectOutputPlug outputPlugV)
        {
            var plugUV = Utility.CreateOutputPlug("uv", "vec2");
            var plugU = Utility.CreateOutputPlug("u", "float");
            var plugV = Utility.CreateOutputPlug("v", "float");
            var blockElement = Utility.CreateDummyOutputBlock(plugUV, plugU, plugV);
            outputPlugUV = new EffectOutputPlug(plugUV, blockElement);
            outputPlugU = new EffectOutputPlug(plugU, blockElement);
            outputPlugV = new EffectOutputPlug(plugV, blockElement);
        }

        #endregion

        #region AreOutputsProperlyConnected

        [TestMethod]
        public void AreOutputsProperlyConnectedUVTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateTextureOutputBlock(out textureUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteUVInputPlug;
            EffectInputPlug remoteUInputPlug;
            EffectInputPlug remoteVInputPlug;
            CreateUvUVInputBlock(out remoteUVInputPlug, out remoteUInputPlug, out remoteVInputPlug);

            ConnectionManager.Connect(outputPlug, remoteUVInputPlug);

            Utility.OutputsShouldBeProperlyConnected(textureUsage, outputPlug, outputSubPlugs);
        }

        [TestMethod]
        public void AreOutputsProperlyConnectedU_VTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateTextureOutputBlock(out textureUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteUVInputPlug;
            EffectInputPlug remoteUInputPlug;
            EffectInputPlug remoteVInputPlug;
            CreateUvUVInputBlock(out remoteUVInputPlug, out remoteUInputPlug, out remoteVInputPlug);

            ConnectionManager.Connect(outputSubPlugs[0], remoteUInputPlug);
            ConnectionManager.Connect(outputSubPlugs[1], remoteVInputPlug);

            Utility.OutputsShouldBeProperlyConnected(textureUsage, outputPlug, outputSubPlugs);
        }

        [TestMethod]
        public void AreOutputsProperlyConnectedUTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateTextureOutputBlock(out textureUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteUVInputPlug;
            EffectInputPlug remoteUInputPlug;
            EffectInputPlug remoteVInputPlug;
            CreateUvUVInputBlock(out remoteUVInputPlug, out remoteUInputPlug, out remoteVInputPlug);

            ConnectionManager.Connect(outputSubPlugs[0], remoteUInputPlug);

            Utility.OutputsShouldBeProperlyConnected(textureUsage, outputPlug, outputSubPlugs);
        }

        [TestMethod]
        public void AreOutputsProperlyConnectedVTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateTextureOutputBlock(out textureUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteUVInputPlug;
            EffectInputPlug remoteUInputPlug;
            EffectInputPlug remoteVInputPlug;
            CreateUvUVInputBlock(out remoteUVInputPlug, out remoteUInputPlug, out remoteVInputPlug);

            ConnectionManager.Connect(outputSubPlugs[1], remoteVInputPlug);

            Utility.OutputsShouldBeProperlyConnected(textureUsage, outputPlug, outputSubPlugs);
        }

        [TestMethod]
        public void AreOutputsProperlyConnectedUV_UTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateTextureOutputBlock(out textureUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteUVInputPlug;
            EffectInputPlug remoteUInputPlug;
            EffectInputPlug remoteVInputPlug;
            CreateUvUVInputBlock(out remoteUVInputPlug, out remoteUInputPlug, out remoteVInputPlug);

            ConnectionManager.Connect(outputPlug, remoteUVInputPlug);
            ConnectionManager.Connect(outputSubPlugs[0], remoteUInputPlug);

            Utility.OutputsShouldBeProperlyConnected(textureUsage, outputPlug, outputSubPlugs);
        }

        [TestMethod]
        public void AreOutputsProperlyConnectedUV_VTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateTextureOutputBlock(out textureUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteUVInputPlug;
            EffectInputPlug remoteUInputPlug;
            EffectInputPlug remoteVInputPlug;
            CreateUvUVInputBlock(out remoteUVInputPlug, out remoteUInputPlug, out remoteVInputPlug);

            ConnectionManager.Connect(outputPlug, remoteUVInputPlug);
            ConnectionManager.Connect(outputSubPlugs[1], remoteVInputPlug);

            Utility.OutputsShouldBeProperlyConnected(textureUsage, outputPlug, outputSubPlugs);
        }

        [TestMethod]
        public void AreOutputsProperlyConnectedUV_U_VTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectOutputPlug outputPlug;
            EffectOutputPlug[] outputSubPlugs;
            CreateTextureOutputBlock(out textureUsage, out outputPlug, out outputSubPlugs);

            EffectInputPlug remoteUVInputPlug;
            EffectInputPlug remoteUInputPlug;
            EffectInputPlug remoteVInputPlug;
            CreateUvUVInputBlock(out remoteUVInputPlug, out remoteUInputPlug, out remoteVInputPlug);

            ConnectionManager.Connect(outputPlug, remoteUVInputPlug);
            ConnectionManager.Connect(outputSubPlugs[0], remoteUInputPlug);
            ConnectionManager.Connect(outputSubPlugs[1], remoteVInputPlug);

            Utility.OutputsShouldBeProperlyConnected(textureUsage, outputPlug, outputSubPlugs);
        }

        #endregion

        #region AreInputsProperlyConnected

        [TestMethod]
        public void AreInputsProperlyConnectedUVTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateTextureInputBlock(out textureUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteUVOutputPlug;
            EffectOutputPlug remoteUOutputPlug;
            EffectOutputPlug remoteVOutputPlug;
            CreateUvUVOutputBlock(out remoteUVOutputPlug, out remoteUOutputPlug, out remoteVOutputPlug);

            ConnectionManager.Connect(remoteUVOutputPlug, inputPlug);

            Utility.InputsShouldBeProperlyConnected(textureUsage, inputPlug, inputSubPlugs);
        }

        [TestMethod]
        public void AreInputsProperlyConnectedU_VTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateTextureInputBlock(out textureUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteUVOutputPlug;
            EffectOutputPlug remoteUOutputPlug;
            EffectOutputPlug remoteVOutputPlug;
            CreateUvUVOutputBlock(out remoteUVOutputPlug, out remoteUOutputPlug, out remoteVOutputPlug);

            ConnectionManager.Connect(remoteUOutputPlug, inputSubPlugs[0]);
            ConnectionManager.Connect(remoteVOutputPlug, inputSubPlugs[1]);

            Utility.InputsShouldBeProperlyConnected(textureUsage, inputPlug, inputSubPlugs);
        }

        [TestMethod]
        public void AreInputsProperlyConnectedUTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateTextureInputBlock(out textureUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteUVOutputPlug;
            EffectOutputPlug remoteUOutputPlug;
            EffectOutputPlug remoteVOutputPlug;
            CreateUvUVOutputBlock(out remoteUVOutputPlug, out remoteUOutputPlug, out remoteVOutputPlug);

            ConnectionManager.Connect(remoteUOutputPlug, inputSubPlugs[0]);

            Utility.InputsShouldNotBeProperlyConnected(textureUsage, inputPlug, inputSubPlugs);
        }

        [TestMethod]
        public void AreInputsProperlyConnectedVTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateTextureInputBlock(out textureUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteUVOutputPlug;
            EffectOutputPlug remoteUOutputPlug;
            EffectOutputPlug remoteVOutputPlug;
            CreateUvUVOutputBlock(out remoteUVOutputPlug, out remoteUOutputPlug, out remoteVOutputPlug);

            ConnectionManager.Connect(remoteVOutputPlug, inputSubPlugs[1]);

            Utility.InputsShouldNotBeProperlyConnected(textureUsage, inputPlug, inputSubPlugs);
        }

        [TestMethod]
        public void AreInputsProperlyConnectedUV_UTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateTextureInputBlock(out textureUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteUVOutputPlug;
            EffectOutputPlug remoteUOutputPlug;
            EffectOutputPlug remoteVOutputPlug;
            CreateUvUVOutputBlock(out remoteUVOutputPlug, out remoteUOutputPlug, out remoteVOutputPlug);

            ConnectionManager.Connect(remoteUVOutputPlug, inputPlug);
            ConnectionManager.Connect(remoteUOutputPlug, inputSubPlugs[0]);

            Utility.InputsShouldNotBeProperlyConnected(textureUsage, inputPlug, inputSubPlugs);
        }

        [TestMethod]
        public void AreInputsProperlyConnectedUV_VTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateTextureInputBlock(out textureUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteUVOutputPlug;
            EffectOutputPlug remoteUOutputPlug;
            EffectOutputPlug remoteVOutputPlug;
            CreateUvUVOutputBlock(out remoteUVOutputPlug, out remoteUOutputPlug, out remoteVOutputPlug);

            ConnectionManager.Connect(remoteUVOutputPlug, inputPlug);
            ConnectionManager.Connect(remoteVOutputPlug, inputSubPlugs[1]);

            Utility.InputsShouldNotBeProperlyConnected(textureUsage, inputPlug, inputSubPlugs);
        }

        [TestMethod]
        public void AreInputsProperlyConnectedUV_U_VTest()
        {
            TexCoordUsageDescriptor textureUsage;
            EffectInputPlug inputPlug;
            EffectInputPlug[] inputSubPlugs;
            CreateTextureInputBlock(out textureUsage, out inputPlug, out inputSubPlugs);

            EffectOutputPlug remoteUVOutputPlug;
            EffectOutputPlug remoteUOutputPlug;
            EffectOutputPlug remoteVOutputPlug;
            CreateUvUVOutputBlock(out remoteUVOutputPlug, out remoteUOutputPlug, out remoteVOutputPlug);

            ConnectionManager.Connect(remoteUVOutputPlug, inputPlug);
            ConnectionManager.Connect(remoteUOutputPlug, inputSubPlugs[0]);
            ConnectionManager.Connect(remoteVOutputPlug, inputSubPlugs[1]);

            Utility.InputsShouldNotBeProperlyConnected(textureUsage, inputPlug, inputSubPlugs);
        }

        #endregion
    }
}
