﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using Blocks.Core;
using EffectCombiner.Primitives.Blocks;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.Semantic;
using EffectDefinitions;
using Renderer2D.Core;

namespace EffectCombiner.Primitives.Operations
{
    /// <summary>
    /// ブロックエレメントを置換するコマンドです。
    /// </summary>
    public class ReplaceBlockOperation : OperationBase
    {
        /// <summary>
        /// 元のブロックエレメントの識別子です。
        /// </summary>
        private readonly uint oldInstanceIdentifier;

        /// <summary>
        /// 元のブロックエレメントです。
        /// </summary>
        private readonly EffectBlockElementBase oldBlockElement;

        /// <summary>
        /// 元のブロックエレメントの出力プラグ接続情報です。
        /// </summary>
        private readonly OutputPlugInfo[] oldOutputConnections;

        /// <summary>
        /// 元のブロックエレメントの入力プラグ接続情報です。
        /// </summary>
        private readonly InputPlugInfo[] oldInputConnections;

        /// <summary>
        /// 新しいブロックエレメントのブロック定義データです。
        /// </summary>
        private readonly BlockDefinition newBlockDefinition;

        /// <summary>
        /// 新しいブロックエレメントの位置です。
        /// </summary>
        private IPoint newPosition;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="blockManager">ブロックマネージャ</param>
        /// <param name="blockElement">置換対象のブロックエレメント</param>
        /// <param name="newBlockDef">新しいブロック定義</param>
        public ReplaceBlockOperation(
            BlockManagerBase blockManager,
            EffectBlockElementBase blockElement,
            BlockDefinition newBlockDef)
            : base(OperationType.ReplaceBlock, blockManager)
        {
            if (blockElement == null)
                throw new ArgumentNullException("blockElement");
            if (newBlockDef == null)
                throw new ArgumentNullException("newBlockDef");

            this.oldBlockElement = blockElement;
            this.oldInstanceIdentifier = blockElement.InstanceIdentifier;
            this.newBlockDefinition = newBlockDef;
            this.newPosition = new Point(blockElement.Left, blockElement.Top);

            // 入力接続情報を列挙
            {
                // 入力接続情報を列挙する
                var inputConnections = blockElement.WorkflowItem.InputPlugs  // 入力プラグリストについて
                    .Cast<EffectInputPlug>()                                 // IEnumerable<EffectInputPlug> にキャスト
                    .Where(p => p.BlockDefinitionPlug.Semantic == null)      // セマンティックが未指定のものを列挙して
                    .Select((p, i) => Tuple.Create(p, i))                    // この並びでインデックスを付ける
                    .Where(t => t.Item1.RemoteOutputPlug != null);           // 出力プラグに未接続のものは省く (インデックスにはカウントする)

                // 列挙した入力接続情報を InpugPlugInfo にまとめる
                this.oldInputConnections = inputConnections.Select(t => new InputPlugInfo(
                    t.Item2,
                    new PlugInfo(
                        t.Item1.RemoteOutputPlug.Index,
                        ((EffectWorkflowItem)t.Item1.RemoteWorkflowItem).BlockElement.InstanceIdentifier
                    )
                )).ToArray();
            }

            // 出力接続情報を列挙
            {
                // 出力接続情報を列挙する
                var outputConnections = blockElement.WorkflowItem.OutputPlugs  // 出力プラグリストについて
                    .Cast<EffectOutputPlug>()                                  // IEnumerable<EffectOutputPlug> にキャスト
                    .Where(p => p.Plug.Semantic == null)                       // セマンティックが未指定のものを列挙して
                    .Select((p, i) => Tuple.Create(p, i))                      // この並びでインデックスを付ける
                    .Where(t => t.Item1.RemoteInputPlugs.Length > 0);          // 入力プラグに未接続のものは省く (インデックスにはカウントする)

                // 列挙した出力接続情報を OutputPlugInfo にまとめる
                this.oldOutputConnections = outputConnections.Select(t => new OutputPlugInfo(
                    t.Item2,
                    t.Item1.RemoteInputPlugs.Cast<EffectInputPlug>().Select(r => new PlugInfo(
                        r.Index,
                        r.BlockElement.InstanceIdentifier
                    )).ToArray())
                ).ToArray();
            }
        }

        /// <summary>
        /// 編集操作を実行します。
        /// </summary>
        public override void Execute()
        {
            CompositionBlockManager compositionManager = this.BlockManager as CompositionBlockManager;

            var sourceBlockElement = this.FindBlock(this.oldInstanceIdentifier);
            var targetBlockElement = this.BlockManager.CreateBlockElement(this.newBlockDefinition) as EffectBlockElementBase;

            targetBlockElement.SetNewInstanceIdentifier(this.oldInstanceIdentifier);
            targetBlockElement.SetPosition(newPosition.X, newPosition.Y);

            this.BlockManager.ClearSelections();

            this.BlockManager.ReplaceBlock(sourceBlockElement, targetBlockElement);

            // セマンティックを展開する
            {
                var expandedBlocks = SemanticHelper.ExpandSemantics(this.BlockManager, Generation.Globals.BlockDefinitions.Values, targetBlockElement);

                foreach (var eb in expandedBlocks.Cast<GhostBlockElement>())
                {
                    this.BlockManager.AddBlock(eb);
                }
            }

            compositionManager.BeginShaderCodeGenerationLock();

            // 置換したブロックエレメントについて、置換前の接続状態を復元する
            ConnectionUtility.RestoreInputConnections(targetBlockElement, this.oldInputConnections, this.FindBlock);
            ConnectionUtility.RestoreOutputConnections(targetBlockElement, this.oldOutputConnections, this.FindBlock);

            compositionManager.EndShaderCodeGenerationLock();

            targetBlockElement.IsSelected = true;
        }

        /// <summary>
        /// 編集操作をロールバックします。
        /// </summary>
        public override void Rollback()
        {
            CompositionBlockManager compositionManager = this.BlockManager as CompositionBlockManager;

            var sourceBlockElement = this.FindBlock(this.oldInstanceIdentifier);
            var targetBlockElement = this.oldBlockElement;

            this.BlockManager.ClearSelections();

            compositionManager.BeginShaderCodeGenerationLock();

            compositionManager.ReplaceBlock(sourceBlockElement, targetBlockElement);

            // セマンティックを展開する
            {
                var expandedBlocks = SemanticHelper.ExpandSemantics(this.BlockManager, Generation.Globals.BlockDefinitions.Values, targetBlockElement);

                foreach (var eb in expandedBlocks.Cast<GhostBlockElement>())
                {
                    this.BlockManager.AddBlock(eb);
                }
            }

            // 置換したブロックエレメントについて、置換前の接続状態を復元する
            ConnectionUtility.RestoreInputConnections(targetBlockElement, this.oldInputConnections, this.FindBlock);
            ConnectionUtility.RestoreOutputConnections(targetBlockElement, this.oldOutputConnections, this.FindBlock);

            compositionManager.EndShaderCodeGenerationLock();

            targetBlockElement.IsSelected = true;
        }
    }
}
