﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Text;
using OperationManager.Core;
using Blocks.Core;
using EffectCombiner.Primitives.Generation;

namespace EffectCombiner.Primitives.Operations
{
    public enum OperationType
    {
        AddBlock,
        RemoveBlock,
        ReplaceBlock,
        MoveBlock,
        PasteBlock,
        Connect,
        Disconnect,
        ConnectionChange,
        ConstantValueChange,
        ConstantTypeChange,
        CommentTextChange,
        UniformValueChange,
    }

    public abstract class OperationBase : Operation
    {
        public BlockManagerBase BlockManager { get; private set; }

        protected OperationBase(OperationType displayName, BlockManagerBase blockManager)
            : base(displayName.ToString())
        {
            if (blockManager == null)
                throw new ArgumentNullException("blockManager");

            BlockManager = blockManager;
        }

        protected OperationBase(string displayName)
            : base(displayName)
        {
        }

        public abstract override void Rollback();
        public abstract override void Execute();

        protected EffectBlockElementBase FindBlock(uint instanceIdentifier)
        {
            return FindBlock<EffectBlockElementBase>(instanceIdentifier);
        }

        protected T FindBlock<T>(uint instanceIdentifier) where T : EffectBlockElementBase
        {
            var block = BlockManager.BlockElements
                .FirstOrDefault(b => b.InstanceIdentifier == instanceIdentifier)
                as EffectBlockElementBase;

            if (block == null)
            {
                var message = string.Format(Messages.EXCEPTION_IMPOSSIBLE_TO_FIND_BLOCK, instanceIdentifier);
                throw new InvalidOperationException(message);
            }

            var typedBlock = block as T;
            if (typedBlock == null)
            {
                var message = string.Format(Messages.EXCEPTION_INVALID_TYPE_FOR_BLOCK,
                    instanceIdentifier, block.GetType().FullName, typeof(T).FullName);
                throw new InvalidOperationException(message);
            }

            return typedBlock;
        }
    }
}
