﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Blocks.Core;
using EffectCombiner.Primitives.Generation;
using Workflow.Core;

namespace EffectCombiner.Primitives.Operations
{
    public class ConnectionOperation : OperationBase
    {
        private readonly bool isConnecting;
        private readonly PlugInfo output;
        private readonly PlugInfo input;
        private readonly PlugInfo inputRemote;

        public ConnectionOperation(BlockManagerBase blockManager, bool isConnecting, EffectInputPlug inputPlug, EffectOutputPlug outputPlug)
            : base(isConnecting ? OperationType.Connect : OperationType.Disconnect, blockManager)
        {
            if (inputPlug == null)
                throw new ArgumentNullException("inputPlug");
            if (outputPlug == null)
                throw new ArgumentNullException("outputPlug");

            this.isConnecting = isConnecting;

            output = new PlugInfo(outputPlug.Index, outputPlug.BlockElement.InstanceIdentifier);
            input = new PlugInfo(inputPlug.Index, inputPlug.BlockElement.InstanceIdentifier);

            if (inputPlug.RemoteOutputPlug != null)
            {
                var remoteOutputPlug = (EffectOutputPlug)inputPlug.RemoteOutputPlug;

                if (outputPlug.BlockElement.InstanceIdentifier != remoteOutputPlug.BlockElement.InstanceIdentifier)
                    inputRemote = new PlugInfo(remoteOutputPlug.Index, remoteOutputPlug.BlockElement.InstanceIdentifier);
            }
        }

        public override void Rollback()
        {
            if (isConnecting)
                Disconnect();
            else
                Connect();
        }

        public override void Execute()
        {
            if (isConnecting)
                Connect();
            else
                Disconnect();
        }

        private void Connect()
        {
            var outputBlock = FindBlock(output.InstanceIdentifier);
            var inputBlock = FindBlock(input.InstanceIdentifier);

            var outputPlug = outputBlock.WorkflowItem.OutputPlugs[output.Index];
            var inputPlug = inputBlock.WorkflowItem.InputPlugs[input.Index];

            ConnectionManager.Connect(outputPlug, inputPlug);
        }

        private void Disconnect()
        {
            var inputBlock = FindBlock(input.InstanceIdentifier);
            var inputPlug = inputBlock.WorkflowItem.InputPlugs[input.Index];

            ConnectionManager.Disconnect(inputPlug);

            if (inputRemote != null)
            {
                var outputBlock = FindBlock(inputRemote.InstanceIdentifier);
                var outputPlug = outputBlock.WorkflowItem.OutputPlugs[inputRemote.Index];

                ConnectionManager.Connect(outputPlug, inputPlug);
            }
        }
    }
}
