﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectCombiner.Primitives.Blocks.BlockHitTesters;
using EffectCombiner.Primitives.Generation;
using EffectDefinitions;
using Renderer2D.Core;

namespace EffectCombiner.Primitives.Extensions
{
    public struct PlugGroup
    {
        public Plug Plug;
        public SubPlug[] SubPlugs;
    }

    public static class DefinitionExtensions
    {
        public static PlugGroup[] GetGroupedPlugs(this Plug[] plugs)
        {
            var groups = new List<PlugGroup>();

            for (var i = 0; i < plugs.Length; i++)
            {
                var plug = plugs[i];

                if (plug is SubPlug)
                    continue;

                var subList = new List<SubPlug>();

                for (i++; i < plugs.Length; i++)
                {
                    var subPlug = plugs[i] as SubPlug;
                    if (subPlug == null)
                        break;
                    subList.Add(subPlug);
                }

                groups.Add(new PlugGroup
                {
                    Plug = plug,
                    SubPlugs = subList.ToArray(),
                });
            }

            return groups.ToArray();
        }

        /// <summary>
        /// プラグの表示位置を決定します。
        /// </summary>
        /// <param name="blockDefinition">ブロック定義</param>
        /// <param name="blockSize">ブロックサイズ</param>
        /// <param name="inputs">入力プラグの表示位置</param>
        /// <param name="outputs">出力プラグの表示位置</param>
        public static void GetPlugPositions(this BlockDefinition blockDefinition, ISize blockSize, out IPoint[] inputs, out IPoint[] outputs)
        {
            var top = 0.0;

            top += Globals.VisualResources.RegularBlockTitleTextVerticalPadding +
                   Globals.VisualResources.TitleTextHeight +
                   Math.Max(Globals.VisualResources.RegularBlockTitleTextVerticalPadding, Globals.VisualResources.PlugRadius);

            if (string.IsNullOrEmpty(blockDefinition.SubName) == false)
            {
                top += Globals.VisualResources.TitleTextHeight;
            }

            var inputPlugsList = new List<IPoint>();
            var outputPlugsList = new List<IPoint>();

            var storeTop = top;
            for (var i = 0; i < blockDefinition.InputPlugs.Length; i++)
            {
                inputPlugsList.Add(new Point(/*6.0 +*/ Globals.VisualResources.PlugRadius, top + (Globals.VisualResources.PlugTextHeight / 2.0)));
                top += Math.Max(Globals.VisualResources.PlugTextHeight, 2.0 * Globals.VisualResources.PlugRadius);
            }

            top = storeTop;
            for (var i = 0; i < blockDefinition.OutputPlugs.Length; i++)
            {
                outputPlugsList.Add(new Point(blockSize.Width /*- 6.0*/ - Globals.VisualResources.PlugRadius, top + (Globals.VisualResources.PlugTextHeight / 2.0)));
                top += Math.Max(Globals.VisualResources.PlugTextHeight, 2.0 * Globals.VisualResources.PlugRadius);
            }

            inputs = inputPlugsList.ToArray();
            outputs = outputPlugsList.ToArray();
        }

        public static void SetupPlugPositions(this EffectBlockElementBase blockElement)
        {
            if (blockElement == null)
                throw new ArgumentNullException("blockElement");

            IPoint[] inputPlugCenterPositions;
            IPoint[] outputPlugCenterPositions;
            blockElement.BlockDefinition.GetPlugPositions(new Size(blockElement.Width, blockElement.Height), out inputPlugCenterPositions, out outputPlugCenterPositions);

            blockElement.BlockRenderInfo.InputPlugPositions = inputPlugCenterPositions.Select(p => new System.Drawing.PointF((float)p.X, (float)p.Y)).ToArray();
            blockElement.BlockRenderInfo.OutputPlugPositions = outputPlugCenterPositions.Select(p => new System.Drawing.PointF((float)p.X, (float)p.Y)).ToArray();

            ((BlockElementHitTester)blockElement.BlockHitTester).RecomputeHitTestPaths(blockElement);
        }

        public static string GetSignature(this ShaderTyping.FunctionDefinition func)
        {
            var sb = new StringBuilder();

            sb.AppendFormat("{0};", func.ReturnType);
            sb.Append(string.Join(";", func.Parameters.Select(p => string.Format("[{0}]{1}", p.Direction, p.Type))));

            return sb.ToString();
        }

        public static bool SignatureMatches(this ShaderTyping.FunctionDefinition func, ShaderTyping.FunctionDefinition other)
        {
            if (func == null)
                throw new ArgumentNullException("func");
            if (other == null)
                throw new ArgumentNullException("other");

            return func.GetSignature() == other.GetSignature();
        }

        public static bool IsPolymorphicFunction(this ShaderTyping.FunctionDefinition func)
        {
            if (func == null)
                throw new ArgumentNullException("func");

            if (func.ReturnType.IsVoid)
            {
                if (func.Parameters.Any() == false)
                    return false;
                return func.Parameters.All(p => p.Type.Equals(Generation.Globals.PolymorphicType));
            }

            if (func.ReturnType.Equals(Generation.Globals.PolymorphicType) == false)
                return false;

            if (func.Parameters.Any())
                return func.Parameters.All(p => p.Type.Equals(Generation.Globals.PolymorphicType));

            return true;
        }

        public static ShaderTyping.ParameterDefinition ToTyped(this ShaderTyping.ParameterDefinition param, string type)
        {
            if (param == null)
                throw new ArgumentNullException("param");
            if (string.IsNullOrWhiteSpace(type))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "type"), "type");

            return new ParameterDefinition(null, param.Name, new ShaderTypeDefinition(type), param.Direction, null, null);
        }

        public static ShaderTyping.ParameterDefinition ToPolymorphic(this ShaderTyping.ParameterDefinition param)
        {
            if (param == null)
                throw new ArgumentNullException("param");

            return new ParameterDefinition(null, param.Name, new ShaderTypeDefinition(Generation.Globals.PolymorphicTypeString), param.Direction, null, null);
        }

        public static ShaderTyping.FunctionDefinition ToTyped(this ShaderTyping.FunctionDefinition func, string id, string type)
        {
            if (func == null)
                throw new ArgumentNullException("func");
            if (string.IsNullOrWhiteSpace(id))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "id"), "id");
            if (string.IsNullOrWhiteSpace(type))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "type"), "type");

            return new FunctionDefinition(null, id, func.Name, new ShaderTypeDefinition(type), null, null, null,
                func.Parameters.Select(p => p.ToTyped(type)).Cast<ParameterDefinition>().ToArray());
        }

        public static ShaderTyping.FunctionDefinition ToTyped(this ShaderTyping.FunctionDefinition func, string id)
        {
            if (func == null)
                throw new ArgumentNullException("func");
            if (string.IsNullOrWhiteSpace(id))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "id"), "id");

            return new FunctionDefinition(null, id, func.Name, new ShaderTypeDefinition(Generation.Globals.PolymorphicTypeString), null, null, null,
                func.Parameters.Select(p => p.ToPolymorphic()).Cast<ParameterDefinition>().ToArray());
        }
    }
}
