﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using EffectCombiner.Core.IO;
using EffectCombiner.Generator;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.Semantic;
using EffectDefinitions;
using Renderer2D.Core;
using Renderer2D.Core.WinForms;
using ShaderGenerator.GLSL;
using Workflow.Core;

namespace EffectCombiner.Primitives.Blocks
{
    /// <summary>
    /// プレビューを管理するクラスです。
    /// </summary>
    public class PreviewController
    {
        /// <summary>
        /// CombinerViewer のプロセス名です。
        /// </summary>
        public const string CombinerViewerProcessName = "CombinerViewer";

        /// <summary>
        /// プレビュー出力の型です。
        /// </summary>
        private const string PreviewType = "vec4";

        /// <summary>
        /// ブロックジェネレータです。
        /// </summary>
        private BlockElementGenerator blockGenerator;

        /// <summary>
        /// 定義されているコンバーターです。
        /// </summary>
        private Dictionary<string, BlockDefinition> converterDefinitions;

        /// <summary>
        /// WorkspaceManager です。
        /// </summary>
        private IWorkspaceManager workspaceManager;

        /// <summary>
        /// 送信メッセージのキューです。
        /// </summary>
        private LinkedList<SendMessageInfo> sendMessageQueue;

        /// <summary>
        /// キャプチャのタイムアウトチェックに使うタイマーです。
        /// </summary>
        private Timer timeoutCheckTiemer;

        /// <summary>
        /// Previewer との接続状態の更新に使うタイマーです。
        /// </summary>
        private Timer connectionUpdateTimer;

        /// <summary>
        /// SendMessage の処理中かどうか。
        /// </summary>
        private bool processingSendMessage;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="workspaceManager">WorkspaceManager</param>
        public PreviewController(IWorkspaceManager workspaceManager)
        {
            this.workspaceManager = workspaceManager;
            this.blockGenerator = new BlockElementGenerator();
            this.sendMessageQueue = new LinkedList<SendMessageInfo>();

            this.timeoutCheckTiemer = new Timer();
            this.timeoutCheckTiemer.Interval = 3000;
            this.timeoutCheckTiemer.Tick += this.OnTimeoutExpired;

            this.connectionUpdateTimer = new Timer();
            this.connectionUpdateTimer.Interval = 2000;
            this.connectionUpdateTimer.Tick += this.UpdateConnection;
            this.connectionUpdateTimer.Start();
        }

        /// <summary>
        /// Viewerに送信するメッセージの ID です。
        /// </summary>
        public enum SendMessageType
        {
            /// <summary>
            /// Viewer との通信を確立するためのメッセージ ID です。
            /// </summary>
            Connect = 0,

            /// <summary>
            /// 出力用シェーダを送信するときの ID です。
            /// </summary>
            Output = 1,

            /// <summary>
            /// プレビューシェーダを送信するときの ID です。
            /// </summary>
            Preview = 2,

            /// <summary>
            /// Uniformタグの内容を送信するときの ID です。
            /// </summary>
            Uniform = 3,
        }

        /// <summary>
        /// Viewerから受信するメッセージの種類です。
        /// </summary>
        private enum ReceiveMessageType
        {
            /// <summary>
            /// viewer 初期化メッセージです。
            /// </summary>
            ViewerInitialized = 1,

            /// <summary>
            /// プレビュー画像のパスを受信したときのメッセージ ID です。
            /// </summary>
            PreviewPicturePath = 2,

            /// <summary>
            /// テクスチャーリロード完了メッセージです。
            /// </summary>
            TextureReloaded = 3,

            /// <summary>
            /// プレビューシェーダリロードメッセージです。
            /// </summary>
            PreviewShaderReloaded = 4,

            /// <summary>
            /// 出力シェーダリロードメッセージです。
            /// </summary>
            OutputShaderReloaded = 5,

            /// <summary>
            /// テクスチャリセットメッセージです。
            /// </summary>
            TextureReset = 6,

            /// <summary>
            /// 列挙体の定義数です。
            /// </summary>
            EnumSize = 7
        }

        /// <summary>
        /// Previewer との接続状態が変わったときに発生するイベントです。
        /// </summary>
        public event EventHandler ConnectionChanged;

        /// <summary>
        /// プレビューの MainWindowHandle を取得または設定します。
        /// </summary>
        public static IntPtr EmbededHandle { get; set; }

        /// <summary>
        /// 埋め込まれたプレビューウィンドウのプロセスを取得または設定します。
        /// </summary>
        public static Process EmbededProcess { get; set; }

        /// <summary>
        /// ブロック定義情報を取得または設定します。
        /// </summary>
        public EffectDefinitionsContainer DefinitionContainer { get; set; }

        /// <summary>
        /// Renderer を取得または設定します。
        /// </summary>
        public Renderer Renderer { get; set; }

        /// <summary>
        /// 接続中の Previewer のプロセスを取得します。
        /// </summary>
        public Process PreviewerProcess { get; private set; }

        /// <summary>
        /// 接続中の Previewer のウィンドウハンドルを取得します。
        /// </summary>
        public IntPtr PreviewerWindowHandle { get; private set; }

        /// <summary>
        /// Previewer に接続中かどうか取得します。
        /// </summary>
        public bool IsConnected { get; private set; }

        #region SetUniformValues

        /// <summary>
        /// uniform 変数の値を設定します。
        /// </summary>
        /// <param name="blockManager">ブロックマネージャー</param>
        /// <param name="uniforms">uniform 変数情報</param>
        public void SetUniformValues(CompositionBlockManager blockManager, IEnumerable<KeyValuePair<string, UniformData>> uniforms)
        {
            if (this.IsConnected == false)
            {
                return;
            }

            foreach (var uniform in uniforms)
            {
                string name = uniform.Key;
                string type = uniform.Value.Type;

                byte[] messageData;

                // 送信データを作成
                using (MemoryStream stream = new MemoryStream())
                using (BinaryWriter writer = new BinaryWriter(stream))
                {
                    writer.Write((int)SendMessageType.Uniform);
                    writer.Write(name.Length + 1);
                    writer.Write((name + '\0').ToCharArray(), 0, name.Length + 1);
                    writer.Write(type.Length + 1);
                    writer.Write((type + '\0').ToCharArray(), 0, type.Length + 1);

                    if (uniform.Value is UniformDataFile)
                    {
                        var data = (UniformDataFile)uniform.Value;
                        writer.Write(data.Value.Length + 1);
                        writer.Write((data.Value + '\0').ToCharArray(), 0, data.Value.Length + 1);
                    }
                    else if (uniform.Value is UniformDataFloat)
                    {
                        var data = (UniformDataFloat)uniform.Value;
                        writer.Write(data.Value);
                    }
                    else if (uniform.Value is UniformDataVec2)
                    {
                        var data = (UniformDataVec2)uniform.Value;
                        writer.Write(data.Value[0]);
                        writer.Write(data.Value[1]);
                    }
                    else if (uniform.Value is UniformDataVec3)
                    {
                        var data = (UniformDataVec3)uniform.Value;
                        writer.Write(data.Value[0]);
                        writer.Write(data.Value[1]);
                        writer.Write(data.Value[2]);
                    }
                    else if (uniform.Value is UniformDataVec4)
                    {
                        var data = (UniformDataVec4)uniform.Value;
                        writer.Write(data.Value[0]);
                        writer.Write(data.Value[1]);
                        writer.Write(data.Value[2]);
                        writer.Write(data.Value[3]);
                    }

                    messageData = stream.ToArray();
                }

                // 送信メッセージを作成してキューに追加する
                {
                    SendMessageInfo message = new SendMessageInfo()
                    {
                        SendType = SendMessageType.Uniform,
                        Data = messageData
                    };

                    if (uniform.Value is UniformDataFile)
                    {
                        var data = (UniformDataFile)uniform.Value;

                        if (string.IsNullOrEmpty(data.Value) == false)
                        {
                            message.ReplyMessages = new List<ReceiveMessageType>()
                            {
                                ReceiveMessageType.TextureReloaded
                            };
                        }
                        else
                        {
                            message.ReplyMessages = new List<ReceiveMessageType>()
                            {
                                ReceiveMessageType.TextureReset
                            };
                        }
                    }

                    this.sendMessageQueue.AddLast(message);
                    Debug.Print("Enqueue: " + message.SendType + ":" + name);
                }
            }

            this.TriggerSendMessage();

            foreach (var name in uniforms.Select(p => p.Key))
            {
                // 変更された uniform 変数を持つブロックのプレビューを更新する
                foreach (var blockElement in blockManager.BlockElements)
                {
                    var uniform = blockElement.BlockDefinition.Uniform;

                    if (uniform != null && uniform.Name == name)
                    {
                        this.UpdatePreviewImage((EffectBlockElementBase)blockElement, false);
                    }
                }
            }
        }

        #endregion

        #region UpdateOutputPreview

        /// <summary>
        /// 最終出力プレビューを更新します。
        /// </summary>
        /// <param name="blockManager">BlockManager</param>
        public void UpdateOutputPreview(CompositionBlockManager blockManager)
        {
            if (this.IsConnected == false)
            {
                return;
            }

            // 重複するメッセージを削除しておく
            {
                var msgNode = this.sendMessageQueue.First;

                while (msgNode != null)
                {
                    if (msgNode.Value.IsSent)
                    {
                        msgNode = msgNode.Next;
                        continue;
                    }

                    if (msgNode.Value.SendType == SendMessageType.Output)
                    {
                        this.sendMessageQueue.Remove(msgNode);
                        break;
                    }

                    msgNode = msgNode.Next;
                }
            }

            SendMessageInfo message = new SendShaderCodeMessageInfo()
            {
                SendType = SendMessageType.Output,
                ReplyMessages = new List<ReceiveMessageType>()
                {
                    ReceiveMessageType.OutputShaderReloaded
                },
                PreSendMethod = this.PreSendOutputMessage,
                BlockManager = blockManager
            };

            this.sendMessageQueue.AddLast(message);
            Debug.Print("Enqueue: Output");

            this.TriggerSendMessage();
        }

        /// <summary>
        /// Output メッセージを送信する前の処理を行います。
        /// </summary>
        /// <param name="message">送信メッセージ情報</param>
        /// <returns>出力シェーダを生成できたかどうか</returns>
        private bool PreSendOutputMessage(SendMessageInfo message)
        {
            SendShaderCodeMessageInfo info = (SendShaderCodeMessageInfo)message;

            var blockManager = info.BlockManager;

            // シェーダコードを生成 (Previewer に送るシェーダコードのメイン関数名は "main" に固定)
            string[] shaderCode = blockManager.ProduceShaderCodesSync(true, true, "main");

            // 新しく Previewer に接続したとき等に表示をリセットするため、
            // 出力シェーダが生成できない場合はシェーダリセット用のメッセージを作る
            if (shaderCode.Length < 1)
            {
                using (MemoryStream stream = new MemoryStream())
                using (BinaryWriter writer = new BinaryWriter(stream))
                {
                    writer.Write((int)SendMessageType.Output);
                    writer.Write((string.Empty + '\0').ToCharArray(), 0, string.Empty.Length + 1);
                    info.Data = stream.ToArray();
                }

                return true;
            }

            string outputShaderPath = null;

            // シェーダコードをファイルに出力
            {
                outputShaderPath = Path.Combine(IOConstants.AppDataWorkPath, "output.fsh");

                File.WriteAllText(outputShaderPath, shaderCode[0]);
            }

            // 最終出力プレビュー表示用の送信メッセージを作成
            using (MemoryStream stream = new MemoryStream())
            using (BinaryWriter writer = new BinaryWriter(stream))
            {
                writer.Write((int)SendMessageType.Output);
                writer.Write((outputShaderPath + '\0').ToCharArray(), 0, outputShaderPath.Length + 1);

                info.Data = stream.ToArray();
            }

            return true;
        }

        #endregion

        /// <summary>
        /// ウィンドウメッセージを処理します。
        /// このメソッドでプレビューイメージを更新するので、メインフォーム等から呼び出してください。
        /// </summary>
        /// <param name="msg">メッセージ情報</param>
        public void ProcessWindowMessage(ref Message msg)
        {
            if (msg.Msg != WM_COPYDATA)
            {
                return;
            }

            COPYDATASTRUCT copyData = (COPYDATASTRUCT)Marshal.PtrToStructure(msg.LParam, typeof(COPYDATASTRUCT));

            ReceiveMessageType messageType = (ReceiveMessageType)copyData.dwData;

            // メッセージタイプをチェック
            if (messageType < 0 || messageType >= ReceiveMessageType.EnumSize)
            {
                Debug.Print("Invalid message received: {0}", copyData.dwData);
                return;
            }

            // 受信待ちのメッセージがあるかチェック
            if (this.sendMessageQueue.Any() == false)
            {
                Debug.Print("Invalid message received: {0}", copyData.dwData);
                return;
            }

            SendMessageInfo message = this.sendMessageQueue.First.Value;

            // メッセージが期待したものかチェック
            if (message.ReplyMessages.Contains(messageType) == false)
            {
                Debug.Print("Invalid message received: {0}", copyData.dwData);
                return;
            }

            // プレビュー画像パスを受信したとき、プレビュー画像を更新する
            if (messageType == ReceiveMessageType.PreviewPicturePath && message.SendType == SendMessageType.Preview)
            {
                SendShaderCodeMessageInfo info = (SendShaderCodeMessageInfo)message;

                var data = new byte[copyData.cbData];
                Marshal.Copy(copyData.lpData, data, 0, (int)copyData.cbData);

                string filePath = System.Text.Encoding.UTF8.GetString(data);
                filePath = filePath.Substring(0, filePath.IndexOf('\0'));

                if (string.IsNullOrEmpty(filePath) == false)
                {
                    using (Image image = Image.FromFile(filePath))
                    using (Bitmap bitmap = new Bitmap(image))
                    {
                        info.BlockElement.BlockRenderInfo.PreviewImage =
                            this.Renderer.CreateBitmap(bitmap.ToRendererBitmapDefinition());
                    }
                }
                else
                {
                    info.BlockElement.BlockRenderInfo.PreviewImage = Globals.VisualResources.PreviewErrorImageBitmap;
                }

                this.workspaceManager.InvalidateRender();
            }
            // ビューアの初期化完了メッセージを受信したとき、接続状態を更新する
            else if (messageType == ReceiveMessageType.ViewerInitialized)
            {
                if (this.IsConnected == false)
                {
                    this.IsConnected = true;

                    this.OnConnectionChanged(EventArgs.Empty);
                }
            }

            // 受信待ちメッセージリストを更新
            message.ReplyMessages.Remove(messageType);

            // メッセージキューを更新する
            // TriggerSendMessage の SendMessage 中に Viewer から SendMessage で帰ってきた場合、
            // TriggerSendMessage の方でメッセージキューの更新処理を行います。
            if (this.processingSendMessage == false)
            {
                // 受信待ちのメッセージがなくなっとき、送信メッセージをキューから取り除く
                if (message.ReplyMessages.Any() == false)
                {
                    this.sendMessageQueue.RemoveFirst();
                    Debug.Print("Dequeue (OK): " + message.SendType);

                    this.timeoutCheckTiemer.Stop();
                }
                else
                {
                    // 受信待ちのメッセージが残っているとき、次のメッセージを待つためタイマーをリセットする
                    this.timeoutCheckTiemer.Stop();
                    this.timeoutCheckTiemer.Start();
                }

                this.TriggerSendMessage();
            }
        }

        #region UpdatePreviewImage

        /// <summary>
        /// Viewerと連携してブロックのプレビューを更新します。
        /// blockElementをOutputノードにエッジを繋ぎ、Viewerでプレビュー画像を作成します。
        /// 画像の取得はProcessWindowMessageで行います。
        /// </summary>
        /// <param name="blockElement">プレビューを配置するブロック</param>
        /// <param name="updateOutputShader">出力シェーダを送信するかどうか</param>
        public void UpdatePreviewImage(EffectBlockElementBase blockElement, bool updateOutputShader)
        {
            if (blockElement == null)
            {
                return;
            }

            if (this.IsConnected == false)
            {
                return;
            }

            this.UpdatePreviewInitialize();

            CompositionBlockManager blockManager = (CompositionBlockManager)blockElement.BlockManager;

            // 出力ノードの場合はそのままシェーダを作成する
            if (blockElement.BlockDefinition.Tags.FirstOrDefault() == "Output")
            {
                this.UpdateOutputPreview(blockManager);
                return;
            }

            EffectBlockElementBase previewBlock = null;  // blockElementのクローン
            EffectBlockElementBase outputBlockElement = null;
            var copyBlockElements = new Dictionary<uint, EffectBlockElementBase>();
            var copyBlockInfo = new Dictionary<EffectBlockElementBase, EffectBlockElementBase>();

            // ワークスペースの内容を複製
            {
                // ブロックデータをコピーするときに勝手にシェーダコードが生成されないようにロック
                blockManager.BeginShaderCodeGenerationLock();

                // ブロックデータを複製
                foreach (var sourceBlock in blockElement.BlockManager.BlockElements)
                {
                    var copyBlock = this.blockGenerator.CreateBlockElement(sourceBlock.BlockDefinition);

                    if (copyBlock is EffectBlockElementBase)
                    {
                        if (!copyBlockElements.ContainsKey(sourceBlock.InstanceIdentifier))
                        {
                            copyBlockElements.Add(sourceBlock.InstanceIdentifier, (EffectBlockElementBase)copyBlock);
                            copyBlockInfo.Add((EffectBlockElementBase)copyBlock, (EffectBlockElementBase)sourceBlock);
                        }

                        if (sourceBlock == blockElement)
                        {
                            previewBlock = (EffectBlockElementBase)copyBlock;
                        }

                        if (copyBlock is ConstantBlockElement)
                        {
                            var sourceConstBlock = (ConstantBlockElement)sourceBlock;
                            var copyConstBlock = (ConstantBlockElement)copyBlock;

                            var shaderType =
                                (ShaderTypeDefinition)
                                sourceConstBlock.BlockDefinition.CurrentFunctionDefinition.ReturnType;
                            var constValue = Core.CoreUtility.MultiDimArrayToString(sourceConstBlock.Values);

                            copyConstBlock.SetupConstantBlockElement(shaderType.TypeString, constValue);
                        }

                        if (copyBlock is UserDefinitionBlockElement)
                        {
                            var sourceUserBlock = (UserDefinitionBlockElement)sourceBlock;
                            var copyUserBlock = (UserDefinitionBlockElement)copyBlock;

                            copyUserBlock.CopyUserDefinitionBlockElement(copyUserBlock);
                        }
                    }
                }

                // プラグデータを複製
                var inputPlugs = blockElement.BlockManager.BlockElements
                    .Cast<EffectBlockElementBase>()
                    .SelectMany(b => b.WorkflowItem.InputPlugs.Cast<EffectInputPlug>())
                    .Where(p => p.RemoteOutputPlug != null);

                if (inputPlugs != null)
                {
                    foreach (var plug in inputPlugs)
                    {
                        var sourceBlockId = ((EffectOutputPlug)plug.RemoteOutputPlug).BlockElement.InstanceIdentifier;
                        var sourcePlugIdx = plug.RemoteOutputPlug.Index;

                        var targetBlockId = plug.BlockElement.InstanceIdentifier;
                        var targetPlugIdx = plug.Index;

                        var sourceBlock = copyBlockElements[sourceBlockId];
                        var targetBlock = copyBlockElements[targetBlockId];

                        if (sourceBlock.WorkflowItem.OutputPlugs.Length == 0 ||
                            targetBlock.WorkflowItem.InputPlugs.Length == 0)
                        {
                            return;
                        }

                        ConnectionManager.Connect(
                            sourceBlock.WorkflowItem.OutputPlugs[sourcePlugIdx],
                            targetBlock.WorkflowItem.InputPlugs[targetPlugIdx]);
                    }
                }

                // 出力ブロックエレメントを取得
                {
                    // 出力ブロック定義を取得
                    var outputBlockDefinitions = this.DefinitionContainer.Blocks
                        .Where(p => p.Tags.Any(q => q == "Output"))
                        .Where(p => p.InputPlugs.Count() == 1)
                        .Where(p => p.InputPlugs[0].Type.TypeString == PreviewType).ToArray();

                    // 出力ブロックエレメントを検索
                    foreach (var copyBlockElement in copyBlockElements.Values)
                    {
                        if (outputBlockDefinitions.Any(x => x == copyBlockElement.BlockDefinition))
                        {
                            outputBlockElement = copyBlockElement;
                            break;
                        }
                    }

                    // 存在しなければ出力ブロックエレメントを作成
                    if (outputBlockElement == null)
                    {
                        outputBlockElement = this.blockGenerator.CreateBlockElement(outputBlockDefinitions[0]) as EffectBlockElementBase;
                        copyBlockElements.Add(outputBlockElement.InstanceIdentifier, outputBlockElement);
                    }
                }

                blockManager.EndShaderCodeGenerationLock();
            }

            // 処理対象のブロックエレメントを探索
            var previewBlockElements = this.SearchConnectedPreviewBlocks(previewBlock);

            // 重複するメッセージを削除しておく
            foreach (var previewBlockElement in previewBlockElements)
            {
                var sourceBlockElement = copyBlockInfo[previewBlockElement];

                var msgNode = this.sendMessageQueue.First;

                while (msgNode != null)
                {
                    if (msgNode.Value.IsSent)
                    {
                        msgNode = msgNode.Next;
                        continue;
                    }

                    if (msgNode.Value.SendType == SendMessageType.Preview)
                    {
                        SendShaderCodeMessageInfo info = (SendShaderCodeMessageInfo)msgNode.Value;

                        if (info.BlockElement == sourceBlockElement)
                        {
                            this.sendMessageQueue.Remove(msgNode);
                            break;
                        }
                    }

                    msgNode = msgNode.Next;
                }
            }

            // 送信メッセージを作成してキューに追加
            foreach (var previewBlockElement in previewBlockElements)
            {
                SendMessageInfo message = new SendShaderCodeMessageInfo()
                {
                    SendType = SendMessageType.Preview,
                    ReplyMessages = new List<ReceiveMessageType>()
                    {
                        ReceiveMessageType.PreviewShaderReloaded,
                        ReceiveMessageType.PreviewPicturePath
                    },
                    PreSendMethod = this.PreSendPreviewMessage,
                    BlockManager = blockManager,
                    BlockElement = copyBlockInfo[previewBlockElement],
                    CopyBlockElement = previewBlockElement,
                    OutputBlockElement = outputBlockElement,
                    WorkspaceBlockElements = copyBlockElements
                };

                this.sendMessageQueue.AddLast(message);
                Debug.Print("Enqueue: " + previewBlockElement.BlockDefinition.Name);
            }

            if (updateOutputShader)
            {
                this.UpdateOutputPreview((CompositionBlockManager)blockElement.BlockManager);
            }

            this.TriggerSendMessage();
        }

        /// <summary>
        /// Preview メッセージを送信する前の処理を行います。
        /// </summary>
        /// <param name="message">送信メッセージ情報</param>
        /// <returns>プレビューシェーダを作成できたかどうか</returns>
        private bool PreSendPreviewMessage(SendMessageInfo message)
        {
            SendShaderCodeMessageInfo info = (SendShaderCodeMessageInfo)message;

            string[] shaderCode;

            // シェーダコードを生成
            {
                // ブロックを繋げるするときに勝手にシェーダコードが生成されないようにロック
                info.BlockManager.BeginShaderCodeGenerationLock();

                EffectBlockElementBase previewBlockElement = info.CopyBlockElement;
                EffectBlockElementBase outputBlockElement = info.OutputBlockElement;
                Dictionary<uint, EffectBlockElementBase> workspaceBlockElements = info.WorkspaceBlockElements;

                var connectOutput = this.GetConnectPreviewBlock(previewBlockElement);

                if (connectOutput == null)
                {
                    info.BlockElement.BlockRenderInfo.PreviewImage = Globals.VisualResources.PreviewErrorImageBitmap;
                    return false;
                }

                workspaceBlockElements.Add(connectOutput.InstanceIdentifier, connectOutput);

                // プレビューブロックを出力ブロックに接続
                OutputPlug<PlugValue> outputPlug = connectOutput.WorkflowItem.OutputPlugs.FirstOrDefault();
                InputPlug<PlugValue> inputPlug = outputBlockElement.WorkflowItem.InputPlugs.FirstOrDefault();
                ConnectionManager.Connect(outputPlug, inputPlug);

                var workflowItems = new List<EffectWorkflowItem>();
                var definitions = this.DefinitionContainer.Blocks.ToDictionary(p => p.Guid);
                foreach (var element in workspaceBlockElements.Values)
                {
                    workflowItems.Add(element.WorkflowItem);

                    workflowItems.AddRange(SemanticHelper.ExpandSemantics(this.blockGenerator, definitions.Values, element)
                        .Cast<EffectBlockElementBase>()
                        .Select(b => b.WorkflowItem));
                }

                var contentRepository = Globals.ContentRepository as CustomContentRepository;
                var genSettings = new ShaderGenSettings();
                foreach (var path in Globals.Options.DefinitionPaths.Paths)
                {
                    genSettings.EffectDefinitionsPaths.Add(path.Path, SearchOption.AllDirectories, "*.glsl", "*.edml");
                }

                // シェーダコードを生成 (Viewer に送るシェーダコードのメイン関数名は "main" に固定)
                shaderCode = CodeGenerator.GetShaderCodes(workflowItems, contentRepository, genSettings, "main");

                if (previewBlockElement != connectOutput)
                {
                    this.DissconnectPreviewBlock(previewBlockElement, connectOutput);
                    this.DissconnectPreviewBlock(connectOutput, outputBlockElement);
                }
                else
                {
                    this.DissconnectPreviewBlock(previewBlockElement, outputBlockElement);
                }

                workspaceBlockElements.Remove(connectOutput.InstanceIdentifier);

                info.BlockManager.EndShaderCodeGenerationLock();
            }

            if (shaderCode.Length == 0)
            {
                info.BlockElement.BlockRenderInfo.PreviewImage = Globals.VisualResources.PreviewErrorImageBitmap;
                return false;
            }

            string previewShaderPath = null;

            // シェーダコードをファイルに出力
            {
                previewShaderPath = Path.Combine(IOConstants.AppDataWorkPath, "preview.fsh");

                File.WriteAllText(previewShaderPath, shaderCode[0]);
            }

            // プレビュー表示用のメッセージデータを作成
            using (MemoryStream stream = new MemoryStream())
            using (BinaryWriter writer = new BinaryWriter(stream))
            {
                writer.Write((int)SendMessageType.Preview);
                writer.Write((previewShaderPath + '\0').ToCharArray(), 0, previewShaderPath.Length + 1);

                info.Data = stream.ToArray();
            }

            return true;
        }

        #endregion

        /// <summary>
        /// プレビュー更新の初期化処理を行います。
        /// </summary>
        private void UpdatePreviewInitialize()
        {
            if (this.converterDefinitions == null)
            {
                this.converterDefinitions = new Dictionary<string, BlockDefinition>();
                foreach (
                    var conv in
                    this.DefinitionContainer.Blocks.Cast<BlockDefinition>().Where(p => p.Tags.Contains("Convert")))
                {
                    if (conv.CurrentFunctionDefinition.Name == "ConvFloatToV4_1")
                    {
                        this.converterDefinitions.Add("float", conv);
                    }

                    if (conv.CurrentFunctionDefinition.Name == "ConvV2ToV4_01")
                    {
                        this.converterDefinitions.Add("vec2", conv);
                    }

                    if (conv.CurrentFunctionDefinition.Name == "ConvV3ToV4_1")
                    {
                        this.converterDefinitions.Add("vec3", conv);
                    }
                }
            }
        }

        /// <summary>
        /// 指定されたブロックに繋がれているプレビュー表示機能を持ったブロックの一覧を探索します。
        /// 結果は幅優先探索で探索した順番になります。
        /// </summary>
        /// <param name="blockElement">探索対象のブロックエレメント</param>
        /// <returns>探索して見つかったブロックエレメントの一覧を返します。</returns>
        private List<EffectBlockElementBase> SearchConnectedPreviewBlocks(EffectBlockElementBase blockElement)
        {
            Queue<EffectBlockElementBase> searcher = new Queue<EffectBlockElementBase>();
            List<EffectBlockElementBase> findElements = new List<EffectBlockElementBase>();

            searcher.Enqueue(blockElement);

            // 先に探索対象のブロックがプレビュー表示機能を持っているか見ておく
            if (blockElement.BlockDefinition.Preview.IsEnabled)
            {
                findElements.Add(blockElement);
            }

            // ブロックを幅優先で探索
            while (searcher.Any())
            {
                var queue = searcher.Dequeue();

                // 出力ブラグに繋がれている全てのブロックについて処理する
                foreach (var element in queue.WorkflowItem.OutputPlugs.Cast<EffectOutputPlug>())
                {
                    foreach (var remoteInput in element.RemoteInputPlugs.Cast<EffectInputPlug>())
                    {
                        var remoteElement = remoteInput.BlockElement;

                        // プレビュー表示機能を持っているブロックをリストに追加
                        if (remoteElement.BlockDefinition.Preview.IsEnabled)
                        {
                            if (findElements.Contains(remoteElement) == false)
                            {
                                findElements.Add(remoteElement);
                            }
                        }

                        // 未探索のブロックを探索キューに追加
                        if (searcher.Contains(remoteElement) == false)
                        {
                            searcher.Enqueue(remoteElement);
                        }
                    }
                }
            }

            return findElements;
        }

        /// <summary>
        /// ブロックとの接続を切り離します。
        /// </summary>
        /// <param name="inputBlock">入力側のブロックエレメント</param>
        /// <param name="outputBlock">出力側のブロックエレメント</param>
        private void DissconnectPreviewBlock(EffectBlockElementBase inputBlock, EffectBlockElementBase outputBlock)
        {
            foreach (var outputPlug in outputBlock.WorkflowItem.OutputPlugs.Cast<EffectOutputPlug>())
            {
                foreach (var inputPlug in outputPlug.RemoteInputPlugs.Cast<EffectInputPlug>())
                {
                    if (inputPlug.BlockElement == outputBlock)
                    {
                        ConnectionManager.Disconnect(inputPlug);
                    }
                }
            }
        }

        /// <summary>
        /// 指定されたブロックに適切なコンバータを追加して Output ブロックに繋げられるようにします。
        /// </summary>
        /// <param name="blockElement">ブロック</param>
        /// <returns>出力に繋ぐブロックを返します。</returns>
        private EffectBlockElementBase GetConnectPreviewBlock(EffectBlockElementBase blockElement)
        {
            string typeString = blockElement.BlockDefinition.OutputPlugs[0].Type.TypeString;

            // Math関数用(出力タイプが"*"のとき、変換する)
            if (typeString == Primitives.Generation.Globals.PolymorphicTypeString)
            {
                var polymorphicElement = blockElement as PolymorphicBlockElement;

                var shaderType = polymorphicElement.GetPossibleTypes(polymorphicElement.WorkflowItem, false, 0);

                foreach (var type in shaderType)
                {
                    if (this.converterDefinitions.ContainsKey(type.ToString()) || type.ToString() == PreviewType)
                    {
                        typeString = type.ToString();
                    }
                }
            }

            // Output に直接繋げられるブロックはそのまま返す
            if (typeString == PreviewType)
            {
                return blockElement;
            }

            // Converter を使っても変換できないときは null を返す
            if (this.converterDefinitions.ContainsKey(typeString) == false)
            {
                return null;
            }

            // Output ブロックに繋げられるように Converter を配置する
            EffectBlockElementBase converterElement =
                this.blockGenerator.CreateBlockElement(this.converterDefinitions[typeString]) as EffectBlockElementBase;

            var converterInput = converterElement.WorkflowItem.InputPlugs.Cast<InputPlug<PlugValue>>().FirstOrDefault();
            OutputPlug<PlugValue> outputPlug = blockElement.WorkflowItem.OutputPlugs.Cast<EffectOutputPlug>().FirstOrDefault();

            var connectability = ConnectionUtility.DetermineConnectability(blockElement, 0, converterElement, 0);

            if (connectability != Connectability.Invalid)
            {
                ConnectionManager.Connect(outputPlug, converterInput);
            }

            return converterElement;
        }

        /// <summary>
        /// タイムアウトが発生したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnTimeoutExpired(object sender, EventArgs e)
        {
            if (this.sendMessageQueue.Any() == false)
            {
                return;
            }

            SendMessageInfo firstMessage = this.sendMessageQueue.First.Value;

            // タイムアウトが発生する直前にリクエストが処理された場合、滑り込みセーフ
            if (firstMessage.ReplyMessages != null && firstMessage.ReplyMessages.Any() == false)
            {
                return;
            }

            // タイムアウトが発生したとき、キューに溜まっているリクエストを全てエラーにする
            while (this.sendMessageQueue.Any())
            {
                SendMessageInfo message = this.sendMessageQueue.First.Value;
                this.sendMessageQueue.RemoveFirst();

                if (message.SendType == SendMessageType.Connect)
                {
                    Debug.Print("Dequeue (NG): Connect");

                    // メッセージを再送信するために Previewer の情報をリセットする
                    this.PreviewerProcess = null;
                    this.PreviewerWindowHandle = IntPtr.Zero;
                }
                else if (message.SendType == SendMessageType.Preview)
                {
                    var msg = (SendShaderCodeMessageInfo)message;

                    Debug.Print("Dequeue (NG): Preview: " + msg.BlockElement.BlockDefinition.Name);

                    msg.BlockElement.BlockRenderInfo.PreviewImage = Globals.VisualResources.PreviewErrorImageBitmap;
                }
                else
                {
                    Debug.Print("Dequeue (NG): " + message.SendType );
                }
            }

            this.workspaceManager.InvalidateRender();
        }

        /// <summary>
        /// Previewer との通信状態を更新します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void UpdateConnection(object sender, EventArgs e)
        {
            Process process = null;
            IntPtr windowHandle = IntPtr.Zero;

            // 埋め込まれている Previewer があれば、そちらを優先的に使う
            if (EmbededProcess != null && EmbededProcess.HasExited == false)
            {
                // Previewer 起動時に Floating していると IntPtr.Zero になる
                if (EmbededHandle != IntPtr.Zero)
                {
                    process = EmbededProcess;
                    windowHandle = EmbededHandle;

                    // プロセス ID が変わらず、プロセスインスタンスだけ変わったときは
                    // 再接続しないように元のインスタンスを使う
                    if (this.PreviewerProcess != null && EmbededProcess.Id == this.PreviewerProcess.Id)
                    {
                        process = this.PreviewerProcess;
                    }
                }
            }

            // 埋め込まれている Previewer がなければ、適宜 Previewer を検索する
            if (windowHandle == IntPtr.Zero)
            {
                process = this.PreviewerProcess;

                if (process == null || process.HasExited)
                {
                    process = Process.GetProcessesByName(CombinerViewerProcessName).FirstOrDefault();
                }

                if (process != null && process.HasExited == false)
                {
                    windowHandle = process.MainWindowHandle;
                }
            }

            // Previewer のウィンドウハンドルが埋め込み機能によって変わったときは
            // 再接続しないようにウィンドウハンドルを更新する
            if (process == this.PreviewerProcess && windowHandle != this.PreviewerWindowHandle &&
                windowHandle != IntPtr.Zero && this.PreviewerWindowHandle != IntPtr.Zero)
            {
                this.PreviewerWindowHandle = windowHandle;
            }

            // プロセスかウィンドウハンドルが変わったとき、接続状態を更新
            if (process != this.PreviewerProcess || windowHandle != this.PreviewerWindowHandle)
            {
                // 送信メッセージキューを空にしておく
                if (this.sendMessageQueue.Any())
                {
                    var firstMessage = this.sendMessageQueue.First.Value;

                    this.sendMessageQueue.Clear();

                    // SendMessage の処理中は最初のメッセージだけ残しておく
                    if (this.processingSendMessage)
                    {
                        this.sendMessageQueue.AddFirst(firstMessage);
                    }
                    else
                    {
                        this.timeoutCheckTiemer.Stop();
                    }
                }

                this.PreviewerProcess = process;
                this.PreviewerWindowHandle = windowHandle;

                // 接続状態の変更を通知
                if (this.IsConnected)
                {
                    this.IsConnected = false;

                    this.OnConnectionChanged(EventArgs.Empty);
                }

                // 接続確立用のメッセージを作成してキューに追加
                if (windowHandle != IntPtr.Zero)
                {
                    SendMessageInfo message = new SendMessageInfo()
                    {
                        SendType = SendMessageType.Connect,
                        ReplyMessages = new List<ReceiveMessageType>()
                        {
                            ReceiveMessageType.ViewerInitialized
                        }
                    };

                    // 接続確立用のメッセージデータを作成
                    using (MemoryStream stream = new MemoryStream())
                    using (BinaryWriter writer = new BinaryWriter(stream))
                    {
                        writer.Write((int)SendMessageType.Connect);
                        writer.Write((int)Process.GetCurrentProcess().MainWindowHandle);

                        message.Data = stream.ToArray();
                    }

                    this.sendMessageQueue.AddFirst(message);
                    Debug.Print("Enqueue: Connect");

                    this.TriggerSendMessage();
                }
            }
        }

        /// <summary>
        /// 送信メッセージキューの状態をチェックして、メッセージが送信可能な状態であればメッセージの送信を開始します。
        /// </summary>
        private void TriggerSendMessage()
        {
            // Viewer から直に SendMessage が帰ってきて、さらにその中で TriggerSendMessage を呼び出したときは即リターン
            if (this.processingSendMessage)
            {
                return;
            }

            for (;;)
            {
                // メッセージが存在するかチェック
                if (this.sendMessageQueue.Any() == false)
                {
                    break;
                }

                SendMessageInfo message = this.sendMessageQueue.First.Value;

                // メッセージが送信済みでないかチェック
                if (message.IsSent)
                {
                    break;
                }

                bool resPreSend = true;

                // 送信前に行うメッセージ処理を実行
                if (message.PreSendMethod != null)
                {
                    resPreSend = message.PreSendMethod(message);

                    if (resPreSend == false)
                    {
                        this.sendMessageQueue.RemoveFirst();
                        Debug.Print("Dequeue (NG): " + message.SendType);
                        continue;
                    }
                }

                // メッセージを送信
                // Viewer から直に SendMessage が帰ってきた場合、そのまま ProcessWindowMessage の中にまで入ることがあります。
                {
                    this.processingSendMessage = true;

                    this.SendMessage(message.Data);
                    Debug.Print("SendData: " + message.SendType);

                    this.processingSendMessage = false;
                }

                message.IsSent = true;

                // 受信待ちのメッセージがある場合、待機状態に入る
                if (message.ReplyMessages != null && message.ReplyMessages.Any())
                {
                    this.timeoutCheckTiemer.Start();
                    break;
                }

                // 処理したメッセージをキューから削除
                this.sendMessageQueue.RemoveFirst();
                Debug.Print("Dequeue (OK): " + message.SendType);
            }
        }

        /// <summary>
        /// メッセージを送信します。
        /// </summary>
        /// <param name="sendData">送信する文字列</param>
        /// <returns>メッセージを正常に送れたときtrue、失敗したときfalseを返します。</returns>
        private bool SendMessage(byte[] sendData)
        {
            var handle = this.PreviewerWindowHandle;

            IntPtr sendPtr = Marshal.AllocHGlobal(sendData.Length);
            Marshal.Copy(sendData, 0, sendPtr, sendData.Length);

            int processId = Process.GetCurrentProcess().Id;

            COPYDATASTRUCT copyData = new COPYDATASTRUCT();
            copyData.dwData = new IntPtr(processId);
            copyData.cbData = (uint)sendData.Length;
            copyData.lpData = sendPtr;

            IntPtr copyPtr = Marshal.AllocHGlobal(Marshal.SizeOf(copyData));
            Marshal.StructureToPtr(copyData, copyPtr, false);

            SendMessage(handle, WM_COPYDATA, IntPtr.Zero, ref copyData);

            Marshal.FreeHGlobal(sendPtr);
            Marshal.FreeHGlobal(copyPtr);

            return true;
        }

        /// <summary>
        /// Previewer との接続状態が変わったときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        private void OnConnectionChanged(EventArgs e)
        {
            this.ConnectionChanged?.Invoke(this, e);
        }

        /// <summary>
        /// 送信メッセージ情報です。
        /// </summary>
        private class SendMessageInfo
        {
            /// <summary>
            /// 送信種類を取得または設定します。
            /// </summary>
            public SendMessageType SendType { get; set; }

            /// <summary>
            /// 送信メッセージに対する受信メッセージのリストです。
            /// </summary>
            public List<ReceiveMessageType> ReplyMessages { get; set; }

            /// <summary>
            /// 送信前に行う処理を取得または設定します。
            /// </summary>
            public Func<SendMessageInfo, bool> PreSendMethod { get; set; }

            /// <summary>
            /// 送信データを取得または設定します。
            /// </summary>
            public byte[] Data { get; set; }

            /// <summary>
            /// メッセージが送信されたかどうか取得または設定します。
            /// </summary>
            public bool IsSent { get; set; }
        }

        /// <summary>
        /// シェーダコードの送信メッセージ情報です。
        /// </summary>
        private class SendShaderCodeMessageInfo : SendMessageInfo
        {
            /// <summary>
            /// ブロックマネージャを取得または設定します。
            /// </summary>
            public CompositionBlockManager BlockManager { get; set; }

            /// <summary>
            /// ブロックエレメントを取得または設定します。
            /// </summary>
            public EffectBlockElementBase BlockElement { get; set; }

            /// <summary>
            /// ブロックエレメントのクローンを取得または設定します。
            /// </summary>
            public EffectBlockElementBase CopyBlockElement { get; set; }

            /// <summary>
            /// 出力ブロックエレメントを取得または設定します。
            /// </summary>
            public EffectBlockElementBase OutputBlockElement { get; set; }

            /// <summary>
            /// ワークスペース領域のブロックエレメントリストを取得または設定します。
            /// </summary>
            public Dictionary<uint, EffectBlockElementBase> WorkspaceBlockElements { get; set; }
        }

        #region P/Invoke Win32 API

        /// <summary>
        /// WindowProcのWM_COPYDATA番号です。
        /// </summary>
        private static readonly uint WM_COPYDATA = 0x004A;

        /// <summary>
        /// Win32 API の SendMessage 関数です。
        /// </summary>
        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = false)]
        private static extern IntPtr SendMessage(IntPtr hwnd, uint msg, IntPtr wparam, ref COPYDATASTRUCT lparam);

        /// <summary>
        /// Win32 API の COPYDATASTRUCT 構造体です。
        /// </summary>
        [StructLayout(LayoutKind.Sequential)]
        private struct COPYDATASTRUCT
        {
            /// <summary>
            /// ULONG_PTR dwData
            /// </summary>
            public IntPtr dwData;

            /// <summary>
            /// DWORD cbData
            /// </summary>
            public uint cbData;

            /// <summary>
            /// PVOID lpData
            /// </summary>
            public IntPtr lpData;
        }

        #endregion
    }
}
