﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using OperationManager.Core;
using Blocks.Core;
using EffectCombiner.Primitives.Operations;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.Semantic;
using EffectDefinitions;
using Renderer2D.Core;
using ShaderGenerator.GLSL;
using Workflow.Core;

namespace EffectCombiner.Primitives.Blocks
{
    public enum DraggingMode
    {
        None,
        RectangularSelection,
        WorkspaceMove,
        BlockMove,
        Connection,
    }

    public class MouseInputManager
    {
        private readonly IWorkspaceManager workspaceManager;

        public MouseInputManager(IWorkspaceManager workspaceManager)
        {
            if (workspaceManager == null)
                throw new ArgumentNullException("workspaceManager");

            this.workspaceManager = workspaceManager;
        }

        private void InvalidateRender()
        {
            workspaceManager.InvalidateRender();
        }

        public DraggingMode DraggingMode { get; private set; }

        /// <summary>
        /// ドラッグ中の位置を取得します。
        /// </summary>
        public IPoint DraggingPosition { get; private set; }

        /// <summary>
        /// ドラッグ中のブロックデータを取得します。
        /// </summary>
        public BlockDefinition[] DraggingBlocks { get; private set; }

        /// <summary>
        /// ドラッグ可能なデータタイプの一覧を取得します。
        /// </summary>
        public Type[] DraggableTypes
        {
            get
            {
                return new Type[] { typeof(BlockDefinition[]) };
            }
        }

        public bool IsMakingCircularReference { get; private set; }
        public bool IsInvalidPlug { get; private set; }
        public bool IsValidPlug { get; private set; }

        public bool IsConnecting { get { return BlocksConnectionInfo != null; } }
        public BlockHitInfo BlocksConnectionInfo { get; private set; }

        public IPoint MouseDownPosition { get; private set; }
        public IPoint CurrentMousePosition { get; private set; }

        private IPoint workspaceMouseDownPosition;
        private IPoint deltaMouseDownPosition;
        private BlockHitInfo blockHitTestMouseDown;
        private readonly Dictionary<uint, Tuple<IPoint, ISize>> preMovePosSize =
            new Dictionary<uint, Tuple<IPoint, ISize>>();

        /// <summary>
        /// カーソル位置に応じたリサイズ時の方向を示す値が入ります(テンキー準拠)
        /// </summary>
        private int blockResizeDirection = 0;

        private bool isLeftDragging;
        private bool isMiddleDragging;

        private bool wasControlPressed;

        public void OnMouseDown(MouseInput mouseInput)
        {
            ResetAllPlugsConnectability();

            CurrentMousePosition = new Point(mouseInput.X, mouseInput.Y);
            MouseDownPosition = new Point(mouseInput.X, mouseInput.Y);

            deltaMouseDownPosition = new Point(mouseInput.X, mouseInput.Y);
            workspaceMouseDownPosition = new Point(workspaceManager.WorkspacePosition);
            DraggingMode = DraggingMode.None;

            var isShiftPressed = workspaceManager.KeyboardInputManager.IsShiftPressed;
            var isControlPressed = workspaceManager.KeyboardInputManager.IsControlPressed;

            wasControlPressed = isControlPressed;

            if (mouseInput.LeftButton == MouseButtonState.Pressed && isShiftPressed ||
                mouseInput.MiddleButton == MouseButtonState.Pressed)
            {
                DraggingMode = DraggingMode.WorkspaceMove;
            }
            else if (mouseInput.LeftButton == MouseButtonState.Pressed)
            {
                var bht = PerformHitTest(mouseInput.X, mouseInput.Y);
                blockHitTestMouseDown = bht;

                if (bht == null)
                {
                    UnselectAllBlocks();
                    return;
                }

                if (bht.BlockElement is GhostBlockElement)
                {
                    blockHitTestMouseDown = null;
                    UnselectAllBlocks();
                    return;
                }

                if (bht.PlugType == PlugHitTestResult.None)
                {
                    if (isControlPressed == false)
                        SelectBlock(bht.BlockElement);

                    preMovePosSize.Clear();
                    foreach (var element in workspaceManager.BlockManager.BlockElements.Where(b => b.IsSelected))
                    {
                        preMovePosSize[element.InstanceIdentifier] =
                            new Tuple<IPoint, ISize>(new Point(element.Left, element.Top),
                                                     new Size(element.Width, element.Height));
                    }
                }
                else
                {
                    if (IsInputConnectedToGhostBlock(bht))
                    {
                        blockHitTestMouseDown = null;
                        return;
                    }

                    StartDraggingConnection(bht);
                }
            }
        }

        private bool IsInputConnectedToGhostBlock(BlockHitInfo bht)
        {
            if (bht.PlugType != PlugHitTestResult.Input)
                return false;

            if (bht.BlockElement.BlockDefinition.InputPlugs[bht.PlugIndex].Semantic == null)
                return false;

            if (bht.BlockElement.WorkflowItem.InputPlugs[bht.PlugIndex].RemoteOutputPlug == null)
                return false;

            var remoteBlock = (EffectWorkflowItem)bht.BlockElement.WorkflowItem.InputPlugs[bht.PlugIndex].RemoteOutputPlug.WorkflowItem;

            return remoteBlock.BlockElement is GhostBlockElement;
        }

        private void SelectBlock(EffectBlockElementBase blockElement)
        {
            if (blockElement.IsSelected == false)
            {
                UnselectAllBlocks();
                blockElement.IsSelected = true;

                if (blockElement is CommentBlockElement)
                {
                    // コメントブロックだったら中に含むノードを合わせて選択状態に
                    foreach (var elem in workspaceManager.BlockManager.BlockElements.Where(e => !(e is CommentBlockElement)))
                    {
                        elem.IsSelected = elem.IsWithinSelection(
                            true,
                            blockElement.Left,
                            blockElement.Left + blockElement.Width,
                            blockElement.Top,
                            blockElement.Top + blockElement.Height);
                    }

                    // コメントブロックは常に最背面
                    blockElement.SendToBack();
                }
                else
                {
                    // コメントブロックじゃなかったら前面に
                    blockElement.BringToFront();
                    this.workspaceManager.PreviewController.UpdatePreviewImage(blockElement, false);
                }
            }
        }

        private void InvertBlockSelection(EffectBlockElementBase blockElement)
        {
            blockElement.IsSelected = !blockElement.IsSelected;

            if (blockElement is CommentBlockElement)
            {
                // コメントブロックだったら中に含むノードの選択状態を連動する
                var tmpList = new List<BlockElementBase>();
                foreach (var elem in workspaceManager.BlockManager.BlockElements.Where(e => !(e is CommentBlockElement)))
                {
                    if (elem.IsWithinSelection(
                        true,
                        blockElement.Left,
                        blockElement.Left + blockElement.Width,
                        blockElement.Top,
                        blockElement.Top + blockElement.Height))
                    {
                        tmpList.Add(elem);
                    }
                }

                foreach (var elem in tmpList)
                {
                    elem.IsSelected = blockElement.IsSelected;
                    if (elem.IsSelected)
                    {
                        elem.BringToFront();
                    }
                }

                blockElement.SendToBack();
            }
            else
            {
                blockElement.BringToFront();
                if (blockElement.IsSelected)
                {
                    foreach (var selectedBlock in workspaceManager.BlockManager.BlockElements.ToArray().Reverse().Where(b => b.IsSelected))
                        selectedBlock.BringToFront();
                }
            }
        }

        public bool IsDisconnecting { get; private set; }
        public EffectOutputPlug DisconnectingSource { get; private set; }
        public EffectInputPlug DisconnectingTarget { get; private set; }

        private void StartDraggingConnection(BlockHitInfo bht)
        {
            var blockManager = workspaceManager.BlockManager;

            DraggingMode = DraggingMode.Connection;

            BlocksConnectionInfo = bht;

            IsDisconnecting = false;
            DisconnectingSource = null;
            DisconnectingTarget = null;

            if (bht.PlugType == PlugHitTestResult.Input)
            {
                var inputPlug = bht.BlockElement.WorkflowItem.InputPlugs[bht.PlugIndex];
                var remoteOutputPlug = (EffectOutputPlug)inputPlug.RemoteOutputPlug;
                if (remoteOutputPlug != null)
                {
                    BlocksConnectionInfo = new BlockHitInfo(remoteOutputPlug.BlockElement, remoteOutputPlug.Index, PlugHitTestResult.Output);

                    IsDisconnecting = true;
                    DisconnectingSource = remoteOutputPlug;
                    DisconnectingTarget = (EffectInputPlug)inputPlug;

                    bht = BlocksConnectionInfo;
                }
            }

            foreach (var blockElement in blockManager.BlockElements.Cast<EffectBlockElementBase>())
            {
                if (blockElement == bht.BlockElement)
                {
                    for (var n = 0; n < blockElement.BlockRenderInfo.OutputsConnectability.Length; n++)
                        blockElement.BlockRenderInfo.OutputsConnectability[n] = Connectability.Invalid;
                    for (var n = 0; n < blockElement.BlockRenderInfo.InputsConnectability.Length; n++)
                        blockElement.BlockRenderInfo.InputsConnectability[n] = Connectability.Invalid;

                    if (bht.PlugType == PlugHitTestResult.Input)
                        blockElement.BlockRenderInfo.InputsConnectability[bht.PlugIndex] = Connectability.None;
                    else if (bht.PlugType == PlugHitTestResult.Output)
                        blockElement.BlockRenderInfo.OutputsConnectability[bht.PlugIndex] = Connectability.None;

                    continue;
                }

                if (bht.PlugType == PlugHitTestResult.Input)
                {
                    var i = 0;
                    foreach (var p in blockElement.WorkflowItem.OutputPlugs.Cast<EffectOutputPlug>())
                    {
                        blockElement.BlockRenderInfo.OutputsConnectability[i] = ConnectionUtility.DetermineConnectability(
                            p.BlockElement, i,
                            bht.BlockElement, bht.PlugIndex);
                        i++;
                    }

                    if (blockElement.BlockDefinition.OutputPlugs.Length == 0)
                        blockElement.BlockRenderInfo.IsUsable = false;
                    else
                    {
                        for (var n = 0; n < blockElement.BlockRenderInfo.InputsConnectability.Length; n++)
                            blockElement.BlockRenderInfo.InputsConnectability[n] = Connectability.Invalid;
                    }
                }
                else
                {
                    var i = 0;
                    foreach (var p in blockElement.WorkflowItem.InputPlugs.Cast<EffectInputPlug>())
                    {
                        blockElement.BlockRenderInfo.InputsConnectability[i] = ConnectionUtility.DetermineConnectability(
                            bht.BlockElement, bht.PlugIndex,
                            p.BlockElement, i);
                        i++;
                    }

                    if (blockElement.BlockDefinition.InputPlugs.Length == 0)
                        blockElement.BlockRenderInfo.IsUsable = false;
                    else
                    {
                        for (var n = 0; n < blockElement.BlockRenderInfo.OutputsConnectability.Length; n++)
                            blockElement.BlockRenderInfo.OutputsConnectability[n] = Connectability.Invalid;
                    }
                }
            }

            InvalidateRender();
        }

        private IPoint previousMouseMovePosition;

        public void OnMouseMove(MouseInput mouseInput)
        {
            var blockManager = workspaceManager.BlockManager;
            var isControlPressed = workspaceManager.KeyboardInputManager.IsControlPressed;

            CurrentMousePosition = new Point(mouseInput.X, mouseInput.Y);

            if (previousMouseMovePosition == null)
                previousMouseMovePosition = CurrentMousePosition;

            var dx = 0.0;
            var dy = 0.0;

            previousMouseMovePosition = new Point(CurrentMousePosition);

            if (deltaMouseDownPosition != null)
            {
                dx = mouseInput.X - deltaMouseDownPosition.X;
                dy = mouseInput.Y - deltaMouseDownPosition.Y;

                if (Math.Abs(dx) >= Globals.VisualResources.DragWindowWidth ||
                    Math.Abs(dy) >= Globals.VisualResources.DragWindowHeight)
                {
                    if (mouseInput.LeftButton == MouseButtonState.Pressed && isLeftDragging == false)
                    {
                        isLeftDragging = true;
                    }
                    if (mouseInput.MiddleButton == MouseButtonState.Pressed && isMiddleDragging == false)
                    {
                        isMiddleDragging = true;
                    }
                }
            }

            var bht = blockHitTestMouseDown;

            // コメントノードの四隅にマウスオーバーしているかを検出したい
            if (!isLeftDragging && bht == null)
            {
                var bhtc = PerformHitTest(mouseInput.X, mouseInput.Y);
                if (bhtc != null && bhtc.BlockElement is CommentBlockElement)
                {
                    blockResizeDirection = GetResizeBaseID(mouseInput, bhtc.BlockElement);
                }
                else
                {
                    Cursor.Current = Cursors.Default;
                    blockResizeDirection = 0;
                }
            }

            if (bht != null)
            {
                if (bht.PlugType == PlugHitTestResult.None)
                {
                    if (isControlPressed && wasControlPressed && bht.BlockElement.IsSelected == false)
                    {
                        InvertBlockSelection(bht.BlockElement);
                    }

                    if (isLeftDragging && (DraggingMode == DraggingMode.None || DraggingMode == DraggingMode.BlockMove))
                    {
                        DraggingMode = DraggingMode.BlockMove;
                        var d = blockResizeDirection;

                        if (d == 0 || d == 5)
                        {
                            // リサイズ基点をポイントしてない場合は通常の移動
                            foreach (var blockElement in workspaceManager.BlockManager.BlockElements.Where(b => b.IsSelected))
                            {
                                blockElement.SetPosition(blockElement.Left + dx, blockElement.Top + dy);
                                blockElement.IsDragging = true;
                            }
                        }
                        else
                        {
                            // リサイズ基点をポイントしている場合はリサイズと移動の連携
                            foreach (var blockElement in workspaceManager.BlockManager.BlockElements.Where(b => b.IsSelected).OfType<CommentBlockElement>())
                            {
                                // 方向ロック
                                dx = (d == 2 || d == 8) ? 0 : dx;
                                dy = (d == 4 || d == 6) ? 0 : dy;

                                var tmpX = blockElement.Left;
                                if (d == 1 || d == 4 || d == 7)
                                {
                                    tmpX += dx;
                                    dx = -dx;
                                }

                                var tmpY = blockElement.Top;
                                if (d == 7 || d == 8 || d == 9)
                                {
                                    tmpY += dy;
                                    dy = -dy;
                                }

                                var tmpW = (float)(blockElement.Width + dx);
                                var tmpH = (float)(blockElement.Height + dy);

                                blockElement.SetPosition(tmpX, tmpY);
                                blockElement.SetComputedSize(tmpW, tmpH);
                                blockElement.IsDragging = true;
                            }
                        }

                        deltaMouseDownPosition = new Point(mouseInput.X, mouseInput.Y);
                    }
                }
            }
            else if (isLeftDragging && (DraggingMode == DraggingMode.None || DraggingMode == DraggingMode.RectangularSelection))
            {
                DraggingMode = DraggingMode.RectangularSelection;

                if (isControlPressed == false)
                    UnselectAllBlocks();

                var left = Math.Min(MouseDownPosition.X, CurrentMousePosition.X);
                var right = Math.Max(MouseDownPosition.X, CurrentMousePosition.X);
                var top = Math.Min(MouseDownPosition.Y, CurrentMousePosition.Y);
                var bottom = Math.Max(MouseDownPosition.Y, CurrentMousePosition.Y);
                bool completelyIncludeToSelect = Globals.Options.EnvironmentSettings.CompletelyIncludeToSelect;

                foreach (var blockElement in workspaceManager.BlockManager.BlockElements)
                {
                    blockElement.IsInvertingSelection = blockElement.IsWithinSelection(completelyIncludeToSelect, left, right, top, bottom);
                }

                InvalidateRender();
            }
            else if ((isLeftDragging || isMiddleDragging) && DraggingMode == DraggingMode.WorkspaceMove)
            {
                DraggingMode = DraggingMode.WorkspaceMove;
                workspaceManager.UpdateWorkspacePosition(new Point(workspaceMouseDownPosition.X + dx, workspaceMouseDownPosition.Y + dy));

                InvalidateRender();
            }

            IsMakingCircularReference = false;
            IsInvalidPlug = false;
            IsValidPlug = false;

            foreach (var blockElement in blockManager.BlockElements.OfType<EffectBlockElementBase>())
            {
                blockElement.BlockRenderInfo.IsPartOfCircularReference = false;
            }

            if (IsConnecting)
            {
                BlockHitInfo targetPlug;

                if (Globals.Options.EnvironmentSettings.PlugSnapping.IsActive == false)
                {
                    targetPlug = PerformHitTest(mouseInput.X, mouseInput.Y);
                    MouseSnap = null;
                }
                else
                {
                    targetPlug = FindClosestHitTest(mouseInput.X, mouseInput.Y, BlocksConnectionInfo.PlugType == PlugHitTestResult.Output);
                    MouseSnap = targetPlug;
                }

                if (targetPlug != null)
                {
                    if (targetPlug.PlugType != PlugHitTestResult.None)
                        CheckTargetPlugValidity(mouseInput, targetPlug);

                    if (blockHitTestMouseDown.PlugType == PlugHitTestResult.Input)
                    {
                        if (BlocksConnectionInfo.PlugType == PlugHitTestResult.Output)
                        {
                            var outputPlug = BlocksConnectionInfo.BlockElement.WorkflowItem.OutputPlugs[BlocksConnectionInfo.PlugIndex];
                            var index = outputPlug.Index;
                            ((EffectOutputPlug)outputPlug).BlockElement.BlockRenderInfo.OutputsConnectability[index] = Connectability.None;
                        }
                    }
                    else if (blockHitTestMouseDown.PlugType == PlugHitTestResult.Output)
                    {
                        blockHitTestMouseDown.BlockElement.BlockRenderInfo.OutputsConnectability[blockHitTestMouseDown.PlugIndex] = Connectability.None;
                    }
                }

                InvalidateRender();
            }
        }

        public void OnMouseUp(MouseInput mouseInput)
        {
            var blockManager = workspaceManager.BlockManager;
            var isControlPressed = workspaceManager.KeyboardInputManager.IsControlPressed;

            ResetAllPlugsConnectability();

            if (DraggingMode == DraggingMode.None)
            {
                if (mouseInput.LeftButton == MouseButtonState.Pressed)
                {
                    if (isControlPressed && wasControlPressed && blockHitTestMouseDown != null)
                    {
                        InvertBlockSelection(blockHitTestMouseDown.BlockElement);
                    }
                }
            }
            else if (DraggingMode == DraggingMode.RectangularSelection)
            {
                if (isControlPressed == false || wasControlPressed == false)
                {
                    UnselectAllBlocks();
                    foreach (var blockElement in blockManager.BlockElements)
                    {
                        blockElement.IsSelected = blockElement.IsInvertingSelection;
                        blockElement.IsInvertingSelection = false;
                    }
                }
                else
                {
                    foreach (var blockElement in blockManager.BlockElements)
                    {
                        if (blockElement.IsInvertingSelection)
                        {
                            blockElement.IsSelected = !blockElement.IsSelected;
                            blockElement.IsInvertingSelection = false;
                        }
                    }
                }
            }
            else
            {
                if (blockHitTestMouseDown != null)
                {
                    if (blockHitTestMouseDown.PlugType == PlugHitTestResult.None)
                    {
                        var operations = new List<OperationBase>();
                        foreach (var blockElement in blockManager.BlockElements.Where(b => b.IsSelected))
                        {
                            Tuple<IPoint, ISize> info;
                            if (preMovePosSize.TryGetValue(blockElement.InstanceIdentifier, out info))
                            {
                                operations.Add(new MoveBlockOperation(blockManager, blockElement,
                                    info.Item1,
                                    new Point(blockElement.Left, blockElement.Top),
                                    info.Item2,
                                    new Size(blockElement.Width, blockElement.Height)));
                            }
                        }

                        Globals.MainOperationManager.Add(new AggregateOperation(OperationType.MoveBlock, operations));
                    }
                }
            }

            isLeftDragging = false;
            isMiddleDragging = false;

            foreach (var blockElement in blockManager.BlockElements.Cast<EffectBlockElementBase>())
            {
                blockElement.BlockRenderInfo.IsUsable = true;
            }

            if (IsConnecting && blockHitTestMouseDown != null)
            {
                var bht = MouseSnap ?? PerformHitTest(mouseInput.X, mouseInput.Y);

                if (bht != null)
                {
                    if (bht.PlugType != PlugHitTestResult.None)
                    {
                        if (mouseInput.LeftButton == MouseButtonState.Pressed)
                        {
                            if (bht.BlockElement == BlocksConnectionInfo.BlockElement)
                            {
                                // connect to same block
                            }
                            else if (bht.PlugType == BlocksConnectionInfo.PlugType)
                            {
                                // try to connection input with input or output with output
                            }
                            else if (CheckPlugShaderTypeMatch(bht, BlocksConnectionInfo) == false)
                            {
                                // plug type mismatch
                            }
                            else if (AreBlockHitInfoEquals(bht, BlocksConnectionInfo))
                            {
                                // same plug
                            }
                            else
                            {
                                var sourceWorkflowItem = BlocksConnectionInfo.BlockElement.WorkflowItem;
                                var targetWorkflowItem = bht.BlockElement.WorkflowItem;

                                EffectOutputPlug outputPlug;
                                EffectInputPlug inputPlug;

                                if (bht.PlugType == PlugHitTestResult.Input)
                                {
                                    outputPlug = (EffectOutputPlug)sourceWorkflowItem.OutputPlugs[BlocksConnectionInfo.PlugIndex];

                                    if (AreBlockHitInfoEquals(blockHitTestMouseDown, BlocksConnectionInfo))
                                    {
                                        // out to in
                                        outputPlug = (EffectOutputPlug)sourceWorkflowItem.OutputPlugs[BlocksConnectionInfo.PlugIndex];
                                        inputPlug = (EffectInputPlug)targetWorkflowItem.InputPlugs[bht.PlugIndex];

                                        if (ConnectionManager.AreConnected(outputPlug, inputPlug) == false &&
                                            ConnectionManager.CheckCircularReference(outputPlug, inputPlug) == false)
                                        {
                                            Globals.MainOperationManager.Add(new ConnectionOperation(blockManager, true, inputPlug, outputPlug));
                                            ConnectionManager.Connect(outputPlug, inputPlug);
                                            this.workspaceManager.PreviewController.UpdatePreviewImage(bht.BlockElement, true);
                                        }
                                    }
                                    else
                                    {
                                        // reconnecting in to another in
                                        var disconnectingInputPlug = blockHitTestMouseDown.BlockElement.WorkflowItem.InputPlugs[blockHitTestMouseDown.PlugIndex];
                                        var reconnectingInputPlug = bht.BlockElement.WorkflowItem.InputPlugs[bht.PlugIndex];

                                        if (ConnectionManager.AreConnected(outputPlug, reconnectingInputPlug) == false &&
                                            ConnectionManager.CheckCircularReference(outputPlug, reconnectingInputPlug) == false)
                                        {
                                            var operations = new List<IOperation>
                                            {
                                                new ConnectionOperation(workspaceManager.BlockManager, false, (EffectInputPlug)disconnectingInputPlug, outputPlug),
                                                new ConnectionOperation(workspaceManager.BlockManager, true, (EffectInputPlug)reconnectingInputPlug, outputPlug),
                                            };

                                            blockManager.BeginShaderCodeGenerationLock();

                                            try
                                            {
                                                ConnectionManager.Disconnect(disconnectingInputPlug);
                                                ConnectionManager.Connect(outputPlug, reconnectingInputPlug);
                                                CheckOrphanGhostReconnection((EffectInputPlug)disconnectingInputPlug, operations);
                                                Globals.MainOperationManager.Add(new AggregateOperation(OperationType.ConnectionChange, operations));
                                                this.workspaceManager.PreviewController.UpdatePreviewImage(bht.BlockElement, true);
                                            }
                                            finally
                                            {
                                                blockManager.EndShaderCodeGenerationLock();
                                            }
                                        }
                                    }
                                }
                                else
                                {
                                    outputPlug = (EffectOutputPlug)targetWorkflowItem.OutputPlugs[bht.PlugIndex];
                                    inputPlug = (EffectInputPlug)sourceWorkflowItem.InputPlugs[BlocksConnectionInfo.PlugIndex];

                                    if (ConnectionManager.AreConnected(outputPlug, inputPlug) == false &&
                                        ConnectionManager.CheckCircularReference(outputPlug, inputPlug) == false)
                                    {
                                        Globals.MainOperationManager.Add(new ConnectionOperation(blockManager, true, inputPlug, outputPlug));
                                        ConnectionManager.Connect(outputPlug, inputPlug);
                                        this.workspaceManager.PreviewController.UpdatePreviewImage(bht.BlockElement, true);

                                    }
                                }
                            }
                        }
                    }
                }
                else if (IsDisconnecting)
                {
                    // disconnect
                    var bci = BlocksConnectionInfo;

                    var inputPlug = (EffectInputPlug)blockHitTestMouseDown.BlockElement.WorkflowItem.InputPlugs[blockHitTestMouseDown.PlugIndex];
                    var outputPlug = (EffectOutputPlug)bci.BlockElement.WorkflowItem.OutputPlugs[bci.PlugIndex];

                    var operations = new List<IOperation>
                    {
                        new ConnectionOperation(workspaceManager.BlockManager, false, inputPlug, outputPlug),
                    };

                    blockManager.BeginShaderCodeGenerationLock();

                    try
                    {
                        ConnectionManager.Disconnect(inputPlug);
                        CheckOrphanGhostReconnection(inputPlug, operations);
                        Globals.MainOperationManager.Add(new AggregateOperation(OperationType.Disconnect, operations));
                        this.workspaceManager.PreviewController.UpdatePreviewImage(blockHitTestMouseDown.BlockElement, true);
                    }
                    finally
                    {
                        blockManager.EndShaderCodeGenerationLock();
                    }
                }
            }

            DraggingMode = DraggingMode.None;

            ReleaseDragAllBlocks();

            blockHitTestMouseDown = null;
            BlocksConnectionInfo = null;

            IsDisconnecting = false;
            DisconnectingSource = null;
            DisconnectingTarget = null;

            MouseSnap = null;

            InvalidateRender();
        }

        private void CheckOrphanGhostReconnection(EffectInputPlug inputPlug, List<IOperation> operations)
        {
            var blockManager = workspaceManager.BlockManager;

            var blockElement = inputPlug.BlockElement;

            var expandedBlocks = SemanticHelper.ExpandSemantics(blockElement.BlockManager, Generation.Globals.BlockDefinitions.Values, blockElement);
            foreach (var eb in expandedBlocks)
                blockManager.AddBlock(eb);
        }

        /// <summary>
        /// ワークスペースがドラッグ先のターゲットになったときの処理を行います。
        /// </summary>
        /// <param name="x">マウスの X 座標</param>
        /// <param name="y">マウスの Y 座標</param>
        /// <param name="dataType">ドラッグ中のデータの種類</param>
        /// <param name="data">ドラッグ中のデータ</param>
        /// <returns>ドラッグ & ドロップステータスを返します。</returns>
        public DragDropState OnDragEnter(double x, double y, DragDropDataType dataType, object data)
        {
            return this.OnDragOver(x, y, dataType, data);
        }

        /// <summary>
        /// ワークスペースがドラッグ先のターゲットから外れたときの処理を行います。
        /// </summary>
        public void OnDragLeave()
        {
            this.DraggingBlocks = null;
        }

        /// <summary>
        /// ドラッグ中にマウスを移動したときの処理を行います。
        /// </summary>
        /// <param name="x">マウスの X 座標</param>
        /// <param name="y">マウスの Y 座標</param>
        /// <param name="dataType">ドラッグ中のデータの種類</param>
        /// <param name="data">ドラッグ中のデータ</param>
        /// <returns>ドラッグ & ドロップステータスを返します。</returns>
        public DragDropState OnDragOver(double x, double y, DragDropDataType dataType, object data)
        {
            DragDropState result = DragDropState.None;

            // ファイルをドラッグしているときの処理を行う
            if (dataType == DragDropDataType.File)
            {
                string fileExtension = Path.GetExtension(((string[])data)[0]);

                BlockHitInfo hitInfo = this.PerformHitTest(x, y);

                // ファイル選択用の uniform ノード上ではドラッグを許可する
                if (hitInfo != null)
                {
                    UniformDefinition uniform = hitInfo.BlockElement.BlockDefinition.Uniform;

                    if (uniform != null && uniform.Extension == fileExtension)
                    {
                        result = DragDropState.Copy;
                    }
                }
            }
            // その他の object データをドラッグしているときの処理を行う
            else if (dataType == DragDropDataType.ObjectData)
            {
                if (data is BlockDefinition[])
                {
                    this.DraggingBlocks = (BlockDefinition[])data;
                    this.DraggingPosition = new Point(x, y);

                    result = DragDropState.Copy;
                }
            }

            return result;
        }

        /// <summary>
        /// データがドロップされたときの処理を行います。
        /// </summary>
        /// <param name="x">マウスの X 座標</param>
        /// <param name="y">マウスの Y 座標</param>
        /// <param name="dataType">ドラッグ中のデータの種類</param>
        /// <param name="data">ドラッグ中のデータ</param>
        /// <returns>ドラッグ & ドロップステータスを返します。</returns>
        public DragDropState OnDragDrop(double x, double y, DragDropDataType dataType, object data)
        {
            DragDropState result = DragDropState.None;

            // ファイルをドロップしたときの処理を行う
            if (dataType == DragDropDataType.File)
            {
                string filePath = ((string[])data)[0];
                string fileExtension = Path.GetExtension(filePath);

                BlockHitInfo hitInfo = this.PerformHitTest(x, y);

                // ファイル選択用の uniform ブロックにドロップしたときの処理を行う
                if (hitInfo != null)
                {
                    UniformDefinition uniformDefinition = hitInfo.BlockElement.BlockDefinition.Uniform;

                    if (uniformDefinition != null && uniformDefinition.Extension == fileExtension)
                    {
                        var userDefinitionElement = hitInfo.BlockElement as UserDefinitionBlockElement;

                        var operation = new UniformValueChangeOperation(
                            hitInfo.BlockElement.BlockManager,
                            uniformDefinition.Name,
                            userDefinitionElement.UniformValue,
                            new UniformDataFile(filePath));
                        Globals.MainOperationManager.Add(operation, true);

                        result = DragDropState.Copy;
                    }
                }
            }
            // その他の object データをドロップしたときの処理を行う
            else if (dataType == DragDropDataType.ObjectData)
            {
                // ブロックデータをドロップしたときの処理を行う
                if (data is BlockDefinition[])
                {
                    this.workspaceManager.BlockManager.OnBlockDefinitionsDropped(x, y, (BlockDefinition[])data);

                    this.DraggingBlocks = null;

                    result = DragDropState.Copy;
                }
            }

            return result;
        }

        public BlockHitInfo MouseSnap { get; private set; }

        private BlockHitInfo FindClosestHitTest(double x, double y, bool lookForInput)
        {
            var curLoc = new Point(x, y);

            var smallestDistance = double.MaxValue;
            BlockHitInfo closest = null;

            //lock (workspaceManager.BlockManager.BlocksSyncRoot)
            {
                foreach (var block in workspaceManager.BlockManager.BlockElements.Cast<EffectBlockElementBase>())
                {
                    if (lookForInput)
                    {
                        for (var i = 0; i < block.BlockRenderInfo.InputPlugPositions.Length; i++)
                        {
                            if (block.BlockRenderInfo.InputsConnectability[i] != Connectability.Best &&
                                block.BlockRenderInfo.InputsConnectability[i] != Connectability.Good)
                                continue;

                            var pos = new Point(block.BlockRenderInfo.InputPlugPositions[i].X + block.Left,
                                block.BlockRenderInfo.InputPlugPositions[i].Y + block.Top);

                            var distance = Distance(curLoc, pos);
                            if (distance > Globals.Options.EnvironmentSettings.PlugSnapping.MinimumDistance || distance > smallestDistance)
                                continue;

                            closest = new BlockHitInfo(block, i, PlugHitTestResult.Input);
                            smallestDistance = distance;
                        }
                    }
                    else
                    {
                        for (var i = 0; i < block.BlockRenderInfo.OutputPlugPositions.Length; i++)
                        {
                            if (block.BlockRenderInfo.OutputsConnectability[i] != Connectability.Best &&
                                block.BlockRenderInfo.OutputsConnectability[i] != Connectability.Good)
                                continue;

                            var pos = new Point(block.BlockRenderInfo.OutputPlugPositions[i].X + block.Left,
                                block.BlockRenderInfo.OutputPlugPositions[i].Y + block.Top);

                            var distance = Distance(curLoc, pos);
                            if (distance > Globals.Options.EnvironmentSettings.PlugSnapping.MinimumDistance || distance > smallestDistance)
                                continue;

                            closest = new BlockHitInfo(block, i, PlugHitTestResult.Output);
                            smallestDistance = distance;
                        }
                    }
                }
            }

            return closest;
        }

        private static double Distance(IPoint p1, IPoint p2)
        {
            var dx = p1.X - p2.X;
            var dy = p1.Y - p2.Y;
            return Math.Sqrt(dx * dx + dy * dy);
        }

        private static bool AreBlockHitInfoEquals(BlockHitInfo a, BlockHitInfo b)
        {
            if (a == null || b == null)
                return false;

            return a.BlockElement.InstanceIdentifier == b.BlockElement.InstanceIdentifier &&
                a.PlugType == b.PlugType &&
                a.PlugIndex == b.PlugIndex;
        }

        /// <summary>
        /// ブロック要素とマウス座標から、8方向の辺や点のうち、どこをポイントしているかを取得します
        /// </summary>
        /// <param name="mouseInput">マウス座標</param>
        /// <param name="blockElement">ブロック要素</param>
        /// <returns>テンキー配列準拠での方向(どこもポイントしてない場合は0)</returns>
        private static int GetResizeBaseID(MouseInput mouseInput, BlockElementBase blockElement)
        {
            double x = mouseInput.X;
            double y = mouseInput.Y;

            double l = blockElement.Left;
            double t = blockElement.Top;
            double r = l + blockElement.Width;
            double b = t + blockElement.Height;

            bool nl = (l - x) <= 0.0f && (l - x) >= -10.0f;
            bool nt = (t - y) <= 0.0f && (t - y) >= -10.0f;
            bool nr = (r - x) >= 0.0f && (r - x) <= 10.0f;
            bool nb = (b - y) >= 0.0f && (b - y) <= 10.0f;

            if (nl && nt)
            {
                Cursor.Current = Cursors.SizeNWSE;
                return 7;
            }
            else if (nl && nb)
            {
                Cursor.Current = Cursors.SizeNESW;
                return 1;
            }
            else if (nr && nt)
            {
                Cursor.Current = Cursors.SizeNESW;
                return 9;
            }
            else if (nr && nb)
            {
                Cursor.Current = Cursors.SizeNWSE;
                return 3;
            }
            else if (nl)
            {
                Cursor.Current = Cursors.SizeWE;
                return 4;
            }
            else if (nt)
            {
                Cursor.Current = Cursors.SizeNS;
                return 8;
            }
            else if (nr)
            {
                Cursor.Current = Cursors.SizeWE;
                return 6;
            }
            else if (nb)
            {
                Cursor.Current = Cursors.SizeNS;
                return 2;
            }

            Cursor.Current = Cursors.Default;
            return 0;
        }

        private void CheckTargetPlugValidity(MouseInput mouseInput, BlockHitInfo targetPlug)
        {
            var blockManager = workspaceManager.BlockManager;

            if (mouseInput.LeftButton != MouseButtonState.Pressed)
                return;

            if (targetPlug.BlockElement == BlocksConnectionInfo.BlockElement ||
                    targetPlug.PlugType == BlocksConnectionInfo.PlugType ||
                    CheckPlugShaderTypeMatch(targetPlug, BlocksConnectionInfo) == false)
            {
                IsInvalidPlug = true;
            }
            else
            {
                EffectWorkflowItem a;
                EffectWorkflowItem b;

                if (BlocksConnectionInfo.PlugType == PlugHitTestResult.Output)
                {
                    a = BlocksConnectionInfo.BlockElement.WorkflowItem;
                    b = targetPlug.BlockElement.WorkflowItem;
                }
                else
                {
                    a = targetPlug.BlockElement.WorkflowItem;
                    b = BlocksConnectionInfo.BlockElement.WorkflowItem;
                }

                WorkflowItem<PlugValue>[] route;

                if (ConnectionManager.CheckCircularReference(a, b, out route))
                {
                    IsInvalidPlug = true;
                    IsMakingCircularReference = true;
                    foreach (EffectWorkflowItem w in route)
                        w.BlockElement.BlockRenderInfo.IsPartOfCircularReference = true;
                }
                else
                {
                    IsValidPlug = true;
                    foreach (var tmp in blockManager.BlockElements.Cast<EffectBlockElementBase>())
                        tmp.BlockRenderInfo.IsPartOfCircularReference = false;
                }
            }
        }

        private void ResetAllPlugsConnectability(bool invalidOnly = false)
        {
            var blockManager = workspaceManager.BlockManager;

            foreach (var blockElement in blockManager.BlockElements.Cast<EffectBlockElementBase>())
            {
                for (var i = 0; i < blockElement.BlockRenderInfo.InputsConnectability.Length; i++)
                {
                    if (invalidOnly == false || blockElement.BlockRenderInfo.InputsConnectability[i] == Connectability.Invalid)
                        blockElement.BlockRenderInfo.InputsConnectability[i] = Connectability.None;
                }

                for (var i = 0; i < blockElement.BlockRenderInfo.OutputsConnectability.Length; i++)
                {
                    if (invalidOnly == false || blockElement.BlockRenderInfo.OutputsConnectability[i] == Connectability.Invalid)
                        blockElement.BlockRenderInfo.OutputsConnectability[i] = Connectability.None;
                }
            }
        }

        /// <summary>
        /// プラグ間で接続が可能かどうかチェックします。
        /// </summary>
        /// <param name="plug1">プラグ 1 の情報</param>
        /// <param name="plug2">プラグ 2 の情報</param>
        /// <returns>接続が可能なときtrue、それ以外はfalseを返します。</returns>
        public static bool CheckPlugShaderTypeMatch(BlockHitInfo plug1, BlockHitInfo plug2)
        {
            if (plug1 == null)
                throw new ArgumentNullException("plug1");
            if (plug2 == null)
                throw new ArgumentNullException("plug2");

            // plug1, plug2 どちらかが存在しなければ false
            if (plug1.PlugType == PlugHitTestResult.None ||
                plug2.PlugType == PlugHitTestResult.None)
            {
                return false;
            }

            BlockHitInfo inputPlug = null;
            BlockHitInfo outputPlug = null;

            // plug1, plug2 どちらが入力プラグでどちらが出力プラグか調べる
            if (plug1.PlugType == PlugHitTestResult.Input)
            {
                inputPlug = plug1;
                outputPlug = plug2;
            }
            else if (plug1.PlugType == PlugHitTestResult.Output)
            {
                inputPlug = plug2;
                outputPlug = plug1;
            }
            else
            {
                return false;
            }

            // プラグ同士の適性をチェック
            var connectability = ConnectionUtility.DetermineConnectability(
                outputPlug.BlockElement, outputPlug.PlugIndex,
                inputPlug.BlockElement, inputPlug.PlugIndex);

            return (connectability != Connectability.Invalid);
        }

        private BlockHitInfo PerformHitTest(double x, double y)
        {
            foreach (var blockElement in workspaceManager.BlockManager.BlockElements.OfType<EffectBlockElementBase>())
            {
                if (blockElement.HitTest(x, y) == false)
                    continue;

                int plugIndex;
                var plugHitTestResult = blockElement.PlugHitTest(x, y, out plugIndex);

                if (plugHitTestResult != PlugHitTestResult.None)
                    return new BlockHitInfo(blockElement, plugIndex, plugHitTestResult);

                return new BlockHitInfo(blockElement);
            }

            return null;
        }

        private void UnselectAllBlocks()
        {
            foreach (var blockElement in workspaceManager.BlockManager.BlockElements)
                blockElement.IsSelected = false;
        }

        private void ReleaseDragAllBlocks()
        {
            foreach (var blockElement in workspaceManager.BlockManager.BlockElements)
                blockElement.IsDragging = false;
        }
    }

    public class BlockHitInfo
    {
        public EffectBlockElementBase BlockElement { get; private set; }
        public int PlugIndex { get; private set; }
        public PlugHitTestResult PlugType { get; private set; }

        public BlockHitInfo(EffectBlockElementBase blockElement)
            : this(blockElement, -1, PlugHitTestResult.None)
        {
        }

        public BlockHitInfo(EffectBlockElementBase blockElement, int plugIndex, PlugHitTestResult plugType)
        {
            if (blockElement == null)
                throw new ArgumentNullException("blockElement");

            BlockElement = blockElement;
            PlugIndex = plugIndex;
            PlugType = plugType;
        }
    }
}
