﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Blocks.Core;

namespace EffectCombiner.Primitives.Blocks
{
    /// <summary>
    /// キー入力を管理します。
    /// </summary>
    public class KeyboardInputManager
    {
        /// <summary>
        /// ワークスペースです。
        /// </summary>
        private readonly IWorkspaceManager workspaceManager;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="workspaceManager">ワークスペース</param>
        public KeyboardInputManager(IWorkspaceManager workspaceManager)
        {
            if (workspaceManager == null)
                throw new ArgumentNullException("workspaceManager");

            this.workspaceManager = workspaceManager;
        }

        /// <summary>
        /// キーを押したときの処理を行います。
        /// </summary>
        /// <param name="keyInput">キー情報</param>
        public void OnKeyDown(KeyboardInput keyInput)
        {
            this.IsAltPressed = keyInput.Alt;
            this.IsControlPressed = keyInput.Control;
            this.IsShiftPressed = keyInput.Shift;

            if (this.IsAltPressed == false && this.IsControlPressed == false && this.IsShiftPressed == false)
            {
                if (keyInput.KeyCode == 46 || keyInput.KeyCode == 8) // suppr/del or backspace
                {
                    if (workspaceManager.BlockManager.RemoveSelectedBlocks()) {
                        workspaceManager.BlockManager.InvalidateRender();
                    }
                }
            }
        }

        /// <summary>
        /// キーを押したときの処理を行います。
        /// </summary>
        /// <param name="keyInput">キー情報</param>
        public void OnKeyPress(KeyboardInput keyInput)
        {
        }

        /// <summary>
        /// キーを離したときの処理を行います。
        /// </summary>
        /// <param name="keyInput">キー情報</param>
        public void OnKeyUp(KeyboardInput keyInput)
        {
            this.IsAltPressed = keyInput.Alt;
            this.IsControlPressed = keyInput.Control;
            this.IsShiftPressed = keyInput.Shift;
        }

        /// <summary>
        /// フォーカスを得たときの処理を行います。
        /// </summary>
        /// <param name="keyInput">キー情報</param>
        public void OnGotFocus(KeyboardInput keyInput)
        {
            this.IsAltPressed = keyInput.Alt;
            this.IsControlPressed = keyInput.Control;
            this.IsShiftPressed = keyInput.Shift;
        }

        /// <summary>
        /// フォーカスを失ったときの処理を行います。
        /// </summary>
        /// <param name="keyInput">キー情報</param>
        public void OnLostFocus(KeyboardInput keyInput)
        {
            this.IsAltPressed = keyInput.Alt;
            this.IsControlPressed = keyInput.Control;
            this.IsShiftPressed = keyInput.Shift;
        }

        /// <summary>
        /// Altキーが押されているかどうか取得します。
        /// </summary>
        public bool IsAltPressed { get; private set; }

        /// <summary>
        /// Ctrlキーが押されているかどうか取得します。
        /// </summary>
        public bool IsControlPressed { get; private set; }

        /// <summary>
        /// Shiftキーが押されているかどうか取得します。
        /// </summary>
        public bool IsShiftPressed { get; private set; }
    }
}
