﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectDefinitions;

namespace EffectCombiner.Primitives.Generation.Usage.Descriptors
{
    public class TexCoordUsageDescriptor : IUsageDescriptor
    {
        public string Name
        {
            get { return "texcoord"; }
        }

        private readonly ShaderTyping.ShaderTypeDefinition type = new ShaderTypeDefinition("vec2");
        public ShaderTyping.ShaderTypeDefinition Type
        {
            get { return type; }
        }

        public SubPlug[] ExpandInput(Plug originalInputPlug)
        {
            return GenerateSubPlugs(originalInputPlug, true);
        }

        public SubPlug[] ExpandOutput(Plug originalOutputPlug)
        {
            return GenerateSubPlugs(originalOutputPlug, false);
        }

        private static SubPlug[] GenerateSubPlugs(Plug originalPlug, bool isInput)
        {
            var u = new SubPlug(originalPlug, "u", new ShaderTypeDefinition("float"), "x", isInput, new[] { 0 });
            var v = new SubPlug(originalPlug, "v", new ShaderTypeDefinition("float"), "y", isInput, new[] { 1 });

            u.SetPairedSubPlugs(v);
            v.SetPairedSubPlugs(u);

            return new[] { u, v };
        }

        public bool AreInputsProperlyConnected(EffectInputPlug originalPlug, EffectInputPlug[] subPlugs)
        {
            if (originalPlug == null || subPlugs == null || subPlugs.Length != 2)
                return false;

            var weights = new int[2];

            if (originalPlug.RemoteOutputPlug != null) // uv
            {
                weights[0]++;
                weights[1]++;
            }

            if (subPlugs[0].RemoteOutputPlug != null) // u
                weights[0]++;
            if (subPlugs[1].RemoteOutputPlug != null) // v
                weights[1]++;

            if (weights.Any(w => w > 1)) // invalid, over-connected
                return false;

            if (originalPlug.BlockDefinitionPlug != null &&
                originalPlug.BlockDefinitionPlug.DefaultValues != null &&
                originalPlug.BlockDefinitionPlug.DefaultValues.Length >= 2)
                return true; // not invalid, default values can take over

            // otherwise make sure it is valid
            return weights.All(w => w == 1);
        }

        public bool AreOutputsProperlyConnected(EffectOutputPlug originalPlug, EffectOutputPlug[] subPlugs)
        {
            if (originalPlug == null || subPlugs == null || subPlugs.Length != 2)
                return false;

            var weights = new int[2];

            if (originalPlug.RemoteInputPlugs.Length > 0) // uv
            {
                weights[0]++;
                weights[1]++;
            }

            if (subPlugs[0].RemoteInputPlugs.Length > 0) // u
                weights[0]++;
            if (subPlugs[1].RemoteInputPlugs.Length > 0) // v
                weights[1]++;

            if (weights.Any(w => w > 1)) // invalid, over-connected
                return false;

            // otherwise make sure it is valid
            return weights.All(w => w == 1);
        }
    }
}
