﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectDefinitions;

namespace EffectCombiner.Primitives.Generation.Usage.Descriptors
{
    public class ColorUsageDescriptor : IUsageDescriptor
    {
        public string Name
        {
            get { return "color"; }
        }

        private readonly ShaderTyping.ShaderTypeDefinition type = new ShaderTypeDefinition("vec4");
        public ShaderTyping.ShaderTypeDefinition Type
        {
            get { return type; }
        }

        public SubPlug[] ExpandInput(Plug originalInputPlug)
        {
            return GenerateSubPlugs(originalInputPlug);
        }

        public SubPlug[] ExpandOutput(Plug originalOutputPlug)
        {
            return GenerateSubPlugs(originalOutputPlug);
        }

        private static SubPlug[] GenerateSubPlugs(Plug originalPlug)
        {
            var isInput = originalPlug.IsInput;

            var rgb = new SubPlug(originalPlug, "rgb", new ShaderTypeDefinition("vec3"), "rgb", isInput, new[] { 0, 1, 2 });
            var r = new SubPlug(originalPlug, "r", new ShaderTypeDefinition("float"), "r", isInput, new[] { 0 });
            var g = new SubPlug(originalPlug, "g", new ShaderTypeDefinition("float"), "g", isInput, new[] { 1 });
            var b = new SubPlug(originalPlug, "b", new ShaderTypeDefinition("float"), "b", isInput, new[] { 2 });
            var a = new SubPlug(originalPlug, "a", new ShaderTypeDefinition("float"), "a", isInput, new[] { 3 });

            rgb.SetPairedSubPlugs(a);
            r.SetPairedSubPlugs(g, b, a);
            g.SetPairedSubPlugs(r, b, a);
            b.SetPairedSubPlugs(r, g, a);
            a.SetPairedSubPlugs(rgb, r, g, b);

            return new[] { rgb, r, g, b, a };
        }

        public bool AreInputsProperlyConnected(EffectInputPlug originalPlug, EffectInputPlug[] subPlugs)
        {
            if (originalPlug == null || subPlugs == null || subPlugs.Length != 5)
                return false;

            var weights = new int[4];

            if (originalPlug.RemoteOutputPlug != null) // rgba
            {
                for (var i = 0; i < 4; i++)
                    weights[i]++;
            }

            if (subPlugs[0].RemoteOutputPlug != null) // rgb
            {
                for (var i = 0; i < 3; i++)
                    weights[i]++;
            }

            if (subPlugs[1].RemoteOutputPlug != null) // r
                weights[0]++;
            if (subPlugs[2].RemoteOutputPlug != null) // g
                weights[1]++;
            if (subPlugs[3].RemoteOutputPlug != null) // b
                weights[2]++;
            if (subPlugs[4].RemoteOutputPlug != null) // a
                weights[3]++;

            if (weights.Any(w => w > 1)) // invalid, over-connected
                return false;

            if (originalPlug.BlockDefinitionPlug != null &&
                originalPlug.BlockDefinitionPlug.DefaultValues != null &&
                originalPlug.BlockDefinitionPlug.DefaultValues.Length >= 4)
                return true; // not invalid, default values can take over

            // otherwise make sure it is valid
            return weights.All(w => w == 1);
        }

        public bool AreOutputsProperlyConnected(EffectOutputPlug originalPlug, EffectOutputPlug[] subPlugs)
        {
            if (originalPlug == null || subPlugs == null || subPlugs.Length != 5)
                return false;

            var weights = new int[4];

            if (originalPlug.RemoteInputPlugs.Length > 0) // rgba
            {
                for (var i = 0; i < 4; i++)
                    weights[i]++;
            }

            if (subPlugs[0].RemoteInputPlugs.Length > 0) // rgb
            {
                for (var i = 0; i < 3; i++)
                    weights[i]++;
            }

            if (subPlugs[1].RemoteInputPlugs.Length > 0) // r
                weights[0]++;
            if (subPlugs[2].RemoteInputPlugs.Length > 0) // g
                weights[1]++;
            if (subPlugs[3].RemoteInputPlugs.Length > 0) // b
                weights[2]++;
            if (subPlugs[4].RemoteInputPlugs.Length > 0) // a
                weights[3]++;

            if (weights.Any(w => w > 1)) // invalid, over-connected
                return false;

            // otherwise make sure it is valid
            return weights.All(w => w == 1);
        }
    }
}
